#ifndef __LINE_FINDER__
#define __LINE_FINDER__

#include <string>
#include <vector>

#include "State.h"
#include "Box.h"

/*
 * Abstract class - represents an interface that contains all methods
 * display code needs to work.
 */
class LineFinder {
 public:
    virtual Box getBoundaries() = 0;
    virtual LineList *findLines(Box& box) = 0;
};


/*
 * A naive implementations of LineFinder.
 */
class SimpleLineFinder: public LineFinder {
 protected:
    State *state;
 public:
    SimpleLineFinder(State *state) { this->state = state; }
    virtual Box getBoundaries();
    virtual LineList *findLines(Box& box);
};


/*
 * A better implementation using orthogonal range-query.
 */
class SmartLineFinder: public SimpleLineFinder {
 public:
    SmartLineFinder(State *state) : SimpleLineFinder(state) {}
    virtual Box getBoundaries();
    virtual LineList *findLines(Box& box);
};


/*
 * A client implementation which interacts with a server.
 */
class NetworkLineFinder: public LineFinder {
 protected:
    string hostname;
    int port;
 public:
    NetworkLineFinder(string hostname, int port);
    virtual Box getBoundaries();
    virtual LineList *findLines(Box& box);
};


/*
 * A proxy implementation that caches a number of requests.
 */
class CachedLineFinder: public LineFinder {
 protected:
    struct Pair {
	Box box;
	LineList *lines;
    };

    typedef vector<Pair> Cache;

    int maxSize;
    LineFinder *realFinder;
    Cache cache;

 public:
    CachedLineFinder(LineFinder *realFinder, int maxSize = 10) {
	this->realFinder = realFinder;
	this->maxSize = maxSize;
    }

    virtual Box getBoundaries() {
	return realFinder->getBoundaries();
    }

    virtual LineList *findLines(Box& box);
};


#endif // __LINE_FINDER__
