structure VersionedSeq :>
sig
  type 'a t
  type 'a vseq = 'a t

  exception InvalidVersion

  val tabulate: int * int * (int -> 'a) -> 'a vseq
  val length: 'a vseq -> int
  val nth: 'a vseq -> int -> 'a
  val update: 'a vseq -> (int * 'a) -> 'a vseq
  val inject: 'a vseq -> (int * 'a) Seq.t -> 'a vseq
  val to_seq: 'a vseq -> 'a Seq.t

  val to_string: ('a -> string) -> 'a vseq -> string
end =
struct

  fun cas r (old, new) =
    MLton.eq (old, MLton.Parallel.compareAndSwap r (old, new))

  datatype 'a vseq = VSeq of {status: Word8.word ref, data: 'a array}
  type 'a t = 'a vseq

  exception InvalidVersion

  val valid: Word8.word = 0w1
  val invalid: Word8.word = 0w0

  fun tabulate (lo, hi, f) =
    let
      val n = hi - lo
      val data = ForkJoin.alloc n
    in
      ForkJoin.parform (0, n) (fn i => Array.update (data, i, f (lo + i)));
      VSeq {status = ref valid, data = data}
    end

  fun length (VSeq {data, ...}) = Array.length data

  fun nth (VSeq {data, status}) i =
    let val x = Array.sub (data, i)
    in
      if !status = valid then x else raise InvalidVersion
    end

  fun update (VSeq {data, status}) (i, x) =
    if not (cas status (valid, invalid)) then
      raise InvalidVersion
    else
      ( Array.update (data, i, x)
      ; VSeq {status = ref valid, data = data}
      )

  fun inject (VSeq {data, status}) updates =
    if not (cas status (valid, invalid)) then
      raise InvalidVersion
    else
      ( ForkJoin.parform (0, Seq.length updates) (fn i =>
          let val (j, x) = Seq.nth updates i
          in Array.update (data, j, x)
          end)
      ; VSeq {status = ref valid, data = data}
      )

  fun to_seq (VSeq {data, status}) =
    if not (cas status (valid, invalid)) then
      raise InvalidVersion
    else
      ArraySlice.full data


  fun to_string elem_to_string (VSeq {data, status}) =
    let
      val result = Seq.toString elem_to_string (ArraySlice.full data)
    in
      if !status = valid then result else raise InvalidVersion
    end

end
