structure Ex =
struct

  structure CLA = CommandLineArgs


  (* =========================================================================
   * Fibonacci
   *)

  fun fib n =
    if n <= 1 then
      n
    else
      let val (a, b) = ForkJoin.par (fn () => fib (n - 1), fn () => fib (n - 2))
      in a + b
      end

  fun bench_fib () =
    let
      val n = CLA.parseInt "n" 40
      val _ = print ("n " ^ Int.toString n ^ "\n")
      val result = Benchmark.run (fn () => fib n)
    in
      print (Int.toString result ^ "\n")
    end

  (* =========================================================================
   * Word count
   *)

  fun parallel_sum (lo, hi, f) =
    if lo >= hi then
      0
    else if lo + 1 = hi then
      f (lo)
    else
      let
        val mid = lo + (hi - lo) div 2
        val (left, right) =
          ForkJoin.par (fn () => parallel_sum (lo, mid, f), fn () =>
            parallel_sum (mid, hi, f))
      in
        left + right
      end

  fun wc chars =
    parallel_sum (0, Seq.length chars, fn i =>
      if
        not (Char.isSpace (Seq.nth chars i))
        andalso (i = 0 orelse Char.isSpace (Seq.nth chars (i - 1)))
      then 1
      else 0)

  fun wc_alternate (chars: char Seq.t) =
    let
      fun getter (i) =
        if i = 0 then #" " else Seq.nth chars (i - 1)
    in
      parallel_sum (0, Seq.length (chars), fn i =>
        let
          val c1: char = getter i
          val c2: char = getter (i + 1)
        in
          if Char.isSpace c1 andalso not (Char.isSpace c2) then 1 else 0
        end)
    end


  fun wc_faster chars =
    ForkJoin.reducem op+ 0 (0, Seq.length chars) (fn i =>
      if
        not (Char.isSpace (Seq.nth chars i))
        andalso (i = 0 orelse Char.isSpace (Seq.nth chars (i - 1)))
      then 1
      else 0)



  fun bench_wc () =
    let
      val input = CLA.parseString "input" "../data/words.txt"
      val _ = print ("input " ^ input ^ "\n")
      val contents = ReadFile.contentsSeq input
      val result = Benchmark.run (fn () => wc contents)
    in
      print (Int.toString result ^ "\n")
    end


  fun bench_wc_faster () =
    let
      val input = CLA.parseString "input" "../data/words.txt"
      val _ = print ("input " ^ input ^ "\n")
      val contents = ReadFile.contentsSeq input
      val result = Benchmark.run (fn () => wc_faster contents)
    in
      print (Int.toString result ^ "\n")
    end

  (* =========================================================================
   * Data Races
   *)

  fun parfor (lo, hi, f) =
    if lo >= hi then
      ()
    else if lo + 1 = hi then
      f (lo)
    else
      let
        val mid = lo + (hi - lo) div 2
        val _ = ForkJoin.par (fn () => parfor (lo, mid, f), fn () =>
          parfor (mid, hi, f))
      in
        ()
      end

  fun data_race_ex () =
    let
      val n = CLA.parseInt "n" (1000 * 1000)

      (* A "ref" is mutable: it can be modified. When we (non-atomically)
       * modify the ref concurrently, using many parallel tasks, updates will
       * be lost. This is one instance of a general problem commonly known as
       * a "data race" or "race condition".
       *
       * In this course, we will avoid mutable data to ensure that our code
       * can be safely parallelized.
       *)
      val counter = ref 0
    in
      parfor (0, n, fn i => counter := !counter + 1);
      print (Int.toString (!counter) ^ "\n")
    end


  fun data_race_faster_ex () =
    let
      val n = CLA.parseInt "n" (1000 * 1000)

      (* A "ref" is mutable: it can be modified. When we (non-atomically)
       * modify the ref concurrently, using many parallel tasks, updates will
       * be lost. This is one instance of a general problem commonly known as
       * a "data race" or "race condition".
       *
       * In this course, we will avoid mutable data to ensure that our code
       * can be safely parallelized.
       *)
      val counter = ref 0
    in
      ForkJoin.parform (0, n) (fn i => counter := !counter + 1);
      print (Int.toString (!counter) ^ "\n")
    end


end
