structure Tester:
sig
  structure Result:
  sig
    datatype 'a t = Okay of 'a | Raised of exn
  end

  datatype ('a, 'b) test =
    T of
      { input: unit -> 'a
      , func: 'a -> 'b
      , report: 'a
        -> 'b Result.t
        -> {score: real, summary: string, details: string list}
      }

  (* run tests, dump reports along the way, add up and return total score *)
  val run_tests: string -> ('a, 'b) test list -> real
end =
struct

  structure Result = struct datatype 'a t = Okay of 'a | Raised of exn end

  fun result f =
    Result.Okay (f ())
    handle exn => Result.Raised exn

  datatype ('a, 'b) test =
    T of
      { input: unit -> 'a
      , func: 'a -> 'b
      , report: 'a
        -> 'b Result.t
        -> {score: real, summary: string, details: string list}
      }

  fun boxy (lines: string list) =
    let
      val lines = Seq.fromList lines
      val n = Seq.length lines
    in
      if n = 0 then
        ""
      else if n = 1 then
        "═ " ^ Seq.nth lines 0 ^ "\n"
      else
        "┌ " ^ Seq.nth lines 0 ^ "\n"
        ^
        Seq.reduce op^ "" (Seq.map (fn ln => "│ " ^ ln ^ "\n")
          (Seq.subseq lines (1, n - 2))) ^ "└ " ^ Seq.nth lines (n - 1) ^ "\n"
    end

  fun indent s = "  " ^ s

  fun header_line contents =
    let
      val budget = Int.max (0, 60 - (2 + String.size contents))
      val left = budget div 2
      val right = budget - left
      fun repeat n s =
        Parallel.reduce op^ "" (0, n) (fn _ => s)
    in
      repeat left "═" ^ " " ^ contents ^ " " ^ repeat right "═" ^ "\n"
    end


  fun run_tests header tests =
    let
      val () = print (header_line header)

      val num_tests = List.length tests

      fun do_test (T {input, func, report}) =
        let
          val fresh_input1 = input ()
          val fresh_input2 = input ()
        in
          report fresh_input1 (result (fn () => func fresh_input2))
        end
        handle exn =>
          { score = 0.0
          , summary = "Unexpected error during testing: " ^ exnMessage exn
          , details = []
          }

      fun next (test, (test_num, total_score)) =
        let
          val {score, summary, details} = do_test test
          val total_score' = score + total_score

          val summary_line =
            "Test " ^ Int.toString test_num ^ "/" ^ Int.toString num_tests
            ^ ": " ^ summary
          val lines = summary_line :: List.map indent details
        in
          print (boxy lines);
          (test_num + 1, total_score')
        end

      val (_, total_score) = List.foldl next (1, 0.0) tests
    in
      total_score
    end

end
