type point = real * real
type input = point Seq.t

val resolution = 1000000
fun rand_real seed =
  Real.fromInt (Util.hash seed mod resolution) / Real.fromInt resolution
fun rand_point_in_circle {center = (cx, cy), radius} seed =
  let
    val r = radius * Math.sqrt (rand_real seed)
    val theta = rand_real (seed + 1) * 2.0 * Math.pi
  in
    (cx + r * Math.cos (theta), cy + r * Math.sin (theta))
  end

fun rand_point_on_circle {center = (cx, cy), radius} seed =
  let
    val r = radius
    val theta = rand_real (seed + 1) * 2.0 * Math.pi
  in
    (cx + r * Math.cos (theta), cy + r * Math.sin (theta))
  end

fun rand_point_in_rect {bottom_left = (x0, y0), top_right = (x1, y1)} seed =
  let
    val x = rand_real seed
    val y = rand_real (seed + 1)

    val (x0, x1) = (Real.min (x0, x1), Real.max (x0, x1))
    val (y0, y1) = (Real.min (y0, y1), Real.max (y0, y1))

    val x = x0 + x * (x1 - x0)
    val y = y0 + y * (y1 - y0)
  in
    (x, y)
  end

fun sq x = x * x

val tests =
  [ Seq.fromList [(0.0, 0.0), (1.0, 0.0), (0.0, 1.0)]
  , Seq.tabulate (fn i => (Real.fromInt (i + 1), 1.0 / Real.fromInt (i + 1)))
      100
  , Seq.tabulate (fn i => let val x = Real.fromInt i - 500.0 in (x, x * x) end)
      1000
  , Seq.tabulate
      (fn i => let val x = Real.fromInt i - 500.0 in (x, ~(x * x)) end) 1000
  , Seq.tabulate
      (rand_point_in_rect {bottom_left = (0.0, 0.0), top_right = (1.0, 1.0)})
      1000
  , Seq.tabulate (rand_point_in_circle {center = (0.0, 0.0), radius = 1.0}) 1000
  , Seq.tabulate (rand_point_on_circle {center = (0.0, 0.0), radius = 1.0}) 1000
  , Seq.flatten (Seq.fromList
      [ Seq.tabulate
          (fn i =>
             rand_point_in_rect
               {bottom_left = (0.0, 0.0), top_right = (1.0, 1.0)} (Util.hash i))
          1000
      , Seq.tabulate
          (fn i =>
             rand_point_in_circle {center = (1.0, 0.0), radius = 1.0}
               (Util.hash (i + 1))) 1000
      , Seq.tabulate
          (fn i =>
             rand_point_on_circle {center = (2.0, 0.0), radius = 1.0}
               (Util.hash (i + 2))) 1000
      ])
  ]
  @
  List.tabulate (20, fn i =>
    let
      val seed = Util.hash (Util.hash (Util.hash i + 1))
      val centers =
        Seq.tabulate
          (fn j =>
             rand_point_on_circle {center = (0.0, 0.0), radius = 1.0}
               (seed + Util.hash (Util.hash j mod 100000000 + 3) mod 10000000))
          10
      val circles =
        Seq.mapIdx
          (fn (j, center) =>
             Seq.tabulate
               (fn k =>
                  rand_point_in_circle
                    { center = center
                    , radius = Real.fromInt (Util.hash k mod 1000) / 1000.0
                    }
                    (seed
                     +
                     Util.hash
                       (Util.hash j mod 100000000 + Util.hash k mod 10000000 + 4)
                     mod 10000000)) 1000) centers
    in
      Seq.flatten circles
    end)

(* ======================================================================= *)

structure Log:
sig
  val tick: unit -> unit
  val count: unit -> int
  val clear: unit -> unit
end =
struct
  val log = Array.tabulate (MLton.Parallel.numberOfProcessors, fn _ => 0)
  fun tick () =
    ignore
      (MLton.Parallel.arrayFetchAndAdd (log, MLton.Parallel.processorNumber ())
         1)
  fun count () =
    Array.foldl op+ 0 log
  fun clear () =
    Util.for (0, Array.length log) (fn i => Array.update (log, i, 0))
end

fun logged_tri_area (p, q, r) =
  (Log.tick (); Geometry2D.Point.triArea (p, q, r))

structure MQH = MyQuickhull (type point = point val tri_area = logged_tri_area)
structure RQH =
  ReferenceQuickhull (type point = point val tri_area = logged_tri_area)

fun student input =
  (Log.clear (); let val result = MQH.hull input in (result, Log.count ()) end)

fun reference input =
  (Log.clear (); let val result = RQH.hull input in (result, Log.count ()) end)

(* ========================================================================= *)


fun iptos (a, b) =
  "(" ^ Int.toString a ^ "," ^ Int.toString b ^ ")"

fun leftmost (x, y) =
  if Option.isSome x then x else y

fun diff eq (x, y) =
  Parallel.reduce leftmost NONE (0, Int.min (Seq.length x, Seq.length y))
    (fn i => if eq (Seq.nth x i, Seq.nth y i) then NONE else SOME i)

fun btos true = "true"
  | btos false = "false"

fun copy s =
  Seq.map (fn x => x) s

fun passed summary =
  {score = 1.0, summary = "Passed (" ^ summary ^ ")", details = []}
fun failed details = {score = 0.0, summary = "Failed", details = details}

(* percent improvement of a relative to b *)
fun percent_imp a b =
  let
    val a = Real.fromInt a
    val b = Real.fromInt b
  in
    100.0 * (b - a) / b
  end


fun report_qh input result =
  case result of
    Tester.Result.Raised exn => failed ["raised exception: " ^ exnMessage exn]
  | Tester.Result.Okay (output, num_calls) =>
      let
        val (expected, ref_num_calls) = reference input
      in
        if Seq.length expected <> Seq.length output then
          failed
            ["expected output of length " ^ Int.toString (Seq.length expected)
             ^ " but got output of length " ^ Int.toString (Seq.length output)]
        else
          case diff op= (output, expected) of
            SOME i =>
              failed
                ["expected " ^ Int.toString (Seq.nth expected i) ^ " at index "
                 ^ Int.toString i ^ " but got "
                 ^ Int.toString (Seq.nth output i)]
          | NONE =>
              if num_calls <= Util.ceilDiv ref_num_calls 2 then
                passed
                  (Real.fmt (StringCvt.FIX (SOME 1))
                     (percent_imp num_calls ref_num_calls)
                   ^ "% fewer tri_area calls")
              else
                failed
                  [ "too many calls to tri_area"
                  , "reference did " ^ Int.toString ref_num_calls
                    ^ " and yours did " ^ Int.toString num_calls
                  , "(yours did "
                    ^
                    Real.fmt (StringCvt.FIX (SOME 1))
                      (percent_imp num_calls ref_num_calls)
                    ^ "% fewer, but need at least 50% fewer)"
                  ]
      end


fun make_qh_test inp =
  Tester.T {input = (fn () => copy inp), func = student, report = report_qh}

val _ = Tester.run_tests "Testing MyQuickhull.hull"
  (List.map make_qh_test tests)
