structure Image:
sig
  type img
  type t = img

  type point = real * real

  val fresh:
    { resolution: int
    , bottom_left: point
    , top_right: point
    , background: Color.pixel
    }
    -> img

  val draw_box: img -> {color: Color.pixel} -> point -> point -> unit
  val draw_line: img -> {color: Color.pixel} -> point -> point -> unit
  val draw_point: img -> {color: Color.pixel} -> point -> unit
  val write_to_file: string -> img -> unit
end =
struct

  type point = real * real

  datatype img =
    Img of
      { width: int
      , height: int
      , bottom_left: point
      , top_right: point
      , background: Color.pixel
      , data: Color.pixel array
      }

  type t = img


  fun fresh {resolution, bottom_left, top_right, background} =
    let
      val (x0, y0) = bottom_left
      val (x1, y1) = top_right

      val width = Real.round (Real.fromInt resolution * (x1 - x0))
      val height = Real.round (Real.fromInt resolution * (y1 - y0))

      val data = SeqBasis.tabulate 1000 (0, width * height) (fn _ => background)
    in
      Img
        { width = width
        , height = height
        , bottom_left = bottom_left
        , top_right = top_right
        , background = background
        , data = data
        }
    end


  fun ipart x = Real.floor x
  fun fpart x = x - Real.realFloor x
  fun rfpart x = 1.0 - fpart x

  fun adjust_opacity ({red, green, blue, alpha}: Color.color) alpha' :
    Color.color =
    {red = red, green = green, blue = blue, alpha = alpha * alpha'}


  fun constrain (lo, hi) x =
    Real.min (hi, Real.max (lo, x))


  fun draw_line (Img {width, height, bottom_left, top_right, background, data})
    {color = desired_line_color: Color.pixel} (x0, y0) (x1, y1) =
    let
      (* ix: discrete x
       * jy: discrete y
       * We have to convert the indices for the image format.
       *)
      fun set (ix, jy) elem =
        if 0 <= jy andalso jy < height andalso 0 <= ix andalso ix < width then
          Array.update (data, (height - jy - 1) * width + ix, elem)
        else
          ()

      fun get (ix, jy) =
        if 0 <= jy andalso jy < height andalso 0 <= ix andalso ix < width then
          Array.sub (data, (height - jy - 1) * width + ix)
        else
          (* just dummy data, in this case *)
          background

      fun plot (ix, jy, strength) =
        let
          val current = get (ix, jy)
        in
          set (ix, jy) (Color.colorToPixel (Color.overlayColor
            { fg =
                adjust_opacity (Color.pixelToColor desired_line_color) strength
            , bg = Color.pixelToColor current
            }))
        end


      (* =================================================================== *)

      (* val _ = print ("draw_line " ^ Real.toString x0 ^ " " ^ Real.toString x1 ^ " " *)

      val (x0, y0, x1, y1) =
        if x1 < x0 then (x1, y1, x0, y0) else (x0, y0, x1, y1)


      (* adjust line coordinates to be relative to the image origin *)
      val (xb, yb) = bottom_left
      val (xt, yt) = top_right

      fun xx x =
        Real.fromInt width * (constrain (xb, xt) x - xb) / (xt - xb) + 0.5
      fun yy y =
        Real.fromInt height * (constrain (yb, yt) y - yb) / (yt - yb) + 0.5

      val (x0, y0, x1, y1) = (xx x0, yy y0, xx x1, yy y1)

      val dx = x1 - x0
      val dy = y1 - y0
      val yx_slope = dy / dx
      val xy_slope = dx / dy

      fun normal_loop (x, y) =
        ( (*print ("normal_loop " ^ Real.toString x ^ " " ^ Real.toString y ^ "\n")
          ;*)
          if x > x1 then
            ()
          else
            ( plot (ipart x, ipart y, rfpart y)
            ; plot (ipart x, ipart y + 1, fpart y)
            ; normal_loop (x + 1.0, y + yx_slope)
            ))

      fun steep_up_loop (x, y) =
        ( (*print
            ("steep_up_loop " ^ Real.toString x ^ " " ^ Real.toString y ^ "\n")
          ;*)
          if y > y1 then
            ()
          else
            ( plot (ipart x, ipart y, rfpart x)
            ; plot (ipart x + 1, ipart y, fpart x)
            ; steep_up_loop (x + xy_slope, y + 1.0)
            ))

      fun steep_down_loop (x, y) =
        ( (*print
            ("steep_down_loop " ^ Real.toString x ^ " " ^ Real.toString y ^ "\n")
          ;*)
          if y < y1 then
            ()
          else
            ( plot (ipart x, ipart y, rfpart x)
            ; plot (ipart x + 1, ipart y, fpart x)
            ; steep_down_loop (x - xy_slope, y - 1.0)
            ))

    in
      if Real.abs dx > Real.abs dy then normal_loop (x0, y0)
      else if y1 > y0 then steep_up_loop (x0, y0)
      else steep_down_loop (x0, y0)
    end


  fun draw_box (Img {width, height, bottom_left, top_right, background, data})
    {color: Color.pixel} (x0, y0) (x1, y1) =
    let
      fun set (ix, jy) =
        if 0 <= jy andalso jy < height andalso 0 <= ix andalso ix < width then
          Array.update (data, (height - jy - 1) * width + ix, color)
        else
          ()

      val (x0, x1) = (Real.min (x0, x1), Real.max (x0, x1))
      val (y0, y1) = (Real.min (y0, y1), Real.max (y0, y1))

      (* adjust coordinates to be relative to the image origin *)
      val (xb, yb) = bottom_left
      val (xt, yt) = top_right

      fun xx x =
        Real.floor
          (Real.fromInt width * (constrain (xb, xt) x - xb) / (xt - xb) + 0.5)
      fun yy y =
        Real.floor
          (Real.fromInt height * (constrain (yb, yt) y - yb) / (yt - yb) + 0.5)

      val (x0, y0, x1, y1) = (xx x0, yy y0, xx x1, yy y1)
    in
      ForkJoin.parform (y0, y1) (fn y =>
        ForkJoin.parform (x0, x1) (fn x => set (x, y)))
    end


  fun draw_point (Img {width, height, bottom_left, top_right, background, data})
    {color: Color.pixel} (x, y) =
    let
      (* val _ = print
        ("draw_point | " ^ Real.toString x ^ " " ^ Real.toString y ^ " | ") *)
      (* ix: discrete x
       * jy: discrete y
       * We have to convert the indices for the image format.
       *)
      fun set (ix, jy) =
        if 0 <= jy andalso jy < height andalso 0 <= ix andalso ix < width then
          Array.update (data, (height - jy - 1) * width + ix, color)
        else
          ()

      (* adjust line coordinates to be relative to the image origin *)
      val (xb, yb) = bottom_left
      val (xt, yt) = top_right
      val x = Real.floor (Real.fromInt width * (x - xb) / (xt - xb) + 0.5)
      val y = Real.floor (Real.fromInt height * (y - yb) / (yt - yb) + 0.5)

    (* val _ = print (Int.toString x ^ " " ^ Int.toString y ^ "\n") *)
    in
      set (x - 1, y);
      set (x, y - 1);
      set (x, y);
      set (x, y + 1);
      set (x + 1, y)
    end


  fun write_to_file path (Img {width, height, data, ...}) =
    PPM.write path {width = width, height = height, data = ArraySlice.full data}

end
