package nanomunchers.ui;

import heurgame.PlayerToken;
import heurgame.event.PlayerBroadcaster;
import heurgame.event.PlayerEvent;
import heurgame.ui.PlayerColors;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
//import java.awt.Point;
//import java.awt.event.MouseEvent;
//import java.awt.event.MouseListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

import nanomunchers.bot.NanoBot;
import nanomunchers.event.FrameListener;
import nanomunchers.graph.Edge;
import nanomunchers.graph.Graph;
//import nanomunchers.graph.KGraphFactory;
import nanomunchers.graph.Node;
/*
 * Created on Oct 27, 2004
 */

/**
 * Arguably the largest piece of the Nanomunchers game is the simulation that
 * shows the state of the game at each frame.  This Simulation class merely
 * plays a set of frames.  The Simulator class is the one that builds the frames 
 * themselves.
 * 
 * @author David Kaplin
 */
public class NanoSimulation extends Canvas {
    private Graph graph;
    private Graphics bufferGraphics;
	private Image offscreen;
	private int hoffset,woffset;
	private int lastWidth=-1;
	private int lastHeight = -1;
	private PlayerColors playerColors;
	private PlayerToken[] eaten;
	private Vector edgeLookup;
	private Vector currentFrame;
	private Vector allFrames;
	private Hashtable nanoDeath;
	private Hashtable nanoLookup;
	private Hashtable playerLookup;
	private double interpolation = .1;
	private double currentPosition = 0;
	private Hashtable nanoPlacesStart;
	private Hashtable nanoPlacesEnd;
	private PlayerBroadcaster playerSpeaker;
	private boolean simulationRunning = false;
	private boolean showNames = true;
	private boolean showCodes = true;
	private volatile double simulationSpeed = 3.0;
	private volatile int framesToSkip = 0;
    private Drawing graphDrawing = new Drawing();
	private FrameListener frameListener;
    
	/**
	 * @param frame to be notified when frames are either being added or played.
	 */
	public void setFrameListener(FrameListener frame){
	    frameListener = frame;
	}
	
	/**
	 * Initializes the simulation
	 * 
	 * @param graph to be displayed
	 */
	public void setup(Graph graph){
	    this.graph = graph;
	    graphDrawing.setGraph(graph);
	    eaten = new PlayerToken[graph.getUniqueEdges().length];
	    edgeLookup = new Vector();
	    for(int i=0;i<graph.getNodes().length;i++){
	        edgeLookup.add(graph.getNodes()[i]);
	        //eaten[i]=null;
	    }
	    for(int i=0;i<eaten.length;i++){
	        eaten[i] = null;
	    }
	    nanoLookup = new Hashtable();
	    playerLookup = new Hashtable();
	    nanoPlacesStart = new Hashtable();
	    nanoPlacesEnd = new Hashtable();
	    currentFrame = new Vector();
	    allFrames = new Vector();
	/**
	 * Allows a mapping back from a nanobot to its player
	 * @param owner of the bot
	 * @param nb that is owned
	 * @return object ref to this particular bot
	 */
	}
	public Integer register(PlayerToken owner, NanoBot nb){
	    Integer id = new Integer(nanoLookup.size());
	    nanoLookup.put(id,nb);
	    playerLookup.put(id,owner);
	    return id;
	}
	/**
	 * Constructs a piece of a frame
	 * 
	 * @param nanoId object ref of a registered bot
	 * @param place where the bot is
	 * @param munched how many nodes has it munched
	 * @param alive whether the bot is alive
	 * @param naturaldeath whether or not the bot exhausted its food supply
	 */
	public void nextMove(Integer nanoId, Node place,int munched,boolean alive,boolean naturaldeath){
	    NanoBit bit = new NanoBit();
	    bit.id = nanoId;
	    bit.spot = place;
	    bit.dead = !alive;
	    bit.diedNaturally = naturaldeath;
	    bit.munched = munched;
	    graph = place.getGraph();
	    graphDrawing.setGraph(graph);
	    currentFrame.add(bit);
	}
	/**
	 * Finishes the current frame
	 */
	public void nextFrame(){
	    frameListener.frameCreated((allFrames.size()));
	    allFrames.add(currentFrame);
	    currentFrame = new Vector();
	}
	/**
	 * Changes the speed of the simulation without skipping frames.
	 * If frames were skipped at a previous setting, there will not
	 * be any frames skipped at this ratio.
	 * 
	 * @param s Larger numbers yeild a SLOWER speed.
	 */
	public void setSimulationSpeed(double s){
	    simulationSpeed = s;
	    framesToSkip = 0;
	}
	/**
	 * Slows down the simulation either by adding frames (if any were skipped)
	 * or by lengthening the delay between frames
	 * 
	 * @return true if the speed or frames were changed
	 */
	public boolean decreaseSpeed(){
	    if (simulationSpeed < .001){
	        if (framesToSkip > 0){
	            framesToSkip--;
	            return true;
	        }
	        return false;
	    }
	    if (simulationSpeed > 20){
	        return false;
	    }
	    simulationSpeed *=2.0;
	    return true;
	}
	/**
	 * Either shortens the delay or skips frames.
	 * 
	 * @return true if the speed could be increased
	 */
	public boolean increaseSpeed(){
	    if (simulationSpeed < .001){
	        if (framesToSkip < 5){
	            framesToSkip++;
	            return true;
	        }
	        return false;
	    }
	    simulationSpeed /=2.0;
	    return true;
	}
	/**
	 * Reads through the frames and plays them at whatever
	 * the current speed is.  Speed may be changed during
	 * playback.
	 * 
	 * With the aid of the PlayerBroadcaster the play method replays
	 * the scores as they occured in the simulation.
	 */
	public void play(){
	    nanoDeath = new Hashtable();
	    simulationRunning = true;
	    graph.clear();
	    
	    for(int i=0;i<eaten.length;i++){
	        eaten[i]=null;
	    }
	    for(int i=0;i<nanoLookup.size();i++){
	        NanoBot bot = (NanoBot)nanoLookup.get(new Integer(i));
	        bot.clear();
	    }
	    int skipCounter = 0;
	    for(int i=0;i<(allFrames.size()-1);i++){
	        
	        Vector thisFrame = (Vector)allFrames.get(i);
	        Hashtable currentScores = new Hashtable();
	        for(int j=0;j<thisFrame.size();j++){
                NanoBit bit = (NanoBit)thisFrame.get(j);
                PlayerToken owner = (PlayerToken)playerLookup.get(bit.id);
                
                if (currentScores.get(owner)==null){
                    currentScores.put(owner,new Integer(0));
                }
                int runningScore = ((Integer)currentScores.get(owner)).intValue(); 
                currentScores.put(owner,new Integer(runningScore+bit.munched));
            }
	        Iterator siterator = currentScores.keySet().iterator();
	        while(siterator.hasNext()){
	            PlayerToken who = (PlayerToken)siterator.next();
	            PlayerEvent e = new PlayerEvent();
	            e.player = who;
	            e.disqualified=false;
	            e.leader=false;
	            e.score = ((Integer)currentScores.get(who)).intValue();
	            e.validMove = true;
	            playerSpeaker.announcePlayerMoved(e);
	            playerSpeaker.announcePlayerStatusChanged(e);
	        }
	        synchronized (nanoLookup){
	            for(int j=0;j<thisFrame.size();j++){
	                NanoBit bit = (NanoBit)thisFrame.get(j);
	                NanoBot bot = (NanoBot)nanoLookup.get(bit.id);
	                nanoPlacesStart.put(bit.id,bit.spot);
	                if (nanoDeath.get(bit.id)!=null){
	                    if (((Boolean)nanoDeath.get(bit.id)).booleanValue()==false){
	                        continue;
	                    }
	                }
	                nanoDeath.put(bit.id,new Boolean(bit.diedNaturally));
	            }
	            Vector nextFrame = (Vector)allFrames.get(i+1);
	            for(int j=0;j<nextFrame.size();j++){
	                NanoBit bit = (NanoBit)nextFrame.get(j);
	                NanoBot bot = (NanoBot)nanoLookup.get(bit.id);
	                nanoPlacesEnd.put(bit.id,bit.spot);
	                if (nanoDeath.get(bit.id)!=null){
	                    if (((Boolean)nanoDeath.get(bit.id)).booleanValue()==false){
	                        continue;
	                    }
	                }
	                nanoDeath.put(bit.id,new Boolean(bit.diedNaturally));
	            }
	            Set s = nanoLookup.keySet();
	            Iterator edges = s.iterator();
	            while (edges.hasNext()){
	                Integer id = (Integer)edges.next();
	                Node a = (Node)nanoPlacesEnd.get(id);
	                Node b = (Node)nanoPlacesStart.get(id);
	                if (a==null || b==null){
	                    continue;
	                }
	                if (a.getPoint().equals(b.getPoint())){
	                    continue;
	                }
	                Edge[] gEdges =graph.getUniqueEdges(); 
	                for(int e=0;e<gEdges.length;e++){
	                    if (gEdges[e].doesContain(a) &&gEdges[e].doesContain(b)){
	                        eaten[e] = (PlayerToken)playerLookup.get(id);
	                    }
	                }
	            }
	            currentScores = new Hashtable();
		        for(int j=0;j<nextFrame.size();j++){
	                NanoBit bit = (NanoBit)nextFrame.get(j);
	                PlayerToken owner = (PlayerToken)playerLookup.get(bit.id);
	                
	                if (currentScores.get(owner)==null){
	                    currentScores.put(owner,new Integer(0));
	                }
	                int runningScore = ((Integer)currentScores.get(owner)).intValue(); 
	                currentScores.put(owner,new Integer(runningScore+bit.munched));
	            }
		        siterator = currentScores.keySet().iterator();
		        while(siterator.hasNext()){
		            PlayerToken who = (PlayerToken)siterator.next();
		            PlayerEvent e = new PlayerEvent();
		            e.player = who;
		            e.disqualified=false;
		            e.leader=false;
		            e.score = ((Integer)currentScores.get(who)).intValue();
		            e.validMove = true;
		            playerSpeaker.announcePlayerMoved(e);
		            playerSpeaker.announcePlayerStatusChanged(e);
		        }
		        skipCounter++;
		        if (skipCounter>=this.framesToSkip){
		            skipCounter = 0;
		        }else{
		            continue;
		        }
		        frameListener.frameStarted(i+1,(allFrames.size()));
		        if (framesToSkip>0){
		            update(this.getGraphics());
		            try {
		            Thread.sleep(50);
		            } catch (InterruptedException e) { }
		            continue;
		        }
		        while(currentPosition <= 1.0){
		            try {
		                update(this.getGraphics());
		                currentPosition += interpolation;
		                Thread.sleep((long)(10*simulationSpeed));
		                
		            } catch (InterruptedException e){
		            
		            }
		        }
	        }
	        currentPosition = 0;
	    }
	    frameListener.frameStarted((allFrames.size()),(allFrames.size()));
        if (framesToSkip>0){
            update(this.getGraphics());
        }
	    simulationRunning = false;
	}
    /**
     * Sets up a new simulation
     * @param pc Colors for each player.
     * @param playerSpeaker to notify any listeners of score changes.
     */
    public NanoSimulation(PlayerColors pc,PlayerBroadcaster playerSpeaker){
	    this.playerSpeaker = playerSpeaker;
        playerColors = pc;
        addMouseListener(new MouseAdapter(){
            private int clickCount = 0;
            public void mouseClicked(MouseEvent e){
                Thread t = new Thread(){
                    public void run(){
                        clickCount++;
                        clickCount%=4;
                        switch(clickCount){
                        	case 0:
                        	    showNames = true;
                        	    showCodes = true;
                        	break;
                        	case 1:
                        	    showNames = true;
                        	    showCodes = false;
                        	break;
                        	case 2:
                        	    showNames = false;
                        	    showCodes = true;
                        	break;
                        	case 3:
                        	    showNames = false;
                        	    showCodes = false;
                        	break;
                        }
                        
                        if (simulationRunning==false){
                            repaint();
                        }                            
                    }
                    
                };
                t.start();
            }
        });
    }
    /**
     * Only paints if the simulation is not running.
     *  
     * @see java.awt.Component#paint(java.awt.Graphics)
     */
    public void paint(Graphics g){
        if (simulationRunning){
            return;
        }
        update(g);
    }
    /**
     * If the graph is not null draws the simulation.
     *  
     * @see java.awt.Component#update(java.awt.Graphics)
     */
    public void update(Graphics g){
        if (graph==null){
            return;
        }
        int width = getWidth();
        int height = getHeight();
        if (offscreen == null || (width != lastWidth) || (height != lastHeight)) {
			offscreen = createImage(width, height);
			bufferGraphics = offscreen.getGraphics();
		}
        lastWidth = width;
        lastHeight = height;
        bufferGraphics.setColor(Color.black); //getBackground());
		bufferGraphics.fillRect(0, 0, width, height);
        
        woffset = width/(graph.getWidth()+1);
        hoffset = height/(graph.getHeight()+1);
        bufferGraphics.setColor(new Color(0,0,0));
        if (((width+1) > graph.getWidth()) && ((height+1) > graph.getHeight())){
            graphDrawing.drawGraph(bufferGraphics,Color.GRAY,width,height);
            int nanoRad = (width < height) ? woffset/5 : hoffset/5;
            Edge[] edges = graph.getUniqueEdges();
            int sx,sy,dx,dy;
            for(int i=0;i<edges.length;i++){
                if (eaten[i]!=null){
                    graphDrawing.highlightEdge(bufferGraphics,playerColors.getColor(eaten[i]),edges[i],4);                    
                }
            }
            //draw the bots
            for(int i=0;i<nanoPlacesStart.size();i++){
                
                Integer id = new Integer(i);
                Node start = (Node)nanoPlacesStart.get(id);
                Node end =   (Node)nanoPlacesEnd.get(id);
                
                if (end==null){
                    end = start;
                }
                sx = woffset + woffset*start.getX();
                dx = woffset + woffset*end.getX();
                sy = hoffset + hoffset*start.getY();
                dy = hoffset + hoffset*end.getY();
                
                int ix = (int)((double)sx + ((double)(dx - sx))*currentPosition);
                int iy = (int)((double)sy + ((double)(dy - sy))*currentPosition);
                PlayerToken thisOwner = (PlayerToken)playerLookup.get(id);
                Color playerColor = playerColors.getColor(thisOwner);
                bufferGraphics.setColor(playerColor);
                NanoBot bot = (NanoBot)nanoLookup.get(id);
                boolean conflictedDeath = false;
                
                if (sx!=dx || sy!=dy){
                    //bufferGraphics.fill3DRect(ix-nanoRad,iy-nanoRad,2*nanoRad,2*nanoRad,true);
                    graphDrawing.drawBotAlive(bufferGraphics,playerColor,ix,iy,dx,dy);
                }else{
                    boolean naturalCauses = ((Boolean)nanoDeath.get(id)).booleanValue();
                    //bufferGraphics.fill3DRect(ix-nanoRad,iy-nanoRad,2*nanoRad,2*nanoRad,false);
                    if (naturalCauses == false){
                        graphDrawing.drawBotSlain(bufferGraphics,playerColor,ix,iy,dx,dy);
                    }else{
                        graphDrawing.drawBotDead(bufferGraphics,playerColor,ix,iy,dx,dy);
                    }
                }
                bufferGraphics.setColor(Color.gray);
                if (showCodes){
                    bufferGraphics.fillRect(dx-15,dy+5,30,10);
                }
                if (showNames){
                    bufferGraphics.fillRect(dx-15,dy-15,30,10);
                }
                bufferGraphics.setColor(Color.white);
                if (showCodes){
                bufferGraphics.drawString(bot.getCode(),dx-15,dy+15);
                }
                if (showNames){
                    bufferGraphics.drawString(thisOwner.getName(),dx-15,dy-5);
                }
            }
        }
        g.drawImage(offscreen, 0, 0, this);
    }
    /**
     * Debugging method
     * @param args
     */
    public static void main(String args[]){
        final Frame f = new Frame("NanoSimulation");
        f.setLayout(new java.awt.BorderLayout());
        GraphEditPanel gep = new GraphEditPanel();
        gep.setSize(500,350);
        f.add(gep);
        
        f.addWindowListener(new WindowAdapter(){
            public void windowClosing(WindowEvent e){
                f.dispose();
                System.exit(1);
            }
        });
        f.pack();
        f.show();
    }
    
    /**
     * A bit of information within a frame in the simulation. 
     *
     * @author David Kaplin
     */
    class NanoBit{
        public Integer id;
        public Node spot;
        public boolean dead;
        public boolean diedNaturally;
        public int munched;
    }
}
