package nanomunchers.graph;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Vector;

/*
 * Created on Oct 26, 2004
 */

/**
 * The only way to build a new graph is through the Factory.
 * 
 * @author David Kaplin
 */
public class GraphFactory {
    /**
     * Builds a graph with a random number of edges removed
     * 
     * @param src intial graph before removal
     * @param chanceNodeRemoved 0.0 - 1.0 of removal
     * @return a new graph that contains less edges.
     */
    public static Graph buildRandomized(Graph src, double chanceNodeRemoved){
        Vector keepNodes = new Vector();
        Vector killNodes = new Vector();
        Vector keepEdges = new Vector();
        Node[] nodes = src.getNodes();
        Edge[] edges = src.getUniqueEdges();
        for(int i=0;i<nodes.length;i++){
            if (Math.random()>chanceNodeRemoved){
                killNodes.add(nodes[i]);
            }else{
                keepNodes.add(nodes[i]);
            }
        }
        for(int e=0;e<edges.length;e++){
            boolean keep = true;
            for(int i=0;i<killNodes.size();i++){
                Node killNode = (Node)killNodes.get(i);
                if (edges[e].doesContain(killNode)){
                    keep = false;
                    break;
                }
            }
            if (keep){
                keepEdges.add(edges[e]);
            }
        }
        Edge[] finalEdges = new Edge[keepEdges.size()];
        Node[] finalNodes = new Node[keepNodes.size()];
        for(int i=0;i<finalNodes.length;i++){
            finalNodes[i] = (Node)keepNodes.get(i); 
        }
        for(int j=0;j<finalEdges.length;j++){
            finalEdges[j] = (Edge)keepEdges.get(j);
        }
        return buildFromParts(finalNodes,finalEdges);
    }
    /**
     * Creats a new graph from a set of nodes and edges
     * 
     * @param nodes from a previous graph
     * @param edges from a previous graph
     * @return a new graph
     */
    public static Graph buildFromParts(Node[] nodes, Edge[] edges){
        return new GeneralGraph(nodes,edges);
    }
    /**
     * Builds a graph that resembles a city grid.
     * 
     * @param width of the grid
     * @param height of the grid
     * @return a new Graph
     */
    public static Graph buildGridGraph(int width,int height){
        return new GridGraph(width,height);
    }
    /**
     * Saves a given graph to a file.
     * 
     * @param name of the file to save
     * @param toSave Graph data to save.
     * @throws IOException related to file issues
     */
    public static void saveFile(String name,Graph toSave) throws IOException{
            PrintWriter pw = new PrintWriter(new FileWriter(name));
            pw.println("#Automatically Generated Graph");
            pw.println("#Nodes start here");
            Node[] nodes = toSave.getNodes();
            for(int i=0;i<nodes.length;i++){
                pw.print(nodes[i].getId());
                pw.print(',');
                pw.print(nodes[i].getX());
                pw.print(',');
                pw.println(nodes[i].getY());
            }
            Edge[] edges = toSave.getUniqueEdges();
            for(int i=0;i<edges.length;i++){
                pw.print(edges[i].getNodes()[0].getId());
                pw.print(',');
                pw.println(edges[i].getNodes()[1].getId());
            }
            pw.flush();
            pw.close();
    }
    
    /**
     * Attempts to load a graph from anywhere in the web.
     * 
     * @param url location of the graph
     * @return a new graph from the data retrieved.
     * @throws IOException any possible problems with the file or connection.
     */
    public static Graph loadHttpURL(String url) throws IOException{
        URL netRef;
            netRef = new URL(url);
            HttpURLConnection httpUrl = (HttpURLConnection) netRef.openConnection();
            httpUrl.setRequestMethod("GET");
            httpUrl.setInstanceFollowRedirects(true);
            httpUrl.setDoInput(true);
            httpUrl.setAllowUserInteraction(false);
            httpUrl.setRequestProperty("User-Agent","Mozilla/5.0 (compatible; MSIE 3.0; Nanomuncher Server)");
            InputStreamReader reader =new InputStreamReader(httpUrl.getInputStream());
            System.out.println("Response code: "+httpUrl.getResponseCode());
            Graph output = loadFromReader(reader);
            reader.close();
            httpUrl.disconnect();
            return output;
    }
    
    public static Graph loadFile(String name) throws FileNotFoundException{
            return loadFromReader(new FileReader(name));
    }
    private static Graph loadFromReader(Reader reader){
        Node[] nodes;
        Edge[] edges;
        Vector nodeLines = new Vector();
        Vector edgeLines = new Vector();
        System.out.println("Loading Graph...");
        try {
            BufferedReader br = new BufferedReader(reader);
            int lastChance = 0;
            String line = br.readLine();
            while(line != null){
                //System.out.println("GOT: "+line);
                String[] parts = line.split(",");
                if ((parts[0].length()> 0) && Character.isDigit(parts[0].charAt(0))==true){
                    //This is NOT a comment
                    if (parts.length==3){
                        //Node
                        nodeLines.add(new Node(null, Integer.parseInt(parts[0]),
                                Integer.parseInt(parts[1]), 
                            Integer.parseInt(parts[2])));
                    }
                    if (parts.length==2){
                        //Edge
                        Node a = (Node) nodeLines.get(Integer.parseInt(parts[0]));
                        Node b = (Node) nodeLines.get(Integer.parseInt(parts[1]));
                        if (a.getId()==97 && b.getId()==98){
                            //We read it!
                        }
                        edgeLines.add(new Edge(a,b));
                    }
                }
                line = br.readLine();
            }
            br.close();
            nodes = new Node[nodeLines.size()];
            edges = new Edge[edgeLines.size()];
            for(int i=0;i<nodeLines.size();i++){
                nodes[i] = (Node)nodeLines.get(i);
            }
            for(int j=0;j<edgeLines.size();j++){
                edges[j] = (Edge)edgeLines.get(j);
            }
            if (nodeLines.size()==0){
                System.out.println("Graph ERROR no nodes!");
                return null;
            }
            System.out.println("Loading Graph DONE");
            return new GeneralGraph(nodes,edges);
        } catch (FileNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return null;
    }
}
