package heurgame.event.turn;

import java.util.Vector;

import heurgame.PlayerToken;
import heurgame.event.TurnEvent;


/*
 * Created on Oct 15, 2004
 */

/**
 * @author David Kaplin
 *
 * In some games, like the Voronoi game the RoundRobinIterator will
 * not lead to a fair environment.  In this case to acheive fairness
 * the round robin order is changed so that the owner of the second
 * turn goes twice and looses their last turn.  Thus keeping the 
 * number of movees the same and fairness.
 * 
 * NOTE:  This iterator is specifically designed for two players.  
 * Any more than two will lead to unfair consequences
 */
public class SecondChanceRRIterator extends RoundRobinIterator {
    private int turnCount = 1;
    private boolean seenSecond = false;
    
    public void setup(int numRounds, Vector players){
        super.setup(numRounds,players);
        turnCount = 1;
        seenSecond = false;
    }
    
    public void next() {
        logState("SecondChance StartofTurn {"+turnCount+"} ");
        TurnEvent e = new TurnEvent();
        e.totalRounds = totalRounds;
        e.roundNumber = currentRound;
        e.player = currentToken;
        notifyTurnListeners(e);
        //Increment round?
        if ((turnCount==2) && (seenSecond==false)){
            //We need to repeat this turn, nothing should change!
            logState("SecondChance Seeing double next?");
            seenSecond = true;
            return;
        }
        if (playerHelper.hasNext() == false) {
            if (latestTokens.size() > 0) {
                playerTokens.clear();
                playerTokens.addAll(latestTokens);
                latestTokens.clear();
            }
            playerHelper = playerTokens.listIterator();
            currentRound++;
        }
        currentToken = (PlayerToken) playerHelper.next();
        turnCount++;
    }

    /**
     * @return True while the current round is less than or equal to the total.
     * If the game has 10 rounds the number of the last round will be 10.
     * This iterator will be exhausted after the first turn of the 10th round,
     * giving the first player his last chance.
     *  
     * @see heurgame.event.turn.AbstractTurnIterator#hasMore()
     */
    public boolean hasMore() {
        boolean specialCase = (2*totalRounds ) == turnCount;
        //Notes on specialCase
        //Since we did not increase the number of turns in the 
        //round where the second player got to play twice
        //instead of 2*totalRounds being the very last play
        //it becomes the limit
        //
        //For correctness 2*totalRounds - 1 is the last valid move
        //thus 2*totalRounds is unacceptable
        return (currentRound <= totalRounds) && !specialCase;
    }
}
