package heurgame;

import heurgame.logging.LogBox;

import java.net.Socket;
import java.io.OutputStreamWriter;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;

/**
 * @author David Kaplin
 *
 * PlayerProxies allow for Protocol Independance.  Whether or not the player
 * is a network player is immaterial.  This class and all of its children 
 * translate the protocol to an interface understood by the entire framework.
 * I sincerely hope it is general enough to be used for any sort of game.
 */
abstract public class PlayerProxy{
    protected LogBox debugging = new LogBox("PlayerProxy");
    /**
     * Each Individual Player has posesson of a single token that identifies 
     * them to the system.  This token is used throughout the framework.
     */
    protected PlayerToken token;
    /**
     * <code>fromPlayer</code> The part we read from the player. 
     */
    protected BufferedReader fromPlayer;
    /**
     * <code>toPlayer</code> The part we write to the player.
     */
    protected PrintWriter toPlayer;
    /**
     * This should be useed in liew of a constructor.
     * If the socket is open.  The server will expect to read the players name
     * in the first line of the output.  Inside only creation of a PlayerToken 
     * takes place.
     * 
     * @param parent The originating PlayerProxy
     * @param origin The source of the Proxy, should be a hostname and port
     * @param s Actual socket connected to the player
     * @throws IOException Will be thrown in case of socket problems.
     */
    public final void setup(PlayerProxy parent, String origin, Socket s) throws IOException{
        fromPlayer = new BufferedReader(new InputStreamReader(s.getInputStream()));
        toPlayer =new PrintWriter( new BufferedWriter(new OutputStreamWriter(s.getOutputStream())));
        String name = fromPlayer.readLine();
        token = new PlayerToken(parent, origin, name);
        debugging.addEntry("Setup","Now representing "+ name +" from "+ origin);
    }
    /**
     * Allows other components to have a partial handle on the player
     * state without giving them too much information.
     * 
     * @return Token that contains the player name and a unique id
     * in the case of rare naming conflicts.  It is recommented to
     * test based on the == equality to remove any possibility of
     * duplication.
     */
    public PlayerToken getToken(){ return token;}
    /**
     * Informs the player that it is their turn.
     * Returns the player's response
     * 
     * @param query game specific information for the player
     * @return The move sent by the player
     */
    abstract public String getMove(String query);
    /**
     * Informs the player that they have been disqualified from the game.
     * 
     * @param reason why the person was disqualified
     */
    abstract public void sendDisqualify(String reason);
    /**
     * Informs the player of the initial status of the game.
     * @param status specific to the individual game.
     */
    abstract public void sendInitialStatus(String status);
    /**
     * Informs the player of its final status in the game.
     * If this message is sent the player was not disqualified 
     * during the game.
     * 
     * @param status specific to the inidividual game
     */
    abstract public void sendFinalStatus(String status);
    /**
     * Sends intermediate game status 
     * 
     * @param status specific information
     */
    abstract public void sendStatus(String status);
    /**
     * Sends the latest change in status
     * 
     * @param status game specific
     */
    abstract public void sendIncrementalStatus(String status);
    /**
     * Uses the protocol to inform the player they have made an
     * invalid move
     * 
     * @param information why the move was invalid
     * @return A better move from the player
     */
    abstract public String sendInvalidMove(String information);
    /**
     * Warns the player.  Could be a precursor to ejection from
     * the game.
     * 
     * @param information specific to an individual game.
     */
    abstract public void sendWarning(String information);
    /**
     * Taylors the debugging method to an implementation specific one
     * 
     * @param debugger A prepackaged system that allows easy logging
     * capability. 		
     */
    abstract public void setupLogging(LogBox debugger);
}
