package nanomunchers.ui;

import java.awt.Graphics;
import java.awt.Color;

import nanomunchers.graph.Edge;
import nanomunchers.graph.Graph;

/*
 * Created on Oct 30, 2004
 */

/**
 * All of the functions for drawing either Nanomunchers or the graph
 * are done through this class.
 *
 * @author David Kaplin
 */
public class Drawing {
    private int hoffset,woffset;
    private int width;
    private int height;
	private Graph graph;
	/**
	 * @param g Graph that may be drawn
	 */
	public void setGraph(Graph g){
	    graph = g;
	}
	/**
	 * May stretch the graph a bit out of proportion,
	 * tries to ensure the whole graph fitting.
	 * 
	 * @param g Graphics context
	 * @param c color for the edges and nodes of the graph
	 * @param width maximum for drawing
	 * @param height maximum for drawing
	 */
	public void drawGraph(Graphics g,Color c,int width,int height){
	    Color oldcolor = g.getColor();
	    g.setColor(c);
	    woffset = width/(graph.getWidth()+1);
        hoffset = height/(graph.getHeight()+1);
	    if (((width+1) > graph.getWidth()) && ((height+1) > graph.getHeight())){
            Edge[] edges = graph.getUniqueEdges();
            int sx,sy,dx,dy;
            for(int i=0;i<edges.length;i++){
                drawEdge(g,c,edges[i],1);
            }
	    }
	    g.setColor(oldcolor);
	}
	/**
	 * @param g Graphics context
	 * @param c Color for edge
	 * @param e being drawn
	 * @param thickness = 1 + 2*thickness pixels for the width of the edge.
	 */
	public void drawEdge(Graphics g, Color c, Edge e, int thickness){
	    Color oldcolor = g.getColor();
	    g.setColor(c);
	    int sx = woffset + woffset*e.getNodes()[0].getX();
        int dx = woffset + woffset*e.getNodes()[1].getX();
        int sy = hoffset + hoffset*e.getNodes()[0].getY();
        int dy = hoffset + hoffset*e.getNodes()[1].getY();
        g.drawLine(sx,sy,dx,dy);
        for(int t=0;t<(thickness/2);t++){
            g.drawLine(sx+t,sy,dx+t,dy);
            g.drawLine(sx-t,sy,dx-t,dy);
            g.drawLine(sx,sy+t,dx,dy+t);
            g.drawLine(sx,sy-t,dx,dy-t);
        }
        g.setColor(oldcolor);
	}
	
	/**
	 * Similar to draw edge, but also draws the end nodes.
	 * 
	 * @param g Graphics context
	 * @param c Color for edge
	 * @param e being drawn
	 * @param thickness = 1 + 2*thickness pixels for the width of the edge.
	 */
	public void highlightEdge(Graphics g, Color c, Edge e, int thickness){
	    Color oldcolor = g.getColor();
	    g.setColor(c);
	    drawEdge(g,c,e,thickness);
	    int nanoRad = (width < height) ? woffset/2 : hoffset/2;
	    
	    nanoRad/=3;
	    
	    int sx = woffset + woffset*e.getNodes()[0].getX();
        int dx = woffset + woffset*e.getNodes()[1].getX();
        int sy = hoffset + hoffset*e.getNodes()[0].getY();
        int dy = hoffset + hoffset*e.getNodes()[1].getY();
	    g.drawOval(dx-(nanoRad-1),dy-(nanoRad-1),2*(nanoRad-1),2*(nanoRad-1));
        g.drawOval(sx-(nanoRad-1),sy-(nanoRad-1),2*(nanoRad-1),2*(nanoRad-1));
        g.setColor(oldcolor);
	}
	/**
	 * @param g Graphics context
	 * @param c color for the bot's body
	 * @param x position of the center of the bot's body
	 * @param y position of the center of the bot's body
	 * @param nextX future position for the bot's body
	 * @param nextY future position for the bot's body
	 */
	public void drawBotAlive(Graphics g,Color c, int x, int y,int nextX, int nextY){
	    Color oldcolor = g.getColor();
	    g.setColor(c);
	    int nanoRad = (width < height) ? woffset/2 : hoffset/2;
	    if (nanoRad> 2){
	        nanoRad-=2;
	    }
	    g.fill3DRect(x-nanoRad/2,y-nanoRad/2,nanoRad,nanoRad,true);
	    g.setColor(new Color(255,255,255));
	//    g.fillOval(nextX-(nanoRad/2),nextY-(nanoRad/2),nanoRad,nanoRad);
	    g.fillOval(nextX-(nanoRad/3),nextY-(nanoRad/3),2*(nanoRad/3),2*(nanoRad/3));
	    g.setColor(oldcolor);
	}
	/**
	 * Draws a bot that has exhausted the food supply.
	 * 
	 * @param g Graphics context
	 * @param c color for the bot's body
	 * @param x position of the center of the bot's body
	 * @param y position of the center of the bot's body
	 * @param nextX future position for the bot's body.
	 * @param nextY future position for the bot's body. 
	 */
	public void drawBotDead(Graphics g,Color c, int x, int y,int nextX, int nextY){
	    Color oldcolor = g.getColor();
	    g.setColor(c);
	    int nanoRad = (width < height) ? woffset/2 : hoffset/2;
	    if (nanoRad> 2){
	        nanoRad-=2;
	    }
	    g.fill3DRect(x-nanoRad/2,y-nanoRad/2,nanoRad,nanoRad,false);
	    g.setColor(new Color(255,255,255));
//	    g.fillOval(nextX-(nanoRad/2),nextY-(nanoRad/2),nanoRad,nanoRad);
	    g.fillOval(nextX-(nanoRad/3),nextY-(nanoRad/3),2*(nanoRad/3),2*(nanoRad/3));
	    g.setColor(oldcolor);
	}
	/**
	 * Draws a bot that was killed by conflict
	 * 
	 * @param g Graphics context
	 * @param c color for the bot's body
	 * @param x position of the center of the bot's body
	 * @param y position of the center of the bot's body
	 * @param nextX future position for the bot's body.
	 * @param nextY future position for the bot's body. 
	 	 */
	public void drawBotSlain(Graphics g,Color c, int x, int y,int nextX, int nextY){
	    Color oldcolor = g.getColor();
	    int nanoRad = (width < height) ? woffset/2 : hoffset/2;
	    if (nanoRad> 2){
	        nanoRad-=2;
	    }
	    g.setColor(c);
	    g.fill3DRect(x-nanoRad/2,y-nanoRad/2,nanoRad,nanoRad,false);
	    g.setColor(new Color(0,0,0));
	    g.fillOval(nextX-(nanoRad/3),nextY-(nanoRad/3),2*(nanoRad/3),2*(nanoRad/3));
	    g.setColor(oldcolor);
	}
}
