package nanomunchers.graph;

import java.util.Vector;

/*
 * Created on Oct 26, 2004
 */

/**
 * A Graph that looks like:
 * <pre>
 * o-o-o-o
 * | | | |
 * o-o-o-o
 * | | | |
 * o-o-o-o
 * </pre>
 * 
 * Formally, No node has a degree higher than 4.  
 * All nodes have at least 2 neighbors.
 * And the graph is Planar.
 * 
 * @author David Kaplin
 */
public class GridGraph implements Graph {
    protected Node[] nodes;
    protected Edge[] edges;
    protected int graphwidth;
    protected int graphheight;
    
    /**
     * @param width
     * @param height
     */
    protected GridGraph(int width, int height){
        //Step 1 Calculate Nodes
        this.graphwidth = width;
        this.graphheight = height;
        int nodeCount = (width)*(height);
        nodes = new Node[nodeCount];
        int id=0;
        for(int h=0;h<height;h++){
            for(int w=0;w<width;w++){
              nodes[id] = new Node(this,id,w,h);
              id++;
           }
        }
        //Step 2 Calculate Edges
        int numEdges = (2*(height -1)*(width-1 )) + (height -1 ) + (width - 1);
        edges = new Edge[numEdges];
        int edgeCount = 0;
        for(int i=0;i<(nodes.length - 1);i++){
            Node n = nodes[i];
            int yloc = n.getY();
            int xloc = n.getX();
            if ((xloc< (width-1)) && (yloc < (height -1))){
                //right side
                edges[edgeCount] = new Edge(n,nodes[xloc+1+ width*yloc]);
                edgeCount++;

                edges[edgeCount] = new Edge(n,nodes[xloc+ width*yloc + width]);
                edgeCount++;
            }else if (n.getX() == (width -1)){
                //bottom side (now filling in the edges that would
                //make the rightmost edge of the rectangle)
                edges[edgeCount] = new Edge(n,nodes[xloc+ width*yloc + width]);
                edgeCount++;
            }else{
                //add the right edge (now filling in the bottom of the
                //graph)
                edges[edgeCount] = new Edge(n,nodes[xloc+1+ width*yloc]);
                edgeCount++;
            }
        }
    }
    /** 
     * @see nanomunchers.graph.Graph#getNodes()
     */
    public Node[] getNodes() {
        return nodes;
    }

    /** 
     * @see nanomunchers.graph.Graph#getUniqueEdges()
     */
    public Edge[] getUniqueEdges() {
        return edges;
    }

    /** 
     * Really simple.  The entire graph is fully connected.
     * 
     * @see nanomunchers.graph.Graph#getConnectedComponents()
     */
    public Graph[] getConnectedComponents() {
        Graph[] ret = new Graph[1];
        ret[0] = this;
        return ret;
    }

    /** 
     * @see nanomunchers.graph.Graph#clear()
     */
    public void clear() {
        for(int i=0;i<nodes.length;i++){
            nodes[i].clear();
        }
    }

    /** 
     * @see nanomunchers.graph.Graph#getNeighbors(nanomunchers.graph.Node)
     */
    public Edge[] getNeighbors(Node n) {
        Vector neighbors = new Vector();
        for(int i=0;i<edges.length;i++){
            Edge test = edges[i];
            if (test.doesContain(n)){
                neighbors.add(test);
            }
        }
        Edge[] ret = new Edge[neighbors.size()];
        for(int i=0;i<ret.length;i++){
            ret[i] = (Edge)neighbors.get(i);
        }
        return ret;
    }
    /** 
     * @see nanomunchers.graph.Graph#getWidth()
     */
    public int getWidth(){ return graphwidth;}
    /** 
     * @see nanomunchers.graph.Graph#getHeight()
     */
    public int getHeight(){ return graphheight;}
}
