#include <stdio.h>
#include <assert.h>

#define CUTOFF 0

int getNumContigs (char *filename){
  FILE *data;
  char name[1024];
  int start, inc, num = 0;
  
  if (!(data = fopen (filename, "r"))){
    fprintf (stderr, "makeranges: Error opening %s!\n", filename);
    exit (1);
  }

  while (!feof (data)){
    if (fscanf (data, "%s\n", name) == 1 && strlen (name) > 0){
      num++;
    }
  }
  
  fclose (data);
  return num;
}

int getSeqLen (char *filename){
  FILE *data;
  char buffer[32768];
  char ch;
  int i, len = 0;
  
  if (!(data = fopen (filename, "r"))){
    fprintf (stderr, "makeranges: Error opening %s!\n", filename);
    exit (1);
  }

  fgets (buffer, 32768, data);
  if (buffer[0] != '>'){
    fprintf (stderr, "%s: Not FASTA format!\n", filename);
    exit (1);
  }

  while (!feof (data)){
    fgets (buffer, 32768, data);
    if (buffer[0] == 0) break;
    for (i = strlen (buffer) - 1; i >= 0; i--){
      if (isalpha (buffer[i]) || buffer[i] == '.'){
	len++;
      }
    }
  }
  
  fclose (data);
  return len;
}

void readConservation (int **data, char *filename, int contig, int amt, int seqLen){
  FILE *file;
  int sStart, sEnd, xStart, xEnd, yStart, yEnd, i;

  if (!(file = fopen (filename, "r"))){
    fprintf (stderr, "makeranges: Error opening %s!\n", filename);
    exit (1);
  }

  fscanf (file, "%d\n", &i);
  while (!feof (file)){
    if (fscanf (file, "(%d %d) --> (%d %d), (%d %d)\n",
		&sStart, &sEnd, &xStart, &xEnd, &yStart, &yEnd) == 6){
      assert (xStart >= 1 && xEnd <= seqLen && yStart >= 1 && yEnd <= seqLen);

      // shifting to zero-based
      for (i = xStart; i <= xEnd; i++){
	data[contig][i] += amt;
      }
    }
  }
  fclose (file);
}

void readContigs (int **data, char *filename, int numContigs, int seqLen, int cutoff, int inc){
  FILE *file;
  char name[1024], buffer[1024];
  int start, i;
  
  if (!(file = fopen (filename, "r"))){
    fprintf (stderr, "makeranges: Error opening %s!\n", filename);
    exit (1);
  }

  i = 0;
  while (!feof (file) && i < numContigs){
    if (fscanf (file, "%s\n", name) == 1 && strlen (name) > 0){
      start = cutoff;
      while (start < 100){
	sprintf (buffer, "%s.%d", name, start);
	readConservation (data, buffer, i, (start/10)*(start/10), seqLen);
	start += inc;
      }
      i++;
    }
  }
  
  fclose (file);
}

void filterRanges (int **data, int numContigs, int seqLen){
  int i, j, match, start = 0, end, score = 0;

  for (j = 0; j < numContigs; j++){
    for (i = 1; i <= seqLen; i++){
      match = (data[j][i] > 0) == (data[j][start] > 0);
      if (match) score += data[j][i];
      else {
	end = i - 1;
	if (score >= 0 && score < CUTOFF){
	  for (i = start; i <= end; i++) data[j][i] = 0;
	}
	score = 0;
	start = i;
      }
    }
  }
}

inline int match (int **data, int numContigs, int i, int j){
  int k;
  if (data[0][i] == -1 || data[0][j] == -1) return 0;
  for (k = 0; k < numContigs; k++)
    if ((data[k][i] > 0) != (data[k][j] > 0)) return 0;
  return 1;
}

inline int allzeroes (int **data, int numContigs, int pos){
  int i;

  for (i = 0; i < numContigs; i++)
    if (data[i][pos] != 0) return 0;
  return 1;
}

inline void print (int start, int end, int *score, int numContigs){
  int j;

  printf ("(%7d %7d)", start, end);
  for (j = 0; j < numContigs; j++) printf (" %7d", score[j]);
  printf ("\n");
}

void printRanges (int **data, int numContigs, int seqLen){
  int i, j, start = 0, end;
  int *score = (int *) malloc (sizeof (int) * numContigs);
  int *pattern = (int *) malloc (sizeof (int) * numContigs);

  assert (score);
  assert (pattern);
  
  printf ("numContigs = %d\n", numContigs);
  printf ("seqLen = %d\n", seqLen);

  for (i = 0; i < numContigs; i++) score[i] = 0;
  for (i = 0; i <= seqLen; i++)
    if (!allzeroes (data, numContigs, i)) break;
  if (i > 0) print (0, i - 1, score, numContigs);

  start = end = i;
  while (i <= seqLen){
    if (match (data, numContigs, start, i)){
      end = i;
      for (j = 0; j < numContigs; j++) score[j] += data[j][i];
    }
    else if (!allzeroes (data, numContigs, i)){
      print (start, end, score, numContigs);
      for (j = 0; j < numContigs; j++) score[j] = 0;
      if (end < i - 1) print (end + 1, i - 1, score, numContigs);
      start = end = i;
    }
    i++;
  }

  free (score);
  free (pattern);
}

int main (int argc, char **argv){
  int numContigs, seqLen, i, cutoff, increment;
  int **data;

  if (argc != 5){
    fprintf (stderr, "Usage:\nmakeranges filelist fastaseqfile cutoff increment\n");
    exit (1);
  }
  
  numContigs = getNumContigs (argv[1]);
  seqLen = getSeqLen (argv[2]);
  cutoff = atoi (argv[3]);
  increment = atoi (argv[4]);

  data = (int **) malloc (sizeof (int *) * numContigs); assert (data);
  for (i = 0; i < numContigs; i++){
    data[i] = (int *) malloc (sizeof (int) * (seqLen + 1)); assert (data[i]);
    memset (data[i], 0, sizeof (int) * (seqLen + 1));
    data[i][seqLen] = -1;
  }

  readContigs (data, argv[1], numContigs, seqLen, cutoff, increment);
 
  assert (cutoff >= 0);
  if (cutoff > 0) filterRanges (data, numContigs, seqLen);
  printRanges (data, numContigs, seqLen);

  for (i = 0; i < numContigs; i++) free (data[i]);
  free (data);
  
  return 0;
}
