#include "common.h"
#include "graphgrep.h"
#include "algebra.h"

///////////////////////////////////////////////
//
//	Global variables
//
///////////////////////////////////////////////
bool QUERY_WITH_DEGREE=false;		//flag to control if querying with degree information
bool OUTPUT_DATA_COUNT=false;		//flag to control if showing the current matched record with each operation
bool QUIET_MODE=true;				//flag to control if showing the debug information
bool DEBUG_OPFILE=false;				//flag to control if showing the debug information of OPFILE

int DEFAULT_LP=4;						//declaration of the Default LP

int NUM_INTS=256;						//The Number of Intergers of a FingerPrint. Default is 256. Used in the Frowns Filtering process.
int INT_SIZE=sizeof(int)*8;				//The size of an int (in bits). Default is 32. Used in Frowns Filtering process.
int NUM_FP=400;						//The group size of fingerprints. Default is 4000. Used in Frowns Filtering process.

char opfilename[100];					//op filename
char queryfilename[100];					//query filename
bool FILTERING=true;					//flag to control if using filtering mechinism
int nGraph=1;							//Number of graphs that you want to match in DB. This is only useful when FILTERING set false

const int NodeTypeBitLen=sizeof(NodeType)*8; //The size of an NodeType (in bits). The information is needed when storing data into BekerlyDB


vector<int> nGraphList;					//The graph list for matching
///////////////////////////////////////////////


////////////////////////////////////////////
//
// Class: Filtering
// Usage: Used for filtering graphs before matching
//
/////////////////////////////////////////////
class Filtering{
private:
	unsigned int* _fp;		//pointer to an FingerPrint
	bool checkFP(unsigned int* query_fp,unsigned int* graph_fp);	//function to check if 2 fps are matched
public:
	Filtering(unsigned int* fp):_fp(fp){};	//Constructor. Must pass a FP as parameter
	void run();							//Start filtering process
};

////////////////////////////////////////////
//
// Class: GraphGrep
// Usage: Do subgraph matching
//
/////////////////////////////////////////////
class GraphGrep
{
public:
	GraphGrep(){};							//Default Constructor
	void run();								//Start Matching
};

////////////////////////////////////////////
//
//	Implementation of Filtering Class
//
/////////////////////////////////////////////

//Checking 2 fp to see if the graph_fp contains the query_fp. 
bool Filtering::checkFP(unsigned int* graph_fp,unsigned int* query_fp){	
  unsigned int logicAnd=0;

  //check each int of the fp int array
  for(int i = 0; i < NUM_INTS; i++)	
    { 
	  if(query_fp[i]!=0){
		  logicAnd = graph_fp[i] & query_fp[i];  	//boolean operation
		  if(logicAnd != query_fp[i])
			  return false;
	  }
    }

  return true;
}

void Filtering::run(){
	//Initial BerkeleyDB and setting its parameters
	Db db(0, 0);
	db.set_error_stream(&cerr);
	db.set_errpfx("GraphGrep:");
	db.set_pagesize(1024);		/* Page size: 1K. */

	db.set_cachesize(0, 32 * NUM_INTS*NUM_FP, 0);
	db.open(NULL, "fingerprint.db", NULL, DB_BTREE, DB_RDONLY, 0664);

	try {
		Dbt key;		//key
		Dbt data;		//value
		Dbc *dbcp;		//DB cursor
		db.cursor(NULL, &dbcp, 0);

		int count=0,f_count=0;

		db.cursor(NULL, &dbcp, 0);
		while (dbcp->get(&key, &data, DB_NEXT) == 0) {		//fetch every data
			char *key_string = (char *)key.get_data();	

			//Here we divide the FP into groups, each group contains NUM_FP fingerprints.
			int igraph_end=atoi(key_string);
			int remainder=igraph_end % NUM_FP;
			int base=igraph_end / NUM_FP;
			
			int igraph_count=(remainder==0?NUM_FP: (igraph_end - (base)*NUM_FP));
			int igraph_start=igraph_end - igraph_count;

			//for debuging
			if(!QUIET_MODE){
				cout<< "igraph_start" <<igraph_start<<endl;
				cout<< " igraph_end" <<igraph_end <<endl;
				cout<< " igraph_count" <<igraph_count <<endl;
				cout<<"fp" <<*_fp<<endl;
			}

			unsigned int *data_string=(unsigned int*)data.get_data();
			
			for(int i=0;i<igraph_count;i++){
				unsigned int *p_fp=data_string+i*NUM_INTS;

				if(checkFP(p_fp,_fp)){
					//printf("key: %s len:%d\n",key_string,strlen(key_string));
					f_count++;
					nGraphList.push_back(igraph_start+i);
				}
			}

			//printf("data size:%d\n",strlen(data_string));
	
			count++;		
		}
		//cout<<"total graphs:"<<count*NUM_FP<<endl;
		cout<<"Graphs after filtered:"<<f_count<<endl<<endl;

	}
	catch (DbException &dbe) {
		cerr << "GraphGrep: " << dbe.what() << "\n";
	}
	//db.close(0);
}

////////////////////////////////////////////
//
//	Implementation of GraphGrep Class
//
/////////////////////////////////////////////
void GraphGrep::run()
{
	setclock("DB start time",0);			
	Db db(0, 0);

	db.set_errpfx("GraphGrep:");
	db.set_pagesize(1024);		/* Page size: 1K. */

	db.set_cachesize(0, 32 * 1024, 0);
	db.open(NULL, "graphdata.db", NULL, DB_BTREE, DB_RDONLY, 0664);

	try {
		setclock("DB start time",1);			

		setclock("#Matching_time",0);

		OPFile opfile(opfilename,queryfilename,&db);

		vector<string> graphIDList;
	
		
		if(FILTERING){		
			graphIDList.reserve(nGraphList.size());
			for(int x=0;x<nGraphList.size();x++){
				char tabname[10];
				sprintf(tabname,"gn%d",nGraphList[x]);
				if(!QUIET_MODE)
					cout<< "gn"<<nGraphList[x];

				graphIDList.push_back(string(tabname));
			}
		}else{		
			graphIDList.reserve(nGraph);
			for(int x=0;x<nGraph;x++){
				char tabname[10];
				sprintf(tabname,"gn%d",x);
				if(!QUIET_MODE)
					cout<< "gn"<<nGraphList[x];
				graphIDList.push_back(string(tabname));
			}		
		}
		
		opfile.setGraphIDs(graphIDList);

		for(int i=0;i<opfile.getGraphIDs().size();i++){
			opfile.setCurrentGraphID(opfile.getGraphIDs()[i]);
			opfile.run();
			opfile.reset();
		}

	}
	catch (DbException &dbe) {
		cerr << "GraphGrep: " << dbe.what() << "\n";
	}
	setclock("#Matching_time",1);

	setclock("closeDB time",0);	
	//db.close(0);
	setclock("closeDB time",1);	
}

////////////////////////////////////////////
//
//	Static functions
//
/////////////////////////////////////////////

//read graph from a query file
static GraphADT* readGraph(ifstream& f){	
	GraphADT* retG;
	char line[51];

	while(!f.eof()){
		f.getline(line,50);
		if(line[0]=='#'){
			f.getline(line,50);
			int numNode=atoi(line);
			retG=new GraphADT(numNode,false);
			for(int i=0;i<numNode;i++){	//read node
				f.getline(line,50);
				(*retG).addNode(i,line);
			}
			f.getline(line,50);
			int numEdge=atoi(line);
			for(int j=0;j<numEdge;j++){
				f.getline(line,50);
				string tmp(line);
				vector<string> nodes=sep_words(tmp);
				(*retG).insert(Edge(nodes[0],nodes[1]));
			}

		}
	}

	return retG;
}

//generating opfile from a input query file
static bool GenOpfile(char* queryfile){  
	setclock("#Query_creating_time",0);
	GraphADT* g;

	if(queryfile!=""){
		ifstream fin(queryfile);
		if(! fin){
			cout<<queryfile <<" does not exist\n";
			exit(0);
		}		
		g=readGraph(fin);
		fin.close();
	}else{
	    return false;
	}

	//IO<GraphADT>::show(*g);
	vDFS_GraphGrep<GraphADT> myDFS(*g,DEFAULT_LP);

	char tmp[100];
	sprintf(tmp,"%s.out",queryfile);
	myDFS.writeOutputOrder(tmp);

	setclock("#Query_creating_time",1);

	if(FILTERING){
		setclock("#Filtering_time",0);
		unsigned int* test_fp=myDFS.getFingerPrint();
		Filtering filter(test_fp);
		filter.run();
		setclock("#Filtering_time",1);
	}

	printtime("#Query_creating_time");
	printtime("#Filtering_time");
	return true;
}

////////////////////////////////////////////
//
//	Main Function
//
/////////////////////////////////////////////
static double app_start;
static double app_end;

int main(int argc, char** argv)
{
	try {
		setclock("total running time",0);			
		app_start=clock();

		GraphGrep app;

		bool HasQueryFile=false;
		for (int n = 1; n <argc; n++) {
			if (strcmp(argv[n],"-v")==0) {
				QUIET_MODE=false;
			}
			else if (strcmp(argv[n],"-o")==0) {
				OUTPUT_DATA_COUNT=true;
			}
			else if (strcmp(argv[n],"-noFL")==0) {
				FILTERING=false;
			}
			else if (strcmp(argv[n],"-lp")==0) {
				DEFAULT_LP=atoi(argv[++n]);
			}
			else if (strcmp(argv[n],"-fp")==0) {
				NUM_INTS=atoi(argv[++n]);
			}
			else if (strcmp(argv[n],"-n")==0) {
				nGraph=atoi(argv[++n]);
				nGraphList.reserve(nGraph);
			}
			else if (strcmp(argv[n],"-m")==0) {				
				strcpy(queryfilename,argv[++n]);

				char fileout[100];
				sprintf(fileout,"%s.out",queryfilename);

				strcpy(opfilename,"opfile");
				
				(void)remove(fileout);
				HasQueryFile=true;

			}else if(strcmp(argv[n],"-degree")==0) {
				QUERY_WITH_DEGREE=true;
			}
			else {		
				cout<< "Usage: GraphGrep -m queryfile [-o -v -noFL -degree -n numGraphs -lp lengthpath -fp length of FP]\n"<<
					"\t-o               output count\n"<<
					"\t-v               verbose mode\n"<<
					"\t-degree          query with degree\n"<<
					"\t-noFL            disable filtering\n"<<
					"\t-n  numGraphs    input number of graphs in DB(only with option -noFL)\n"<<
					"\t-lp lengthpath   input different LP, default is 4\n"<<
					"\t-fp length of FP input different length of FP, default is 256"<<endl;
				exit(2);
			}
		}  // end for(....)

		if(HasQueryFile){
			if(GenOpfile(queryfilename))
				app.run();
			else{
				cout<<"error happened in generating opfile\n";
				exit(0);
			}

			setclock("total running time",1);	
			app_end=clock();

			printtime("#Matching_time");
			//printtime("create from DB time");		
			
			cout<<"#Total_running_time :"<<(app_end-app_start)/(CLOCKS_PER_SEC)<<"\n";
		}else{
				cout<< "Usage: GraphGrep -m queryfile [-o -v -noFL -degree -n numGraphs -lp lengthpath -fp length of FP]\n"<<
					"\t-o               output count\n"<<
					"\t-v               verbose mode\n"<<
					"\t-degree          query with degree\n"<<
					"\t-noFL            disable filtering\n"<<
					"\t-n  numGraphs    input number of graphs in DB(only with option -noFL)\n"<<
					"\t-lp lengthpath   input different LP, default is 4\n"<<
					"\t-fp length of FP input different length of FP, default is 256"<<endl;
		
		}

		return (EXIT_SUCCESS);

	}
	catch (DbException &dbe) {
		cerr << "GraphGrep: " << dbe.what() << "\n";
		return (EXIT_FAILURE);
	}
}
