/**
 * 
/**
 * StratPal
 * @author Dennis Shasha
 * @author Anastasia Kuznetsov
 * 10/01/2005
 */

package StratPal;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.LinkedList;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

/**
 * Input parser reads input from specified file, creates QNodes for each question
 * and forms correct relationships between questions based on if statements.
 *
 * For now, only ASCII format is accepted.
 * 
 * Questions must start with a unique question name, which consists of one alphanumeric
 * character followed by one or more integers. 
 * For now, an opptional subclause (ex: Q1, Q1.a, Q1a) is not supported.
 *  
 * The question must be followed by a blank line.
 * 
 * Each question may be preceded by one or more if statements that detail possible
 * answers to the previous questions that would lead to this quesiton. 
 * The if statements must be of the format:
 * [if "keyword" to "question name"]
 * 
 * Go to statements are not supported at this time.
 * 
 *  
 */

public class InputParser {
	

	private BufferedReader bufferIn;
	private FileReader freader;
	
	/**
	 * Question DAG that represents all questions.
	 */
	private QuestionSet qset;
	
	/**
	 * List to keep track of questions who's answers have not yet all been defined.
	 */
	private LinkedList incomplete;
	
	
	/** Ignore invalid references to questions in if Statements. */
	private boolean ignore = false;
	
	/** Input Stream from an URL. */
	private InputStreamReader inStream=null;
		
	private void init (URL url){
		try{
	    	inStream = new InputStreamReader(url.openStream());
	    	bufferIn = new BufferedReader(inStream);
	    	
	    	}
	    	catch(Exception e){
	    		e.printStackTrace();
	    	}
	    freader = null;
	    qset = new QuestionSet();
		incomplete = new LinkedList();
	}
	private void init(File fname){
		try{
			freader = new FileReader (fname);
			bufferIn = new BufferedReader(freader);
		
		} catch (Exception e){
			e.printStackTrace();
			}
		qset = new QuestionSet();
		incomplete = new LinkedList();
		
	}
	private void init(BufferedReader reader){
		try{
			
			bufferIn = reader;
		
		} catch (Exception e){
			e.printStackTrace();
			}
		qset = new QuestionSet();
		incomplete = new LinkedList();
	}
	
	/**
	 * Close input streams.
	 */
	private void exit(){
		try {
			bufferIn.close();
			if(freader != null)
				freader.close();
			ignore = false;
			if(inStream!= null)
				inStream.close();
			
		} catch (IOException e) {
			
			e.printStackTrace();
		}
	}
	
	/**
	 * Parses specified file into a directed acyclic graph of questions.
	 * @param reader File of questions.
	 * @return QuestionSet DAG of questions
	 * @throws Exception if there are errors reading the file.
	 */
	public QuestionSet parse(BufferedReader reader) throws Exception{
		init(reader);
		return parse();
		
	}
	
	/**
	 * Parses a text document of questions.
	 * 
	 * @param url URL of question document.
	 * @return set of questions in parsed form.
	 * @throws Exception if question document is not well-defined.
	 */
	public QuestionSet parse(URL url) throws Exception {
		
		init(url);
		return parse();
	}
	
	private QuestionSet parse () throws NullPointerException, Exception{
		String line = bufferIn.readLine();
		while(true){
			LinkedList lines = new LinkedList();
			
			if(!bufferIn.ready())
				break;
			
			while(line!= null && isBlank(line)){
				line = bufferIn.readLine();		
			}
	
			while(line!=null && !isBlank(line)){
				lines.addLast(line);
				line = bufferIn.readLine();		
			}
			process(lines);
		}
	
		exit();
		qset.setIncrement();
		return qset;
	}
		
		
	/**
	 * Parses lines that represent a question and optional if statements.
	 * @param lines Question block.
	 * @throws Exception 
	 * @throws NullPointerException 
	 */
	private void process(LinkedList lines) throws NullPointerException, Exception {
		int index=0;
		while(isIf((String)lines.get(index))){
			index++;
		}
		String name = processQuestion(index, lines);
		processIfs(index, lines, name);
		
		
	}

	/**
	 * Parses if statements and establishes relationships in DAG
	 * @param index Index where question starts/if statements end.
	 * @param lines Question block.
	 * @param qname	Name of question that was just parsed.
	 */
	private void processIfs (int index, LinkedList lines, String qname) 
	throws NullPointerException, Exception {
		for (int i =0; i<index; i++){
			String line = (String)lines.get(i);
			
			if((!line.startsWith("[if")||!line.startsWith("[If"))&&line.split(" to ").length!=2){
				throw new Exception("Could not parse if statement: "+line);
			}
			line = line.substring(4, line.length()-1);
			String keyword = line.split(" to ")[0];
			String question = line.split(" to ")[1];
			question = question.split(" ", 2)[0];
			question.trim();
			
			QNode qnode = qset.get(question);
			if (qnode == null)
				if(!ignore)
				throw new NullPointerException("Statement: "+(String)lines.get(i)+" refers to unknown question "+
					question);
				else continue;
			if(!qnode.containsAnswer(keyword))
				qnode.addAnswer(keyword);
			qset.get(qname).addCondition(question, keyword);
		}
						
	}

	/**
	 * Removes punctuation marks and spaces from a string.
	 * @param string String to be cleaned.
	 * @return	string that does not contain any spaces or punctuation marks.
	 */
	private String removeBlanks(String string) {
		string = string.replaceAll("[., ]]*", "");
		return string;
	}


	/**
	 * Parses question.
	 * @param index where questions start.
	 * @param lines question block.
	 * @return name of question that was just parsed
	 */
	private String processQuestion(int index, LinkedList lines)
	throws NullPointerException, Exception{
		boolean isGoTo = false;
		boolean isChild = false;
		int numNewLines =0;
		
		String line = (String)lines.get(index);
		
		if(index ==lines.size()-1){
			numNewLines += (line.length()/75);
			if(line.length()%75>0){
				numNewLines++;
			}
		}
		String name = line.split(": ", 2)[0];
		String question = line.split(" ", 2)[1];
		index++;
		for(int i = index; i<lines.size(); i++){
			question += '\n';
			numNewLines++;
			String text = (String)lines.get(i);
			numNewLines += (text.length()/75);
		
			if(text.length()%75>0){
				numNewLines++;
			}
		
			if(isGoTo(text)){
				index = i;
				isGoTo = true;
				break;
			}
			question += text;
		}
	
		int scope = (new StringTokenizer(name, ".", false)).countTokens();
		
		QNode node = new QNode(name, question);
		node.setNewLines(numNewLines);
		
		if(!incomplete.isEmpty()){
			String pname = (String)incomplete.getFirst();
			
			while(!incomplete.isEmpty() &&
					(new StringTokenizer(pname, ".", false)).countTokens() >= scope){
				incomplete.removeFirst();
				if(!incomplete.isEmpty())
					pname = (String)incomplete.getFirst();
			}
			if(!incomplete.isEmpty()){
				String parent = (String)incomplete.getFirst();
				qset.get(parent).addChild(name);
				isChild = true;
				
			}
		}
		incomplete.addFirst(name);	
		qset.add(name, node);
		if(!isChild)
			qset.addOrdered(name);
		if(isGoTo)
			processGoTo(index, lines, name);
		
		
		return name;
	}


	/**
	 * Process go to  statements.
	 * 
	 * @param index index of go to statement in the list of lines
	 * @param lines linked list of lines
	 * @param curName name of the current question being parsed
	 * @throws NullPointerException  if go-to statement refers to a non-existent question
	 * @throws Exception if go-to statement is not formed correctly
	 */
	private void processGoTo(int index, LinkedList lines, String curName)
		throws NullPointerException, Exception{
		
		for (int i =index; i<lines.size(); i++){
			String gotoQuestion = null;
			String line = (String)lines.get(i);
			if((!line.startsWith("[if")||!line.startsWith("[If"))&&line.split(" to ").length!=3&&
					line.split(" go to ", 2).length!=2){
				
				throw new Exception("Could not parse if statement: "+line);
			}
			line = line.substring(4, line.length()-1);
			String keyword = line.split(" to ")[0];
			String toQuestion = line.split(" to ")[1];
			gotoQuestion = line.split(" go to ", 2)[1];
			gotoQuestion = removeBlanks(gotoQuestion);
			toQuestion = toQuestion.split(" ", 2)[0];
			toQuestion=removeBlanks(toQuestion);
			
			QNode qnode = qset.get(toQuestion);
			if(qnode==null)
				if(!ignore)
				throw new NullPointerException("Statement: "+(String)lines.get(index)+" refers to unknown question "+
						toQuestion);
				else
					continue;
			if(!qnode.containsAnswer(keyword))
				qnode.addAnswer(keyword);
			(qset.get(curName)).addGoTo(toQuestion, keyword, gotoQuestion);
		
		}
						
	}
	
	/**
	 * Checks if string is an if statement.
	 * @param string line in the file.
	 * @return true if line is an if statement.
	 */
	private boolean isIf(String string) {
		return string.toLowerCase().startsWith("[if ");
	}
	
	/**
	 * Checks if string is blank.
	 * @param line line of text.
	 * @return true if line contains only white space characters.
	 */
	public boolean isBlank(String line){
		return(line.matches("^[ \t\n\f\r]*$"));
	}
	
	/**
	 * Checks if string is a go-to statement.
	 * 
	 * @param line line of text.
	 * @return true if line contains only white space characters.
	 */
	public boolean isGoTo(String string){
		return (string.toLowerCase().matches("^.*if.* go to .*"));
	}


	/**
	 * If user says to ignore invalid references to questions.
	 *
	 */
	public void ignoreInvalidReferences() {
		ignore = true;
		
		
	}

}
