//graph_search_utility.h
#pragma once

#include <vector>
#include <algorithm>
#include <utility>
#include <string>
#include <sstream>
#include <iostream>
#include <fstream>

#include <boost/regex.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/depth_first_search.hpp>
#include <boost/graph/visitors.hpp>

#include "GraphGrepSXLibrary.h"

using namespace std;
using namespace boost;
using namespace GraphGrepSXLibrary;

namespace GraphGrepSXLibrary {

typedef int ERRCODE;

void throw_gb_error(string message, ERRCODE ec = 0)
{
	cout << message << endl;
	exit(0);
}

template <typename MutableGraph>
void resize_vertices(MutableGraph g,size_t n)
{
	g.m_vertices.resize(n);
}

template<typename RT, class T, typename Trait, typename Alloc> 
void tostring(const RT& source, std::basic_string<T, Trait, Alloc>& the_string)
{
  std::basic_ostringstream<T, Trait, Alloc> temp_ss;
  temp_ss << source;
  the_string = temp_ss.str();
}

template<typename RT> 
std::string tostring(const RT& source)
{
	string the_string;
	tostring(source, the_string);
	return the_string;
}


template<typename RT, typename T, typename Trait, typename Alloc>
RT stringto(const std::basic_string<T, Trait, Alloc>& the_string )
{
    std::basic_istringstream<T, Trait, Alloc> temp_ss(the_string);
    RT num;
    temp_ss >> num;
    return num;
}

typedef std::map<std::string, node_label_t> label_map_t;
node_label_t label_val = 'A';
label_map_t label_map;


template<class Archive> void
load_label_map(Archive &ar, label_map_t &map)
{
	size_t map_size;
	ar.read((char*)&map_size,sizeof(size_t));
	
	for(size_t j=0; j<map_size; j++)
	{
		char buffer[3];//sistemare
		ar.read(buffer,sizeof(string));
		string str_lbl(buffer);
		node_label_t lbl;
		ar.read(&lbl,sizeof(node_label_t));
		map.insert(pair<string, node_label_t>(str_lbl, lbl));						
	}
	
}

template<class Archive> void
save_label_map(Archive &ar, const label_map_t &map)
{
	label_map_t::const_iterator it;
	
	size_t f_size = map.size();
	ar.write((char*)&f_size,sizeof(size_t));

	for(it= map.begin(); it!=map.end(); it++)
	{
		string str_lbl = it->first;
		ar.write(str_lbl.c_str(),sizeof(string));
		node_label_t lbl =it->second;
		ar.write((char*)&lbl,sizeof(node_label_t));			
	}		
}


long get_labelmap_disksize(const label_map_t &map){
	long size=0;

	size+=sizeof(size_t);

	label_map_t::const_iterator it;

	for(it= map.begin(); it!=map.end(); it++)
	{
		string str_lbl = it->first;
		size+=(long)(sizeof(char) * str_lbl.length());
		size+=sizeof(char);
	}

	return size;
}

long get_labelmap_memorysize(const label_map_t &map){
	long size=sizeof(label_map_t);

	label_map_t::const_iterator it;
	for(it= map.begin(); it!=map.end(); it++){
		size+=sizeof(string) + sizeof(node_label_t);
		string str_lbl = it->first;
		size+=(long)(sizeof(char) * str_lbl.length());
	}

	return size;
}

template <typename MutableGraph> bool 
read_graph_gg(std::istream& in, MutableGraph& graph)
{
	property_map<graph_t, vertex_color_t>::type vertexLabelProp = get(vertex_color, graph);
	bool bVertexLabelProp = true;

	enum State {
		AttendGraph,
		LoadNumNodes,
		LoadNodes,
		LoadNumEdges,
		LoadEdges,
		End
	} state = AttendGraph;
	int count;
	int index;
	int r = 0;

	string graph_name;

	while (!in.eof() && state != End)
	{
		switch (state) 
		{
			case AttendGraph:
				{
					in>>graph_name;

					if(graph_name=="")
						break;

					r++;
					cout<<graph_name<<endl;
					if (graph_name[0] == '#')
					{
						state = LoadNumNodes;
					}
					else
					{
						string error = "Errore grafo: " + tostring(graph_name);
						error += string(" nella riga ") + tostring(r) + ". Atteso #ID.";
						throw_gb_error(error);
					}
					break;
				}
			case LoadNumNodes:
				{
					string num_nodes;
					in>>num_nodes;
					r++;
					count = stringto<int>(num_nodes);
					assert(count);
					index = 0;
					state = LoadNodes;
					//else
					//{
					//	string error = "Errore grafo: " + tostring(graph_name);
					//	error += string(" nella riga ") + tostring(r) + ". Atteso numero di nodi.";
					//	throw_gb_error(error);		
					//}
					break;
				}
			case LoadNodes:
				{
					add_vertex(graph);
					if (bVertexLabelProp)
					{
						string name;
						in>>name;
						r++;

						node_label_t lbl;
						string vertexlabel(name);
						label_map_t::iterator it = label_map.find(vertexlabel);

						if(it==label_map.end())
						{
							label_map.insert(pair<string, node_label_t>(vertexlabel, label_val));
							lbl = label_val;
							label_val++;
						}
						else
							lbl = it->second;

						put(vertexLabelProp,index,lbl);
					}

					index++;
					if(index == count)
						state = LoadNumEdges;
					break;
				}
			case LoadNumEdges:
				{
					string num_edge;
					in>>num_edge;
					r++;
					
					count = stringto<int>(num_edge);
					assert(count);
					index = 0;
					state = LoadEdges;

					break;
				}
			case LoadEdges:
				{
					string id1, id2;
					in>>id1;
					in>>id2;
					r++;

					int v1 = stringto<int>(id1);
					int v2 = stringto<int>(id2);
					add_edge(v1,v2,graph);
					index++;
					if(index == count)
						state = End;
					break;
				}
		};
	}

	if (state != End && !in.eof())
		throw_gb_error(string("Fine file inattesa."));

	return true;
}

template <typename MutableGraph> bool 
write_graph_gg(std::ostream& out, MutableGraph& graph)
{
	out<<"#"<<get_property(graph, graph_name)<<endl;
	
	int v = num_vertices(graph);
	out<<v<<endl;

	vertex_it vi, vi_end;
	for(boost::tie(vi,vi_end) = vertices(graph); vi != vi_end; ++vi) 
	{
		out<<get(vertex_name, graph, *vi)<<endl;
	}

	int e = num_edges(graph);
	out<<e<<endl;

	edge_it ei, ei_end;
	for(boost::tie(ei,ei_end) = edges(graph); ei != ei_end; ++ei) 
	{
		out<<source(*ei,graph)<<' '<<target(*ei,graph)<<endl;
	}

	return true;
}

bool to_vento_db(const string &db_file, const string &ts_file, graph_set_t &translate)
{
	ifstream db_stream;
	db_stream.open(db_file.c_str(), ios_base::in);
	if (!db_stream.is_open())
	{
		cout << "Impossibile aprire il file database: '" << db_file.c_str() << "'" << endl;
		return 0;
	}
	
	ofstream ts_stream;
	ts_stream.open(ts_file.c_str(), ios_base::out);
	if (!ts_stream.is_open())
	{
		cout << "Impossibile salvare il file db per vento: '" << ts_file.c_str() << "'" << endl;
		return 0;
	}
	
	int index = 0;
	bool filter=false;
	graph_set_t::iterator it = translate.begin();

	while (!db_stream.eof())
	{
		string line;
		getline(db_stream, line);
		
		string::size_type loc = line.find("#", 0 );   
		if(loc!=string::npos)
		{
			if(it==translate.end())
				break;

			if(*it==index)
			{
				filter=true;
				it++;	
			}
			else
			{
			
				filter=false;
			}

			index++;
		}
	
		if(filter)
		{
			ts_stream<<line<<endl;
		}
	}

	ts_stream.close();
	db_stream.close();
 	return true;
}


int verified_candidates_calc(const string &vento_out_file)
{
	ifstream out_stream;
	out_stream.open(vento_out_file.c_str(), ios_base::in);
	if (!out_stream.is_open())
	{
		cout << "Impossibile aprire il file database: '" << vento_out_file.c_str() << "'" << endl;
		return 0;
	}
	
	int index = 0;
	bool wait_for_match=false;

	while (!out_stream.eof())
	{
		string line;
		getline(out_stream, line);
		
		string::size_type loc = line.find("Graph:", 0 );   
		if(loc!=string::npos)
		{
			wait_for_match = true;
		}
		else
		{
			string::size_type loc = line.find("(");
			if(loc!=string::npos && wait_for_match)
			{
				index++;
				wait_for_match = false;
			}
		}
	}

	out_stream.close();
 	return index;
}

int matches_calc(const string &vento_out_file)
{
	ifstream out_stream;
	out_stream.open(vento_out_file.c_str(), ios_base::in);
	if (!out_stream.is_open())
	{
		cout << "Impossibile aprire il file database: '" << vento_out_file.c_str() << "'" << endl;
		return 0;
	}
	
	int count = 0;
	bool wait_for_match=false;

	while (!out_stream.eof())
	{
		string line;
		getline(out_stream, line);
		
		string::size_type loc = line.find("Graph:", 0 );   
		if(loc==string::npos)
		{
			count++;
		}
	}

	out_stream.close();
 	return count-1;
}

// ----------------- 
typedef std::ifstream::pos_type pos_type;
pos_type get_file_size( const std::string& filename )
{
	std::ifstream InFile( filename.c_str(),
	std::ios_base::binary | std::ios_base::ate );

	if( !InFile )
	throw std::runtime_error( "ERROR: Could not obtain file size "
	"for [" + filename + "]\n" );

	return InFile.tellg();
}

}//end name_space

#ifdef WIN32 // definita nelle direttive del preprocessore in Visual Studio 7.0

#include <Windows.h>
typedef LARGE_INTEGER TIMEHANDLE;

extern LARGE_INTEGER _freq;

inline TIMEHANDLE start_time()
{
    static int first = 1;

    if(first) {
        if (!QueryPerformanceFrequency(&_freq))
			cout << "No high-resolution performance counter installed" << endl;

		if (_freq.QuadPart == 0)
			cout << "High-resolution performance counter available but _freq = 0" << endl;
        first = 0;
    }
    TIMEHANDLE tstart;
	QueryPerformanceCounter(&tstart);
	return tstart;
}
inline double end_time(TIMEHANDLE th)
{
    TIMEHANDLE tend;
    QueryPerformanceCounter(&tend);

	double res = (double)(tend.QuadPart - th.QuadPart)/(_freq.QuadPart); 
	return res;
}

#pragma warning( disable : 4267 4018)

#else // sotto linux

#include <sys/time.h>

typedef struct timeval TIMEHANDLE;

extern struct timezone _tz;
inline TIMEHANDLE start_time()
{
    TIMEHANDLE tstart;
    gettimeofday(&tstart, &_tz);
    return tstart;
}
inline double end_time(TIMEHANDLE th)
{
    TIMEHANDLE tend;
    double t1, t2;

	gettimeofday(&tend,&_tz);

    t1 =  (double)th.tv_sec + (double)th.tv_usec/(1000*1000);
    t2 =  (double)tend.tv_sec + (double)tend.tv_usec/(1000*1000);
    return t2-t1;
}
#endif //WIN32

#ifdef WIN32
LARGE_INTEGER _freq;
#else
struct timezone _tz;
#endif
