function [diagnostic,R] = termination(iv, v, Mk, N, fi, lq, radius, solver, d)
% [diagnostic,R] = termination(x0, x, Mk, N[, fi, lq, radius, solver, d]) checks
%   termination of [B;C](x0,x)=/\k=1..N (x0 x 1) Mk(:,:,k) (x0 x 1)' >= 0 
%   /\ /\k=N+1..N+M (x0 x 1) Mk(:,:,k) (x0 x 1)' = 0, with optional input 
%   arguments fi = 'float' ['int'], lq = 'quadratic' ['linear', 'linear-positive', 
%   'quadratic-positive'] ranking function, rigid feasibility radius [inf=none], 
%   solver = 'lmilab' ['csdp', sdpt3', 'sedumi', 'sdplr', ...] and minimal rank 
%   decrement d (1 by default, can also be d = sdpvar(1,1)).
%       Returns the matrix R of the ranking function r(x) = xRx' with a   
%   feasibility diagnostic on the termination proof.
if nargin < 4, error('missing argument.'); end;
% display arguments, numerically
% format rational; 
format short e; 
% format long e; 
Mk
% and symbolically
display_Mk(Mk,N,iv,v);
% optional input arguments
if nargin < 5, fi = 'float'; end; % real ranking function
if nargin < 6, lq = 'quadratic'; end; % quadratic ranking function
if nargin < 7, radius = inf; end; % no rigid feasibility radius
if nargin < 8, solver = 'lmilab'; end; % default solver
if nargin < 9, d = 1; end; % default rank decrement
n = (size(Mk,1)-1)/2; % number of program variables
% N is the number of inequality constraints
M = size(Mk,3) - N; % number of equality constraints
if fi(1:3) == 'int'
   X = intvar(n+1,n+1); % Unkown integer ranking matrix R;
else
   X = sdpvar(n+1,n+1); % Unkown real ranking matrix R;
end;
if lq(1:3) == 'lin' % enforce linear ranking function
	X(1:n,1:n)=0;
end;
% for all x: r(x0) = (x0 x 1).M0.(x0 x 1)'
M0 = [X(1:n,1:n) zeros(n,n) X(1:n,n+1);
      zeros(n,n) zeros(n,n) zeros(n,1);
      X(n+1,1:n) zeros(1,n) X(n+1,n+1)];
% for all x0: r(x) = (x0 x 1).M_0.(x0 x 1)'
M_0 = [zeros(n,n) zeros(n,n+1);
       zeros(n+1,n) X];
% for all x0, x: d = (x0 x 1).delta.(x0 x 1)'
delta = [zeros(2*n,2*n) zeros(2*n,1);
         zeros(1,2*n) d];
l0 = sdpvar(N+M,1); % Unknown Lagrange multipliers for r(x) >= 0;
L0 = set([]); % Nonnegativity of Lagrange multipliers for inequality
for k = 1:1:N % constraints; 
 L0 = L0+set(sprintf('l0(%i,1)>=0',k));
end;
CP='M0'; % Lagrangian for constraint r(x) >= 0;
for k = 1:1:N+M
  CP = strcat(CP, sprintf('-l0(%i,1)*Mk(:,:,%i)',k,k));
end;
CP = strcat(CP,'>0'); P = set(CP);
l = sdpvar(N+M,1); % Lagrange multipliers for r(x) - r(x') >= 1;
L = set([]);    % Nonnegativity of unknown Lagrange multipliers 
for k = 1:1:N  % for inequality constraints;
 L = L+set(sprintf('l(%i,1)>=0',k));
end;
CD = 'M0-M_0-delta'; % Lagrangian for constraint r(x) - r(x') >= 1
for k = 1:1:N+M
  CD = strcat(CD, sprintf('-l(%i,1)*Mk(:,:,%i)',k,k));
end;
CD = strcat(CD,'>0'); D = set(CD);
F = P+D+L0+L; % Gather all constraints;
llq = size(lq,2);
if ((llq >= 8) & (lq(llq-7:llq) == 'positive'))
  F = F + set('X>=0');
 end
if ~(double(d)>=1)
  F = F + set('d>0');
end
[diagnostic,R] = solveconstraints(F, X, fi, radius, solver);
disp(' '); disp(diagnostic);
R
if fi(1:3) == 'int';
   intrank (R, v)
else
   fltrank (R, v) 
end
disp(' '); disp('Lagrange multipliers for constraint r(x0) >= 0:');
for k = 1:1:N+M % constraints; 
 disp(sprintf('   l0(%i,1) = %+e',k, double(l0(k,1))));
end;
disp(strcat('Lagrange multipliers for constraint r(x0) - r(x) >=',sprintf(' %d',double(d)),':'));
for k = 1:1:N+M % constraints; 
 disp(sprintf('   l(%i,1) = %+e',k, double(l(k,1))));
end;

   
%******************************************************************************
% Copyright 2004  Patrick.Cousot@ens.fr
%******************************************************************************
% Permission to use, copy, modify, and distribute this software and its
% documentation for any purpose and without fee is hereby granted,
% provided that the above copyright notice appear in all copies and that
% both that the copyright notice and warranty disclaimer appear in
% supporting documentation.
% 
% P. Cousot disclaims all warranties with regard to this software,
% including all implied warranties of merchantability and fitness.  In
% no event shall P. Cousot be liable for any special, indirect or
% consequential damages or any damages whatsoever resulting from loss of
% use, data or profits, whether in an action of contract, negligence or
% other tortuous action, arising out of or in connection with the use or
% performance of this software.
%******************************************************************************
