function [cndsql,comp,pinfeas,dinfeas,blkmat] =...
    sqlcond(A,b,C,blk,X,y,Z)
% SQLCOND   computes the condition number of the 3x3 block system
%           for SQLP evaluated at the point (X,y,Z)
%
% [cndsql,comp,pinfeas,dinfeas,blkmat] = sqlcond(A,b,C,blk,X,y,Z)
%
%  The routine also computes the primal and dual infeasibilities
%  at this point as well as the value of <X,Z> allowing the user
%  to "verify" the optimality conditions at (X,y,Z).
%  Here A is a structure where A.s is an m by ns matrix A.q is an
%  m by nq matrix, and A.l is an m by nl matrix.
%  Convention: the matrix A.s is empty if and only if ns = 0;
%  similarly for A.q and A.l.
%
%  The 3x3 block matrix has the following form
%
%           [ zeros(d,d)      B'         eye(d)   ]
%           [     B       zeros(m,m)   zeros(m,d) ]
%           [     E       zeros(d,m)       F      ]
%
%  where:
%  d = ns + nq + nl, B is the m by d matrix [A.s A.q A.l],
%  E and F are block diagonal matrices with 3 blocks E1, E2, E3,
%  and F1, F2, F3, respectively, with
%     E1 = skron(Z.s,eye(ns), F1 = skron(X.s,eye(ns)),
%     E2 = Arw(Z.q),          F2 = Arw(X.q),
%     E3 = diag(Z.l),         F3 = diag(X.l).
%
%  Here skron(M,N) denotes the symmetric Kronecker product of the
%  two matrices M, N, while Arw(x) denote the block diagonal matrix
%  whose blocks are the arrow matrices of the blocks of x.
%  The point (X,y,Z) is a solution to the SQLP satisfying strict
%  complementarity and primal and dual nondegeneracy if and only
%  if the 3x3 block matrix is nonsingular. Moreover, such a
%  solution is unique. Thus a very large condition number at a
%  point (X,y,Z) satisfying the optimality conditions is a
%  strong indication that the SDP is degenerate.
%
%  input variables
%     - A            matrix of primal constraints
%     - b            rhs of primal constraints
%     - C            cost matrix
%     - blk          block structure vector
%     - X            the computed primal solution
%     - y            the computed dual solution
%     - Z            the computed dual slack at solution
%
%  output variables
%     - cndsql       the condition number of the 3x3 block matrix
%     - comp         the inner product <X,Z>
%     - pinfeas      the norm of primal feasibility residual
%     - dinfeas      the norm of dual feasibility residual
%     - blkmat       the 3x3 block matrix

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/17/97
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
 else
    n.sm = 0;
    n.sv = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
 else
    n.q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
 else
    n.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    error('sqlcond: the block structure is empty.  Aborting...');
 end
 m = 0;
%
% consistency checks
%
 if n.sm > 0
    m = size(A.s,1);
    if n.sv ~= size(A.s,2)
       error('sqlcond: size of A.s is incompatible with blk.s. Aborting...')
    end
 end
 if n.q > 0
    m = max(m,size(A.q,1));
    if n.q ~= size(A.q,2)
       error('sqlcond: size of A.q is incompatible with blk.q. Aborting...')
    end
 end
 if n.l > 0
    m = max(m,size(A.l,1));
    if n.l ~= size(A.l,2)
       error('sqlcond: size of A.l is incompatible with blk.l. Aborting...')
    end
 end
 if (n.sm > 0 & m ~= size(A.s,1)) | (n.q > 0 & m ~= size(A.q,1))...
   | (n.l > 0 & m ~= size(A.l,1))
    error('sqlcond: A.s, A.q, A.l have different number of rows.  Aborting...\n');
 end
%
% compute feasibility residuals
%
 comp = 0;
 dinfeas = 0;
 if n.sm > 0
    vCs = svec(C.s,blk.s);
    vXs = svec(X.s,blk.s);
    vZs = svec(Z.s,blk.s);
    comp = comp + full(vXs'*vZs);
    rps = A.s*vXs;
    dinfeas = dinfeas + norm(vCs - vZs - A.s'*y);
 else
    rps = 0;
 end
 if n.q > 0
    comp = comp + X.q'*Z.q;
    rpq = A.q*X.q;
    dinfeas = dinfeas + norm(C.q - Z.q - A.q'*y);
 else
    rpq = 0;
 end
 if n.l > 0
    comp = comp + X.l'*Z.l;
    rpl = A.l*X.l;
    dinfeas = dinfeas + norm(C.l - Z.l - A.l'*y);
 else
    rpl = 0;
 end
 rp = b - rps - rpq - rpl;
 pinfeas = norm(rp);
%
% now set up the matrix B = [A.s A.q A.l]
%
 d = n.sv+n.q+n.l;
 B = zeros(m,d);
 idx = 0;
 if n.sm > 0
    B(:,(idx+1):(idx+n.sv)) = A.s;
    idx = idx + n.sv;
 end
 if n.q > 0
    B(:,(idx+1):(idx+n.q)) = A.q;
    idx = idx + n.q;
 end
 if n.l > 0
    B(:,(idx+1):(idx+n.l)) = A.l;
    idx = idx + n.l;
 end
%
% and finally the 3x3 block matrix blkmat
%
 s1 = 1; f1 = d;
 s2 = d + 1; f2 = d + m;
 s3 = f2 + 1; f3 = f2 + d;
 blkmat = sparse(f3,f3);
 blkmat(s2:f2,s1:f1) = B;
 blkmat(s1:f1,s2:f2) = B';
 blkmat(s1:f1,s3:f3) = speye(d);
% construct E and F
 E = sparse(d,d);
 F = sparse(d,d);
 idx = 0;
 if n.sm > 0
    Id = speye(n.sm);
    E((idx+1):(idx+n.sv),(idx+1):(idx+n.sv)) = skron(Z.s,Id,blk.s);
    F((idx+1):(idx+n.sv),(idx+1):(idx+n.sv)) = skron(X.s,Id,blk.s);
    idx = idx + n.sv;
 end
 if n.q > 0
    E((idx+1):(idx+n.q),(idx+1):(idx+n.q)) = arw(Z.q,blk.q);
    F((idx+1):(idx+n.q),(idx+1):(idx+n.q)) = arw(X.q,blk.q);
    idx = idx + n.q;
 end
 if n.l > 0
    E((idx+1):(idx+n.l),(idx+1):(idx+n.l)) = diag(Z.l);
    F((idx+1):(idx+n.l),(idx+1):(idx+n.l)) = diag(X.l);
 end
%
 blkmat(s3:f3,s1:f1) = E;
 blkmat(s3:f3,s3:f3) = F;
%
% cheaper to use condest than cond since it is not necessary to
% convert blkmat to a full matrix.
%
 cndsql = condest(blkmat);
%
 fprintf('\nsqlcond: comp                              = %11.3e\n',comp);
 fprintf('sqlcond: primal infeasibility              = %11.3e\n',pinfeas);
 fprintf('sqlcond: dual infeasibility                = %11.3e\n',dinfeas);
 fprintf('sqlcond: cond estimate of 3x3 block matrix = %11.3e\n',cndsql);
%
% END function
