% LSDP   driver for the function flsdp
%
%   Semi Definite Programming Solver using either the XZ or the
%   XZ + ZX method for SDP problems that compute the Lovasz theta
%   function of a graph with n vertices.
%   It is assumed that the matrix G holds a list of the (m-1)
%   edges of the graph stored as follows:
%                 G(k,1) = p, G(k,2) = q,
%   if the k-th edge of the graph joins vertices p and q.
%   There is one constraint matrix for each edge which is of the
%   form
%         A_k = e_p e_q^T + e_q e_p^T,   1 <= k <= m-1
%   and the last constraint matrix is: A_m = eye(n)
%   The matrix C is defined by C(i,j) = - sqrt(w(i)w(j)) where w
%   is a vector of weights, and the vector b, of length
%   m, is [0 ... 0 1].
%
%   Note: lsdp can call either flsdp.m or fsql.m depending upon
%   the value of the flag useXZ; when calling fsql.m, lsdp must
%   build the matrices A and C, and the vector b.
%
%   The following variables must be available in the Matlab workspace:
%      - G               list of edges in the graph
%      - w               positive vector of weights
%      - X               initial guess for primal variable
%      - y               initial guess for dual variable
%      - Z               initial guess for dual slack variable
%      - opt             structure of options
%      -    maxit        maximum iterations allowed
%      -    tau          fraction of step to boundary of psd cone
%      -    steptol      tolerance for short steps
%      -    abstol       absolute tolerance on gap
%      -    reltol       relative tolerance on gap
%      -    gapprogtol   tolerance on sufficient progress
%      -    feasprogtol  tolerance on sufficient progress
%      -    bndtol       upper bound on size of solution
%      -    prtlevel     verbosity level (accessed only by flsdp.m or fsdp.m)
%      - useXZ           flag: 1 = call fdsdp, 0 = call fsql

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/8/97

% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> either X or Z is not strictly positive definite, i.e.
%                       either chol(X) or chol(Z) failed, or Z is singular
%                       (i.e. inversion of Z failed)
%    termflag =  2  ==> XZ method: not used
%                       XZ+ZX method: Z has eigenvalues <= 0 even chol(Z)did not fail
%    termflag =  3  ==> XZ method: SchurComp is numerically indefinite or singular
%                       XZ+ZX method: SchurComp is numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X or Z is not strictly positive definite
%    termflag =  2 and iter = 0 ==> XZ method: not used
%                                   XZ+ZX method: initial Z has eigenvalues <= 0
%                                   even chol(Z)did not fail
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = opt.tau;
 scalefac_orig = scalefac;
 if useXZ,
    if opt.tau > 0.99
       fprintf('\nlsdp: tau is set to %7.5f\n',opt.tau);
       fprintf('   the XZ method does not perform well with such a high ');
       fprintf('value of tau!\n');
       choice = input('   Do you wish to proceed? <y/n> ','s');
       if choice ~= 'y'
          break;
       end;
    end;
    fprintf('\nlsdp: using XZ method...\n\n');
    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',opt.tau,scalefac);
       [X.s,y,Z.s,iter,compval,feasval,objval,termflag] = ...
           flsdp(G,w,X.s,y,Z.s,opt);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\nlsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & opt.maxit == 0,
             fprintf('\nlsdp: Data passed validation test!\n');
          elseif termflag == 7 & opt.prtlevel == 0,
             fprintf('\nlsdp: Data failed validation tests\n');
             fprintf('For more information, rerun lsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (compval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          opt.tau = min(opt.tau,.9);
          scalefac = 100*scalefac;
          linit;
       else,
          break;
       end;
    end;  % runs
 else,                % use XZ+ZX method: must construct the matrices
    n2 = n*(n+1)/2;   % A and C, and the vector b
    m = length(y);
    blk.q = [];
    blk.l = [];
    A.s = sparse(m,n2);
    for i = 1:(m-1),
       Ai = sparse(n,n);
       if G(i,1) < G(i,2),       % this should be true; it is when G
          Ai(G(i,2),G(i,1)) = 1; % is constructed by thetarnd.m
       else
          Ai(G(i,1),G(i,2)) = 1;
       end;
       A.s(i,:) = svec(Ai,n,1)';  % svec only uses lower triangle
    end;
    Ai = speye(n);
    A.s(m,:) = svec(Ai,n,1)';
    C.s = -sqrt(w*w');
    b = zeros(m,1);
    b(m) = 1;
    fprintf('\nlsdp: using XZ+ZX method...\n\n');
    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',opt.tau,scalefac);
       [X,y,Z,iter,compval,feasval,objval,termflag] = ...
           fsql(A,b,C,blk,X,y,Z,opt);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\nlsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & opt.maxit == 0,
             fprintf('\nlsdp: Data passed validation test!\n');
          elseif termflag == 7 & opt.prtlevel == 0,
             fprintf('\nlsdp: Data failed validation tests\n');
             fprintf('For more information, rerun lsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (compval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          if runs == 1,
             opt.tau = min(opt.tau,.99);
          elseif runs == 2,
             opt.tau = min(opt.tau,.9);
          end;
          scalefac = 100*scalefac;
          linit;
       else,
          break;
       end;
    end;  % runs
 end;
 opt.tau = tau_orig;
 scalefac = scalefac_orig;
%
 if iter > 0,
    if termflag == 6,
       fprintf('\nlsdp: reached the maximum number of iteration\n');
    end;
    compval = compval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    elapsedtime = etime(clock,tstart);
    cputend = cputime - cputstart;
    if runs > 1,
       fprintf('\nlsdp: number of runs = %3.0f\n',min(runs,3));
    end;
    fprintf('\nlsdp: elapsed time                = %- 9.3f seconds\n',elapsedtime);
    fprintf('lsdp: elapsed cpu time            = %- 9.3f seconds\n',cputend);
    fprintf('lsdp: number of iterations        = %- 4.0f\n',iter);
    fprintf('lsdp: final value of X . Z        = %- 11.3e\n',compval(iter+1));
    fprintf('lsdp: final primal infeasibility  = %- 11.3e\n',feasval(iter+1,1));
    fprintf('lsdp: final dual infeasibility    = %- 11.3e\n',feasval(iter+1,2));
    fprintf('lsdp: primal objective value      = %- 24.16e\n',objval(iter+1,1));
    fprintf('lsdp: dual objective value        = %- 24.16e\n',objval(iter+1,2));
%
% Lovasz theta function is the negative of what the code computes due
% to a max-min conflict in the formulation
%
    fprintf('lsdp: Lovasz theta function value = %- 24.16e\n',...
           -0.5*(objval(iter+1,1)+objval(iter+1,2)));
 end;
