% MAKESQL   creates a random SQLP problem with a solution having
%           prescribed ranks
%
%   The optimal point is saved in (Xsol,ysol,Zsol) and
%   the optimal objective value in objsol.
%
% The following variables must be available in the Matlab workspace:
%     - blk          structure of block info
%     -     blk.s    SD block structure
%     -     blk.q    QC block structure
%     -     blk.l    LP block structure
%    - m             number of constraints
%    - rx            cone boundary conditions for Xsol
%    -      rx.s     rank vector of Xsol.s
%    -      rx.q     vector: 0 => zero, 1 => bdry, 2 => interior
%    -      rx.l     scalar: number of positive components
%    - rz            cone boundary conditions for Zsol
%    -      rz.s     rank vector for Zsol.s
%    -      rz.q     vector: 0 => zero, 1 => bdry, 2 => interior
%    -      rz.l     scalar: number of positive components
%
%   Of course, complementarity must hold, so that for each block i
%   SD:   rx.s(i) + rz.s(i) <= blk.s(i)
%   QC:   rx.q(i) = 2 => rz.q(i) = 0 and rz.q(i) = 2 => rx.q(i) = 0
%         rx.q(i) = 1 => rz.q(i) < 2 and rz.q(i) = 1 => rx.q(i) < 2
%   LP:   rx.l + rz.l <= blk.l

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/17/97
%
 if ~(exist('rx') & exist('rz')),
    error('makesql: rx and/or rz not defined!');
 end;
 if ~exist('blk'),
    error('makesql: block structure blk not defined!');
 end;
 if ~exist('m'),
    error('makesql: number of constraints m not defined!');
 end;
%
% validity checks on "rank" structures r and s
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
    nblk.s = length(blk.s);
 else
    n.sm = 0;
    n.sv = 0;
    nblk.s = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
    nblk.q = length(blk.q);
 else
    n.q = 0;
    nblk.q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
    nblk.l = 1;
 else
    n.l = 0;
    nblk.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    error('makesql: the block structure is empty.');
    return
 end
 invalid = 0;
 if n.sm > 0
    invalid = invalid | ceil(rx.s)~=floor(rx.s) | ceil(rz.s)~=floor(rz.s)|...
            max(rx.s>blk.s) | max(rx.s<0) | max(rz.s>blk.s) | max(rz.s<0);
 end
 if n.q > 0
    invalid = invalid | ceil(rx.q)~=floor(rx.q) | ceil(rz.q)~=floor(rz.q)|...
            max(rx.q>blk.q) | max(rx.q<0) | max(rz.q>blk.q) | max(rz.q<0);
 end
 if n.l > 0
    if length(rx.l) ~= 1 | length(rz.l) ~= 1
       error('makesql: rx.l and rz.l must be scalars!');
    else
       invalid = invalid | ceil(rx.l)~=floor(rx.l) | ceil(rz.l)~=floor(rz.l)|...
               rx.l>blk.l | rx.l<0 | rz.l>blk.l | rz.l<0;
    end
 end
 if invalid
    error('makesql: Invalid entries in rx and rz!');
 end
%
% First check complementarity
% SD part
%
 if n.sm > 0
    if max(rx.s + rz.s > blk.s)
       error('makesql: complementarity violated for the SD part.');
    end
 end
%
% QP part
%
 if n.q > 0
    if max(((rx.q==2) .* rz.q) + ((rz.q==2) .* rx.q)) ~= 0,
       error('makesql: complementarity violated for the QC part.');
    end
 end
%
% LP part
%
 if n.l > 0
    if (rx.l + rz.l > blk.l),
       error('makesql: complementarity violated for the LP part.');
    end
 end
%
% Check strict complementarity
% SD part
%
 if n.sm > 0
    if max(rx.s + rz.s < blk.s),
       fprintf('\nmakesql: strict complementarity violated for the SD part\n');
    end
 end
%
% QP part
%
 if n.q > 0
    if max(((rx.q==0).*(rz.q~=2)) + ((rx.q==1).*(rz.q~=1)) +...
       + ((rz.q==1).*(rx.q~=1)) + ((rz.q==0).*(rx.q~=2)))
       fprintf('\nmakesql: strict complementarity violated for the QC part\n');
    end
 end
%
% LP part
%
 if n.l > 0
    if rx.l + rz.l < blk.l
       fprintf('\nmakesql: strict complementarity violated for the LP part\n');
    end
 end
%
% Construct the problem
%
 if nblk.s > 1
    Xsol.s = sparse(n.sm,n.sm);
 elseif nblk.s == 1
    Xsol.s = zeros(n.sm);
 else
    Xsol.s = [];
 end
 if n.q > 0
    Xsol.q = zeros(n.q,1);
 else
    Xsol.q = [];
 end
 if n.l > 0
    Xsol.l = zeros(blk.l,1);
 else
    Xsol.l = [];
 end
 if nblk.s > 1
    Zsol.s = sparse(n.sm,n.sm);
 else
    Zsol.s = zeros(n.sm,n.sm);
 end
 Zsol.q = zeros(size(Xsol.q));
 Zsol.l = zeros(size(Xsol.l));
%
 ysol = 2*rand(m,1) - 1;
 b = zeros(m,1);
 objsol = 0;
%
% the SD part
%
 if n.sm > 0
    T = brandsym(blk.s);
    [lam,Q] = blkeig(T,blk.s);
    fin = 0;
    for i = 1:nblk.s
       bsize = blk.s(i);
       start = fin + 1;
       fin = fin + bsize;
       U = zeros(bsize);
       V = U;
       U(1:rx.s(i),1:rx.s(i)) = diag(rand(rx.s(i),1) + 0.1);
       V(bsize-rz.s(i)+1:bsize,bsize-rz.s(i)+1:bsize) = diag(rand(rz.s(i),1) + 0.1);
       Xsol.s(start:fin,start:fin) = U;
       Zsol.s(start:fin,start:fin) = V;
    end
    Xsol.s = Q*Xsol.s*Q';
    Zsol.s = Q*Zsol.s*Q';
%
% the rows of A.s hold the vector form of the matrices Ak, 1 <= k <= m
%
    A.s = zeros(m,n.sv);
    C.s = Zsol.s;
    for i = 1:m
       randmat = brandsym(blk.s);
       C.s = C.s + ysol(i)*randmat;        % so problem is dual feasible
       A.s(i,:) = svec(randmat,blk.s)';
    end
    b = A.s*svec(Xsol.s,blk.s);            % so problem is primal feasible
    objsol = sum(sum(C.s .* Xsol.s));      % optimal objective value
 end
%
% Now, the QC part
%
 if n.q > 0
    fin = 0;
    for i = 1:nblk.q
       bsize = blk.q(i);
       start = fin + 1;
       fin = fin + bsize;
       if rx.q(i) == 0
          Xsol.q(start:fin) = 0;       % if x = 0, then x'*z = 0 is satisfied
          if rz.q(i) == 0
             Zsol.q(start:fin) = 0;
          else
             u = 2*rand(fin-start,1) - 1;
             Zsol.q(start) = norm(u);
             t = Zsol.q(start)*Zsol.q(start) - u'*u;
             if rz.q(i) == 2
                Zsol.q(start) = Zsol.q(start) + 1;
             elseif t < 0
                Zsol.q(start) = Zsol.q(start) - t;
             end
             Zsol.q(start+1:fin) = u;
          end
       elseif rz.q(i) == 0             % if z = 0, then x'*z = 0 is satisfied
          u = 2*rand(fin-start,1) - 1; % rx.q(i) is now positive
          Xsol.q(start) = norm(u);
          t = Xsol.q(start)*Xsol.q(start) - u'*u;
          if rx.q(i) == 2
             Xsol.q(start) = Xsol.q(start) + 1;
          elseif t < 0
             Xsol.q(start) = Xsol.q(start) - t;
          end
          Xsol.q(start+1:fin) = u;
       else    % by complementarity, only remaining case is rx.q(i) = 1 = rz.q(i)
          u = 2*rand(bsize,1) - 1;
          u(1) = norm(u(2:bsize));
          t = u(1)*u(1) - u(2:bsize)'*u(2:bsize);
          if t < 0
             u(1) = u(1) - t;
          end
          Xsol.q(start:fin) = u;    % so Xsol.q is on boundary of its cone
          N = null(u');    % note: first column v of N will have the property that
          v = N(:,1);      % v(1) = +/- 1/sqrt(2) and v lies on boundary of its cone
          if v(1) < 0
             v = -v;
          end
          t = v(1) - norm(v(2:bsize));
          Zsol.q(start:fin) = v;
       end
    end
    A.q = 2*rand(m,n.q) - 1;
    C.q = A.q'*ysol + Zsol.q;
    b = b + A.q*Xsol.q;
    objsol = objsol + C.q'*Xsol.q;
 end
%
% finally the LP part
%
 if n.l > 0
    A.l = 2*rand(m,blk.l) - 1;
    if rx.l > 0
       Xsol.l(1:rx.l) = rand(rx.l,1) + 1;
    end
    if rz.l > 0
       Zsol.l(n.l-rz.l+1:n.l) = rand(rz.l,1) + 1;
    end
    C.l = A.l'*ysol + Zsol.l;
    b = b + A.l*Xsol.l;
    objsol = objsol + C.l'*Xsol.l;
 end
%
 clear U V T lam Q invalid start fin bsize i randmat
