function [lam,Q,indef] = blkeig(X,blk,sortflag)
% BLKEIG Computes eigenvalues and eigenvectors of a block diagonal
%        matrix X with block structure blk.
%
% [lam,Q,indef] = blkeig(X,blk,sortflag)  (sortflag is optional)
%
%   nargout = 1 ==> compute eigenvalues lam only
%   nargout = 2 ==> compute eigenvalues lam and eigenvectors Q
%   nargout = 3 ==> compute eigenvalues, eigenvectors, and returns
%                   indef = 1 if X has a nonpositive eigenvalue,
%                   indef = 0 otherwise.
%
% Note: the eigenvalues are returned as a vector.
%
% input variables:
%     - X         a symmetric block diagonal matrix
%     - blk       block structure vector
%     - sortflag  0 => eigenvalues unsorted
%                 1 => eigenvalues sorted blockwise in ascending order
%                -1 => eigenvalues sorted blockwise in descending order
%
% output variables:
%     - lam       the vector of eigenvalues
%     - Q         the orthogonal matrix of eigenvectors
%     - indef     flag: 0 if X is positive definite, 1 otherwise

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 5/30/97
%
 if nargin < 3, sortflag = 0; end;
 nblk = length(blk);
 n = sum(blk);
 lam = zeros(n,1);
 if nblk > 1
    Q = sparse(n,n);
 else
    Q = zeros(n);
 end
 indef = 0;
 fin = 0;
%
 for i=1:nblk
    start = fin + 1;
    fin = fin + blk(i);
    if nargout >= 2   % compute eigenvectors
       Z = full(X(start:fin,start:fin));
       [Qblk,t] = eig(Z);
       t = diag(t);
       if sortflag ~= 0
          [t,I] = sort(t);
          if sortflag > 0
             lam(start:fin) = t;
             Q(start:fin,start:fin) = Qblk(:,I);
          else
             lam(fin:-1:start) = t;
             Q(start:fin,fin:-1:start) = Qblk(:,I);
          end
       else
          lam(start:fin) = t;
          Q(start:fin,start:fin) = Qblk;
       end
       if nargout == 3
          if min(t) <= 0
             indef = 1;
          end
       end
    else    % compute eigenvalues only
       if sortflag ~= 0
          t = eig(X(start:fin,start:fin));
          t = sort(t);
          if sortflag > 0
              lam(start:fin) = t;
          else
              lam(fin:-1:start) = t;
          end
       else
          lam(start:fin) = eig(X(start:fin,start:fin));
       end
    end
 end    % i loop
%
% END function
