function [strictc,v] = scomp(X,Z,blk,tol)
% SCOMP  checks whether X and Z satisfy strict complementarity (SC)
%
% [strictc,v] = scomp(X,Z,blk,tol)
%
% input variables:
%     - X            primal variable
%     -     X.s      SD primal variable
%     -     X.q      QC primal variable
%     -     X.l      LP primal variable
%     - Z            dual variable
%     -     Z.s      SD dual slack variable
%     -     Z.q      QC dual slack variable
%     -     Z.l      LP dual slack variable
%     - blk          block info structure
%     -     blk.s    block info for SD
%     -     blk.q    block info for QC
%     -     blk.l    block info for LP
%     - tol          tolerance
%
% output variables:
%     - strictc      1 if strict complementarity holds
%                    0 if strict complementarity is violated
%                   -1 if complementarity does not hold
%     - v            vector of flags (1 if SC holds, 0 if SC violated):
%     -     v.s      SD blocks
%     -     v.q      QC blocks
%     -     v.l      LP block

%
% This routine first checks the global complementarity condition
% (one of the termination conditions for the algorithm fsql.m).
% If violated, it quits immediately with strictc = -1.  Otherwise,
% it checks strict complementarity block by block, using 10*sqrt(tol)
% to determine what is "small".  Note the use of the square root, since if
% strict complementarity is violated, usually quantities which are mathematically
% zero are not reduced to very small values.
% For the semidefinite part, for each block, the sorted eigenvalues of X and Z
% are examined; SC is violated if both are zero.
% For the quadratic part, for each block, SC is said to hold if
% (i) x and z are both nonzero but on the boundary of the quadratic cone,
% or (ii) x is zero and z is not on the boundary, or (iii) z is zero and
% x is not on the boundary.  SC is violated in the other cases
% (both x and z are zero, or one is zero and the other is on the boundary).
% The output vectors v.s and v.q and scalar v.l indicate the result
% for each block, with a value of 1 if SC holds in that block, and 0 if it
% is violated.  The summary flag strictc is then set to the minimum value
% of v.s, v.q and v.l.

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/17/97
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
    nblk_s = length(blk.s);
    v.s = [];
 else
    n.sm = 0;
    n.sv = 0;
    nblk_s = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
    nblk_q = length(blk.q);
    v.q = [];
 else
    n.q = 0;
    nblk_q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
    v.l = [];
 else
    n.l = 0;
 end
%
% Note:  n.sm = 0 means that the mixed problem does not contain any SD part
%        n.q  = 0 means that the mixed problem does not contain any QC part
%        n.l  = 0 means that the mixed problem does not contain any LP part
%
 if n.sm + n.q + n.l <= 0
    fprintf('scomp: The block structure is empty.\n');
    return
 end
%
% First check that complementarity is sufficiently small
%
 comp = 0;
 if n.sm > 0
    comp = comp + sum(sum(X.s .* Z.s));
 end
 if n.q > 0
    comp = comp + X.q'*Z.q;
 end
 if n.l > 0
    comp = comp + X.l'*Z.l;
 end
 if tol <= 0
    fprintf('scomp: the tolerance must be positive: tol = %12.8e.\n',tol);
    strictc = -1;
    return
 end
 if comp >= tol
    fprintf('scomp: complementarity does not hold for (X,Z).\n');
    strictc = -1;
    return
 end
 strictc = 1;
 btol = 10*sqrt(tol);
%
% First the semidefinite part
%
 if n.sm > 0
    lam = blkeig(X.s,blk.s,-1);
    ome = blkeig(Z.s,blk.s,1);
    I = lam < btol;
    J = ome < btol;
    w = I .* J;
    if max(w) > 0    % strict complementarity does not hold
       strictc = 0;
    end
    if nargout == 2
       fin = 0;
       for j = 1:nblk_s
          start = fin + 1;
          fin = fin + blk.s(j);
          v.s(j) = ~(max(w(start:fin)));
       end
    end
 end
%
% now the QC part
%
 if n.q > 0
    [t,vx] = qcpos(X.q,blk.q);
    [t,vz] = qcpos(Z.q,blk.q);
    I2 = vx < btol;     % determine which of the squared quadratic
    J2 = vz < btol;     % norms of the blocks are zero
    stidx(1) = 1;       % index vector of the first entry of each block
    for j = 2:nblk_q
       stidx(j) = stidx(j-1) + blk.q(j-1);
    end
    I = X.q(stidx) < btol;       % determine which blocks have a
    J = Z.q(stidx) < btol;       % zero first entry
    w = (I .* J2) + (J .* I2);
    w = w > 0;
    if max(w) > 0    % strict complementarity does not hold
       strictc = 0;
    end
    if nargout == 2
       for j = 1:nblk_q
          v.q(j) = ~w(j);
       end
    end
 end
%
% finally the LP part
%
 if n.l > 0
    I = X.l < btol;
    J = Z.l < btol;
    w = I .* J;
    if max(w) > 0    % strict complementarity does not hold
       strictc = 0;
       v.l = 0;
    else
       v.l = 1;
    end
 end
%
% END function
