function [A,b,flag] = preproc(A,b,blk,tol)
% PREPROC   checks for consistency and redundancy of the system
%           [A.s A.q A.l]x = b
%
% [A,b,flag] = preproc(A,b,blk,tol)
%
% Here A is a structure where A.s is an m by n1 matrix (if not
% empty), A.q is an m by n2 matrix, and A.l is an m by n3 matrix,
% b is an m-vector, and blk is the block info structure.
% Uses QR factorization and a value tol to estimate the rank
% (e.g. tol = 1.0e-8).
% On exit,
% if flag = 0, [A.s A.q A.l] was not rank deficient and A, b are unchanged.
% if flag = 1, [A.s A.q A.l] was rank deficient, but the system was
%              consistent; rows of [A.s A.q A.l] were deleted so that the
%              remaining set of rows are linearly independent; corresponding
%              entries of b were deleted.
% if flag = 2, [A.s A.q A.l] was rank deficient and the system was also
%              inconsistent. No more processing is done:  A and b are
%              unchanged.
%
%  input variables
%     - A               structure of constraints data
%     -     A.s         constraint matrix for SD
%     -     A.q         constraint matrix for QC
%     -     A.l         constraint matrix for LP
%     - b               rhs of primal constraints
%     - blk             block info structure
%     -     blk.s       block info for SD
%     -     blk.q       block info for QC
%     -     blk.l       block info for LP (note there must be only one block)
%     - tol             tolerance to determine the rank of R in the QR
%                       factorization of [A.s A.q A.l]^T
%
%  output variables
%     - A               a possibly modified A
%     - b               a possibly modified b
%     - flag            indicates termination status

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/17/97
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
 else
    n.sm = 0;
    n.sv = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
 else
    n.q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
 else
    n.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    fprintf('preproc: the block structure is empty.  Aborting...\n');
    return
 end
 spblocks = struct('s',0,'q',0,'l',0);
 m = 0;
 flag = 0;
%
% consistency checks
%
 if n.sm > 0
    if issparse(A.s)
       spblocks.s = 1;
       A.s = full(A.s);
    end
    m = size(A.s,1);
    if n.sv ~= size(A.s,2)
       error('preproc: size of A.s is incompatible with blk.s. Aborting...')
    end
 end
 if n.q > 0
    if issparse(A.q)
       spblocks.q = 1;
       A.q = full(A.q);
    end
    m = max(m,size(A.q,1));
    if n.q ~= size(A.q,2)
       error('preproc: size of A.q is incompatible with blk.q. Aborting...')
    end
 end
 if n.l > 0
    if issparse(A.l)
       spblocks.l = 1;
       A.l = full(A.l);
    end
    m = max(m,size(A.l,1));
    if n.l ~= size(A.l,2)
       error('preproc: size of A.l is incompatible with blk.l. Aborting...')
    end
 end
 if (n.sm > 0 & m ~= size(A.s,1)) | (n.q > 0 & m ~= size(A.q,1))...
   | (n.l > 0 & m ~= size(A.l,1))
    error('preproc: A.s, A.q, A.l have different number of rows.  Aborting...\n');
 end
%
% now set up the matrix B = [A.s A.q A.l]
%
 B = zeros(m,n.sv+n.q+n.l);
 idx = 0;
 if n.sm > 0
    B(:,(idx+1):(idx+n.sv)) = A.s;
    idx = idx + n.sv;
 end
 if n.q > 0
    B(:,(idx+1):(idx+n.q)) = A.q;
    idx = idx + n.q;
 end
 if n.l > 0
    B(:,(idx+1):(idx+n.l)) = A.l;
    idx = idx + n.l;
 end
%
% now compute QR of B'
%
 [Q,R,colperm] = qr(B');  % recall: B'*colperm = Q*R
%
% and compute rank of R
%
 Rdim = min(size(R));
 mrank = 0;
 idx = 1;
 while abs(R(idx,idx)) > tol
    mrank = mrank + 1;
    idx = idx + 1;
    if idx > Rdim
       break
    end
 end
%
% checks for rank deficiency and system consistency
%
 if mrank < m
    flag = 1;   % rank deficient
    fprintf('preproc: [A.s A.q A.l] is rank deficient\n');
    fprintf('m = %3.0f  rank = %3.0f  tol = %3.1e\n',m,mrank,tol);
    R1 = R(1:mrank,1:mrank);
    bt = colperm'*b;
    b1 = bt(1:mrank);
    b2 = bt(mrank+1:m);
    v1 = R1'\b1;
    S1 = R(1:mrank,mrank+1:m);
    t = norm(S1'*v1 - b2);
    if t > 10*tol
       flag = 2;   % rank deficient and inconsistent
       fprintf('preproc: System is inconsistent: Norm of residual = %3.11e\n',t);
    else
       fprintf('preproc: System is consistent.\n');
    end
 else
    fprintf('preproc: System has maximal rank %d and is consistent\n',mrank);
 end
%
% now eliminate the redundant rows of B and entries of b (i.e. set b = b1)
% and update m
%
 if flag == 1
    m = mrank;
    Btmp = colperm'*B;
    B = Btmp(1:m,:);
    b = b1;
    idx = 0;
    if n.sm > 0
       A.s = B(:,(idx+1):(idx+n.sv));
       idx = idx + n.sv;
       if spblocks.s
          A.s = sparse(A.s);
       end
    end
    if n.q > 0
       A.q = B(:,(idx+1):(idx+n.q));
       idx = idx + n.q;
       if spblocks.q
          A.q = sparse(A.q);
       end
    end
    if n.l > 0
       A.l = B(:,(idx+1):(idx+n.l));
       idx = idx + n.l;
       if spblocks.l
          A.l = sparse(A.l);
       end
    end
 end
%
% END function
