function [X,y,Z,iter,compval,feasval,objval,termflag] = ...
      fsql(A,b,C,blk,X,y,Z,opt)
% FSQL   Mixed Conic Programming Solver
%
% [X,y,Z,iter,compval,feasval,objval,termflag] = fsql(A,b,C,blk,X,y,Z,opt)
%
%  input variables
%     - A                  structure of constraints data
%     -     A.s            constraint matrix for SD
%     -     A.q            constraint matrix for QC
%     -     A.l            constraint matrix for LP
%     - b                  rhs of primal constraints
%     - C                  cost structure
%     -     C.s            cost matrix for SD
%     -     C.q            cost vector for QC
%     -     C.l            cost vector for LP
%     - blk                block info structure
%     -     blk.s          block info for SD
%     -     blk.q          block info for QC
%     -     blk.l          block info for LP (only one block allowed)
%     - X                  initial guess for primal structure
%     -     X.s            SD primal variable
%     -     X.q            QC primal variable
%     -     X.l            LP primal variable
%     - y                  initial guess for dual variable
%     - Z                  initial guess for dual structure
%     -     Z.s            SD dual slack
%     -     Z.q            QC dual slack
%     -     Z.l            LP dual slack
%     - opt                structure of options
%     -     maxit          maximum iterations allowed
%     -     tau            fraction of step to boundary of psd cone
%     -     steptol        tolerance for short steps
%     -     abstol         absolute tolerance on gap
%     -     reltol         relative tolerance on gap
%     -     gapprogtol     tolerance on sufficient progress
%     -     feasprogtol    tolerance on sufficient progress
%     -     bndtol         upper bound on size of solution
%     -     uselyapsol     flag: 1 = use mex-file lyapsol
%                                0 = do not use mex-file lyapsol
%                               -1 = let fsql decide
%     -     prtlevel       verbosity level
%     -     validate       data validation flag
%
%  output variables
%     - X            computed primal solution
%     - y            computed dual solution
%     - Z            computed dual slack solution
%     - iter         the number of iterations performed by the algorithm
%     - compval      vector of gap values of length iter+1
%     - feasval      matrix of primal and dual feasibility residuals,
%                    of size (iter+1)x2
%     - objval       matrix of primal and dual objective values, of
%                    size (iter+1)x2
%     - termflag     termination flag

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/8/97

% termination flag (iter > 0):
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> either X or Z lies outside the product of cones
%    termflag =  2  ==> either X.s or Z.s has a nonpositive eigenvalue even
%                       though chol found the matrix to be positive definite
%    termflag =  3  ==> SchurComp numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X or Z lies outside the product
%                                   of cones
%    termflag =  2 and iter = 0 ==> either Z.s has a nonpositive eigenvalue
%                                   even though chol found the matrix to be
%                                   positive definite
%    termflag =  3 and iter = 0 ==> A may be rank deficient; try preproc
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to the bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to the bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation tests
%
 termflag = 7;
 iter = 0;
 prtlevel = opt.prtlevel;
%
% check number of input arguments
%
 if nargin ~= 8
    if prtlevel > 0
       fprintf('fsql: Incorrect number of input arguments.  Aborting...\n');
    end
    return
 end
%
% some simple data validation
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
    nblk.s = length(blk.s);
 else
    n.sm = 0;
    n.sv = 0;
    nblk.s = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
    nblk.q = length(blk.q);
 else
    n.q = 0;
    nblk.q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
    nblk.l = 1;
 else
    n.l = 0;
    nblk.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    if prtlevel > 0
       fprintf('fsql: The block structure is empty.  Aborting...\n');
    end
    return
 end
%
% Note:  n.sm = 0 means that the mixed problem does not contain any SD part
%        n.q  = 0 means that the mixed problem does not contain any QC part
%        n.l  = 0 means that the mixed problem does not contain any LP part
%
 m = 0;
 if n.sm > 0
    m = size(A.s,1);
 end
 if n.q > 0
    m = max(m,size(A.q,1));
 end
 if n.l > 0
    m = max(m,size(A.l,1));
 end
 if (n.sm > 0 & m ~= size(A.s,1)) | (n.q > 0 & m ~= size(A.q,1))...
   | (n.l > 0 & m ~= size(A.l,1))
    if prtlevel > 0
       fprintf('fsql: A.s, A.q, A.l have different number of rows.  Aborting...\n');
    end
    return
 end
%
% LP matter
% check that X.l and Z.l lie in positive orthant
% also check that X.l and Z.l are column vectors
%
 if n.l > 0
    if min(size(X.l)) ~= 1
       if prtlevel > 0
          fprintf('fsql: initial X.l must be a vector. Aborting...\n');
       end
       return
    elseif size(X.l,2) ~= 1
       X.l = X.l';
    end
    if min(size(Z.l)) ~= 1
       if prtlevel > 0
          fprintf('fsql: initial Z.l must be a vector. Aborting...\n');
       end
       return
    elseif size(Z.l,2) ~= 1
       Z.l = Z.l';
    end
    if min(X.l) < 0       % if initial X.l is outside positive orthant, we quit
       termflag = 1;
       if prtlevel > 0
          fprintf('fsql: initial X.l does not lie inside positive orthant. Aborting...\n');
       end
       return
    end
    if min(Z.l) <= 0       % if initial Z.l is outside positive orthant, we quit
       termflag = 1;
       if prtlevel > 0
          fprintf('fsql: initial Z.l does not lie inside positive orthant. Aborting...\n');
       end
       return
    end
 end
%
% QC matter:
% check that X.q and Z.q lie in Lorentz cone
% also check that X.q and Z.q are full column vectors
% (full is necessary for the mexfiles arwmul and arwimul to work correctly)
%
 if n.q > 0
    if min(size(X.q)) ~= 1
       if prtlevel > 0
          fprintf('fsql: initial X.q must be a vector. Aborting...\n');
       end
       return
    elseif size(X.q,2) ~= 1
       X.q = X.q';
    end
    if issparse(X.q)
       X.q = full(X.q);
    end
    if min(size(Z.q)) ~= 1
       if prtlevel > 0
          fprintf('fsql: initial Z.q must be a vector. Aborting...\n');
       end
       return
    elseif size(Z.q,2) ~= 1
       Z.q = Z.q';
    end
    if issparse(Z.q)
       Z.q = full(Z.q);
    end
    dist = qcpos(X.q,blk.q);
    if dist <= 0         % if initial X.q is outside Lorentz cone, we quit
       termflag = 1;
       if prtlevel > 0
          fprintf('fsql: initial X.q does not lie inside Lorentz cone. Aborting...\n');
       end
       return
    end
    dist = qcpos(Z.q,blk.q);
    if dist <= 0          % if initial Z.q is outside Lorentz cone, we quit
       termflag = 1;
       if prtlevel > 0
          fprintf('fsql: initial Z.q does not lie inside Lorentz cone. Aborting...\n');
       end
       return
    end
%
% construct the block matrix A.q*A.q' to be used in qcschur;
% AATq is initialized as a sparse matrix if A.q is sparse, and as full matrix
% otherwise; after AATq is computed the number of nonzero entries is computed;
% if this number is greater than 50% of the total number of entries in a matrix
% of this size, AATq is converted to full.
%
   if issparse(A.q)
      AATq = sparse(m,m*nblk.q);
   else
      AATq = zeros(m,m*nblk.q);
   end
   fin = 0;
   fin2 = 0;
   for i = 1:nblk.q,
      start = fin + 1;
      start2 = fin2 + 1;
      fin = fin + blk.q(i);
      fin2 = fin2 + m;
      Ai = A.q(:,start:fin);
      AATq(:,start2:fin2) = Ai*Ai';
   end
   if nnz(AATq) > .5*m*m*nblk.q
      AATq = full(AATq);
   end
 end
%
% SD matter:
% determine whether to use the mex file lyapsol
%
 if n.sm > 0
    if opt.uselyapsol == 0 | exist('lyapsol') ~= 3
       nolyapsol = 1;
    elseif opt.uselyapsol == 1
       nolyapsol = 0;
    else
       avge_bsize = sum(blk.s)/nblk.s;
       if avge_bsize >= 10 | (avge_bsize >= 5 & max(blk.s) >= 20)
          nolyapsol = 0;
       else
          nolyapsol = 1;
       end
    end
%
% check that C.s, X.s, and Z.s are sparse if the semidefinite
% component has more than one block
%
    if nblk.s > 1 & (~issparse(C.s) | ~issparse(X.s) | ~issparse(Z.s))
       if prtlevel > 0
          fprintf('fsql: multiple blocks, SD matrices must be sparse. Aborting...\n');
       end
       return
    end
%
% now check that X.s and Z.s are block diagonal if validate flag is set
%
    if opt.validate
       if nblk.s > 1
          if size(blk.s,1) ~= nblk.s   % make sure blk is a column vector
             blk.s = blk.s';
          end
          getblk = zeros(n.sm,1);
          cblk(1) = blk.s(1);               % entries of cblk are the cumulative
          for i = 2:nblk.s                  % sums of the block sizes
             cblk(i) = cblk(i-1) + blk.s(i);
          end
          for i = 1:n               % for a given row/column index i, find
             t = find(i <= cblk);   % the index of the block it belongs to
             getblk(i) = t(1);
          end
          [I,J] = find(X.s);
          idx1 = getblk(I);
          idx2 = getblk(J);
          loc = max(idx1 ~= idx2);
          if loc > 0
             if prtlevel > 0
                fprintf('fsql: X.s has invalid block structure. Aborting...\n');
             end
             return
          end
          [I,J] = find(Z.s);
          idx1 = getblk(I);
          idx2 = getblk(J);
          loc = max(idx1 ~= idx2);
          if loc > 0
             if prtlevel > 0
                fprintf('fsql: Z.s has invalid block structure. Aborting...\n');
             end
             return
          end
       end
    end     % opt.validate
%
% Compute chol(X.s), chol(Z.s), eigenvectors and eigenvalues of Z.s and
% Evsum so they are available on entry to the loop and check that X.s and
% Z.s lie in the positive semidefinite cone
%
    [chol_Xs,indef] = chol(X.s);
    if indef > 0
       if prtlevel > 0
          fprintf('fsql: initial X.s is not positive definite. Aborting...\n');
       end
       termflag = 1;
       return
    end
%
    [chol_Zs,indef] = chol(Z.s);
    if indef > 0
       if prtlevel > 0
          fprintf('fsql: initial Z.s is not positive definite. Aborting...\n');
       end
       termflag = 1;
       return
    end
    [Zeigval,Zeigvec,indef] = blkeig(Z.s,blk.s);  % note: Zeigval is a vector
    if indef == 1             % if initial Z is indefinite, we quit
       termflag = 2;
       if prtlevel > 0
          fprintf('fsql: initial Z.s has nonpositive eigenvalues. Aborting...\n');
       end
       return
    end
%
% Compute Evsum from the eigenvalues of Z:
%        Evsum(i,j) = Zeigval(i) + Zeigval(j);
% Evsum is needed to solve the Lyapunov system
%
    if nblk.s > 1
       Evsum = sparse(n.sm,n.sm);
    else
       Evsum = zeros(n.sm);
    end
    fin = 0;
    for i = 1:nblk.s
       bsize = blk.s(i);
       start = fin + 1;
       fin = fin + bsize;
       tmp = Zeigval(start:fin);
       tmp = tmp(:,ones(1,bsize));
       Evsum(start:fin,start:fin) = tmp + tmp';
    end
 end   % end of all validation
%
% Now some initialization
%
 maxit = opt.maxit;
 tau = opt.tau;
 steptol = opt.steptol;
 abstol = opt.abstol;
 reltol = opt.reltol;
 gapprogtol = opt.gapprogtol;
 feasprogtol = opt.feasprogtol;
 bndtol = opt.bndtol;
%
 if n.sm > 0
    vCs = svec(C.s,blk.s);
    vXs = svec(X.s,blk.s);
    vZs = svec(Z.s,blk.s);
    Id2 = 2*speye(n.sm);
    gapnew.s = full(vXs'*vZs);
    rps = A.s*vXs;
    Rd.s = vCs - vZs - A.s'*y;     % a vector
    dual_infeas = norm(Rd.s);
    Rd.s = smat(Rd.s,blk.s);
    pobj.s = full(vCs'*vXs);
    Xnew.s = 0;
    Znew.s = 0;
 else
    vXs = 0;
    vZs = 0;
    gapnew.s = 0;
    rps = 0;
    Rd.s = 0;
    dual_infeas = 0;
    Rc.s = 0;
    pobj.s = 0;
    Zeigvec = 0;
    Zeigval = 0;
    Evsum = 0;
    normXs = 0;
    normZs = 0;
 end
 if n.q > 0
    Idq = zeros(n.q,1);
    start = 1;
    for i = 1:nblk.q
       Idq(start) = 1;
       start = start + blk.q(i);
    end
    gapnew.q = X.q'*Z.q;
    rpq = A.q*X.q;
    Rd.q = C.q - Z.q - A.q'*y;
    pobj.q = C.q'*X.q;
    Xnew.q = 0;
    Znew.q = 0;
 else
    gapnew.q = 0;
    rpq = 0;
    Rd.q = 0;
    Rc.q = 0;
    pobj.q = 0;
    normXq = 0;
    normZq = 0;
 end
 if n.l > 0
    Idl = ones(n.l,1);
    gapnew.l = X.l'*Z.l;
    rpl = A.l*X.l;
    Rd.l = C.l - Z.l - A.l'*y;
    pobj.l = C.l'*X.l;
    Xnew.l = 0;
    Znew.l = 0;
 else
    gapnew.l = 0;
    rpl = 0;
    Rd.l = 0;
    Rc.l = 0;
    pobj.l = 0;
    normXl = 0;
    normZl = 0;
 end
 comp = gapnew.s + gapnew.q + gapnew.l;
 rp = b - rps - rpq - rpl;
 pri_infeas = norm(rp);
 dual_infeas = dual_infeas + norm(Rd.q) + norm(Rd.l);
 compval(1) = comp;
 objval(1,1) = pobj.s + pobj.q + pobj.l;
 objval(1,2) = b'*y;
 feasval(1,1) = pri_infeas;
 feasval(1,2) = dual_infeas;
 Xpos = struct('s',1,'q',1,'l',1);
 Zpos = struct('s',1,'q',1,'l',1);
 Rdnew = struct('s',0,'q',0,'l',0);
 termflag = 6;
%
% print header and initial info
%
 if prtlevel > 0
    fprintf('iter   p_step      d_step     p_infeas    d_infeas      X . Z');
    fprintf('       pobj        dobj\n');
    fprintf('%3.0f %11.3e %11.3e', 0, 0.0, 0.0);
    fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, comp);
    fprintf(' %11.3e %11.3e\n',objval(1,1),objval(1,2));
 end
%
% start main loop
%
 for iter = 1:maxit
%
% construct Schur complement
%
    if n.l > 0
       Xnew.l = X.l ./ Z.l;         % this is marginally faster than
       SchurComp = zeros(m,n.l);    %   A.l*diag(X.l ./ Z.l)*A.l'
       for i = 1:n.l
          SchurComp(:,i) = Xnew.l(i)*A.l(:,i);
       end
       SchurComp = SchurComp*A.l';
    else
       SchurComp = zeros(m);
    end
    if n.q > 0
       SchurComp = SchurComp + qcschur(A.q,AATq,X.q,Z.q,blk.q);
    end
    if n.sm > 0
       SchurComp = SchurComp + sdschur(A.s,X.s,Zeigvec,Evsum,blk.s,nolyapsol);
    end
%
% compute its LU factorization
%
    [Lfactor,Ufactor,Perm] = lu(SchurComp);
%
% check if SchurComp is numerically singular
%
    minpiv = min(abs(diag(Ufactor)));
    if minpiv == 0
       if prtlevel > 0
          fprintf('fsql: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically singular)\n');
          fprintf('      (A may be rank deficient.  Try running preproc)\n');
       end
       termflag = 3;
       iter = iter-1;
       return
    end
%
% predictor step
%
    if n.sm > 0
       XZ = X.s*Z.s;
       Rc.s = -(XZ + XZ');
    end
    if n.q > 0
       Rc.q = -arwmul(X.q,Z.q,blk.q);
    end
    if n.l > 0
       Rc.l = -(X.l .* Z.l);
    end
%
    [dX,dy,dZ] =...
       sqlsol(A,X,Z,Zeigvec,Evsum,Rd,rp,Rc,Lfactor,Ufactor,Perm,blk,n);
%
% symmetrize dX.s and dZ.s : good idea because of rounding
%
    if n.sm > 0
       dX.s = 0.5*(dX.s+dX.s');
       dZ.s = 0.5*(dZ.s+dZ.s');
    end
%
% compute new gap
%
    Xstep = 1;
    Zstep = 1;
    if n.sm > 0
       Xstep = min(Xstep,tau*sdbound(chol_Xs,dX.s,blk.s));  % tau times
       Zstep = min(Zstep,tau*sdbound(chol_Zs,dZ.s,blk.s));  % step to boundary
    end
    if n.q > 0
       Xstep = min(Xstep,tau*qcbound(X.q,dX.q,blk.q));
       Zstep = min(Zstep,tau*qcbound(Z.q,dZ.q,blk.q));
    end
    if n.l > 0
       Xstep = min(Xstep,tau*lpbound(X.l,dX.l));
       Zstep = min(Zstep,tau*lpbound(Z.l,dZ.l));
    end
    if n.sm > 0
       Xnew.s = X.s + Xstep*dX.s;
       Znew.s = Z.s + Zstep*dZ.s;
       gapnew.s = full(sum(sum(Xnew.s .* Znew.s)));
    end
    if n.q > 0
       Xnew.q = X.q + Xstep*dX.q;
       Znew.q = Z.q + Zstep*dZ.q;
       gapnew.q = Xnew.q'*Znew.q;
    end
    if n.l > 0
       Xnew.l = X.l + Xstep*dX.l;
       Znew.l = Z.l + Zstep*dZ.l;
       gapnew.l = Xnew.l'*Znew.l;
    end
    compnew = gapnew.s + gapnew.q + gapnew.l;
%
% corrector step
%
    mu = (compnew/comp)^2 * compnew/(n.sm + n.q + n.l);  % Mehrotra's formula
    if n.sm > 0
       dXdZ = dX.s*dZ.s;
       Rc.s = Rc.s + mu*Id2 - (dXdZ + dXdZ');  % included mu term this time
    end
    if n.q > 0
       dXdZ = arwmul(dX.q,dZ.q,blk.q);
       Rc.q = Rc.q + mu*Idq - dXdZ;            % included mu term this time
    end
    if n.l > 0
       Rc.l = Rc.l + mu*ones(n.l,1) - dX.l .* dZ.l;
    end
%
% Compute the corrector step
%
    [dX,dy,dZ] =...
       sqlsol(A,X,Z,Zeigvec,Evsum,Rd,rp,Rc,Lfactor,Ufactor,Perm,blk,n);
%
% symmetrize dX and dZ
%
    if n.sm > 0
       dX.s = 0.5*(dX.s+dX.s');
       dZ.s = 0.5*(dZ.s+dZ.s');
    end
%
% and compute the stepsizes
%
    Xstep = 1;
    Zstep = 1;
    if n.sm > 0
       Xstep = min(Xstep,tau*sdbound(chol_Xs,dX.s,blk.s));  % tau times
       Zstep = min(Zstep,tau*sdbound(chol_Zs,dZ.s,blk.s));  % step to boundary
    end
    if n.q > 0
       Xstep = min(Xstep,tau*qcbound(X.q,dX.q,blk.q));
       Zstep = min(Zstep,tau*qcbound(Z.q,dZ.q,blk.q));
    end
    if n.l > 0
       Xstep = min(Xstep,tau*lpbound(X.l,dX.l));
       Zstep = min(Zstep,tau*lpbound(Z.l,dZ.l));
    end
%
% end of corrector step and iteration;
%
% decide whether to accept new point or quit
% failure if steps get too short
%
    if Xstep < steptol | Zstep < steptol
       if prtlevel > 0
          fprintf('fsql: stop since steps are too short\n');
       end
       termflag = 5;
       iter = iter-1;
       return
    end
%
    ynew = y + Zstep*dy;
%
% now update X.l and Z.l
%
    if n.l > 0
       Xnew.l = X.l + Xstep*dX.l;
       Znew.l = Z.l + Zstep*dZ.l;
%
% and check that Xnew.l and Znew.l lie in positive orthant
%
       if min(Xnew.l) <= 0    % if Xnew.l is not inside positive orthant, we quit
          termflag = 1;
          Xpos.l = 0;
       end
       if min(Znew.l) <= 0    % if Znew.l is outside positive orthant, we quit
          termflag = 1;       % and return the previous iterate
          Zpos.l = 0;
       end
       gapnew.l = Xnew.l'*Znew.l;
       rpl = A.l*Xnew.l;
       Rdnew.l = C.l - Znew.l - A.l'*ynew;
       pobj.l = C.l'*Xnew.l;
    end
%
% next update X.q and Z.q
%
    if n.q > 0
       Xnew.q = X.q + Xstep*dX.q;
       Znew.q = Z.q + Zstep*dZ.q;
%
% check that Xnew.q and Znew.q lie in Lorentz cone
%
       [dist,qXres] = qcpos(Xnew.q,blk.q);
       if dist <= 0            % if Xnew.q is not inside Lorentz cone, we quit
          termflag = 1;        % and return the previous iterate
          Xpos.q = 0;
       end
       [dist,qZres] = qcpos(Znew.q,blk.q);
       if dist <= 0          % if initial Z.q is outside Lorentz cone, we quit
          termflag = 1;
          Zpos.q = 0;
       end
       gapnew.q = Xnew.q'*Znew.q;
       rpq = A.q*Xnew.q;
       Rdnew.q = C.q - Znew.q - A.q'*ynew;
       pobj.q = C.q'*Xnew.q;
    end
%
% and finally update X.s and Z.s
%
    if n.sm > 0
       Xnew.s = X.s + Xstep*dX.s;
       Znew.s = Z.s + Zstep*dZ.s;
%
% check for positive definiteness of Xnew.s
%
       [chol_Xs,indef] = chol(Xnew.s);  % if Xnew is indefinite, we quit
       if indef > 0
          termflag = 1;
          Xpos.s = 0;
       end
%
% check for positive definiteness of Znew.s
%
       [chol_Zs,indef] = chol(Znew.s);  % if Znew is indefinite, we quit
       if indef > 0
          termflag = 1;
          Zpos.s = 0;
       end
%
% find evecs and evals of Znew.s and Evsum (needed to solve
% the Lyapunov system) and test for nonpositive eigenvalues
%
       [Zeigval,Zeigvec,indef] = blkeig(Znew.s,blk.s);
       if indef == 1 & Zpos.s    % if blkeig says Znew.s is not positive definite
          Zpos.s = 0;            % (though chol said it was), we quit
          termflag = 2;          % and return the current iterate
       end
       if termflag ~= 1 & termflag ~= 2    % otherwise we are going to quit
                                           % so we don't need Evsum
          fin = 0;
          for i = 1:nblk.s
             bsize = blk.s(i);
             start = fin + 1;
             fin = fin + bsize;
             tmp = Zeigval(start:fin);
             tmp = tmp(:,ones(1,bsize));
             Evsum(start:fin,start:fin) = tmp + tmp';
          end
       end
       vXs = svec(Xnew.s,blk.s);
       vZs = svec(Znew.s,blk.s);
       gapnew.s = full(vXs'*vZs);
       rps = A.s*vXs;
       Rdnew.s = vCs - vZs - A.s'*ynew;
       dual_infeas_new = norm(Rdnew.s);
       Rdnew.s = smat(Rdnew.s,blk.s);
       pobj.s = full(vCs'*vXs);
    else
       dual_infeas_new = 0;
    end
%
% decide whether new point provides "substantial"
% improvement over previous iterate
%
    compnew = gapnew.s + gapnew.q + gapnew.l;
    rpnew = b - rps - rpq - rpl;
    pri_infeas_new = norm(rpnew);
    dual_infeas_new = dual_infeas_new + norm(Rdnew.q) + norm(Rdnew.l);
%
% if gap is small, and feasibility is getting worse, then
% reject new point unless gap is greatly improved
%
    if compnew < 100*abstol & gapprogtol*compnew > comp & ...
       (pri_infeas_new > feasprogtol*pri_infeas | ...
        dual_infeas_new > feasprogtol*dual_infeas)
%
% reject the new iterate
%
       termflag = 4;
       if prtlevel > 0
          fprintf('fsql: stop since new point is substantially worse than current iterate\n');
          fprintf('      X . Z = %11.3e\n',compnew);
          fprintf('      pri_infeas = %11.3e\n', pri_infeas_new);
          fprintf('      dual_infeas = %11.3e\n', dual_infeas_new);
       end
       iter = iter-1;
       return
    else
%
% accept new point
%
       X = Xnew;
       y = ynew;
       Z = Znew;
       comp = compnew;
       rp = rpnew;
       Rd = Rdnew;
       pri_infeas = pri_infeas_new;
       dual_infeas = dual_infeas_new;
%
% compute objective values so they can be plotted later
%
       compval(iter+1) = comp;
       objval(iter+1,1) = pobj.s + pobj.q + pobj.l;
       objval(iter+1,2) = b'*y;
       feasval(iter+1,1) = pri_infeas;
       feasval(iter+1,2) = dual_infeas;
%
% and display information
%
       if prtlevel > 0
          fprintf('%3.0f %11.3e %11.3e',iter,Xstep,Zstep);
          fprintf(' %11.3e %11.3e %11.3e',pri_infeas,dual_infeas,comp);
          fprintf(' %11.3e %11.3e\n',objval(iter+1,1),objval(iter+1,2));
       end
    end
%
% termination test
%
    if termflag == 1 | termflag == 2   % X or Z does not lie inside its cone, so quit
       if prtlevel > 0
          fprintf('fsql: stop since limiting accuracy reached\n');
          if ~Xpos.s
             minXeigval = min(blkeig(X.s,blk.s));
             fprintf(' (smallest eigenvalue of X.s = %11.3e)\n', minXeigval);
          end
          if ~Zpos.s
             fprintf(' (smallest eigenvalue of Z.s = %11.3e)\n', min(Zeigval));
          end
          if ~Xpos.q
             fprintf(' (X.q outside cone: violation = %11.3e)\n', min(qXres));
          end
          if ~Zpos.q
             fprintf(' (Z.q outside cone: violation = %11.3e)\n', min(qZres));
          end
          if ~Xpos.l
             fprintf(' (smallest entry of X.l = %11.3e)\n', min(X.l));
          end
          if ~Zpos.l
             fprintf(' (smallest entry of Z.l = %11.3e)\n', min(Z.l));
          end
       end
       return
    end
    total_err = comp + pri_infeas + dual_infeas;
    if n.sm > 0
       normXs = norm(vXs);
       normZs = norm(vZs);
    end
    if n.q > 0
       normXq = norm(X.q);
       normZq = norm(Z.q);
    end
    if n.l > 0
       normXl = norm(X.l);
       normZl = norm(Z.l);
    end
    normX = normXs + normXq + normXl;
    normZ = normZs + normZq + normZl;
    if total_err < min(abstol,reltol*(normX + normZ))
       if prtlevel > 0
          fprintf('fsql: stop since error reduced to desired value\n');
       end
       termflag = 0;
       return
    elseif normXs > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of X.s exceeds bndtol\n');
       end
       termflag = -2;
       return
    elseif normXq > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of X.q exceeds bndtol\n');
       end
       termflag = -2;
       return
    elseif normXl > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of X.l exceeds bndtol\n');
       end
       termflag = -2;
       return
    elseif normZs > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of Z.s exceeds bndtol\n');
       end
       termflag = -1;
       return
    elseif normZq > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of Z.q exceeds bndtol\n');
       end
       termflag = -1;
       return
    elseif normZl > bndtol
       if prtlevel > 0
          fprintf('fsql: stop since norm of Z.l exceeds bndtol\n');
       end
       termflag = -1;
       return
    end
 end
 if maxit == 0, iter = 0; end
%
% END function
