function [cnddual,B] = dualcond(A,blk,Z,eigtol)
% DUALCOND  estimates whether an SDP is dual degenerate (with respect to
%           its block structure) at its computed solution.
%           It constructs the matrix B whose rows are vector
%           representations of the matrices Q1' Ak Q1, k=1,...,m,
%           where the columns of Q1 form an orthonormal basis for the null
%           space of Z (corresponding to eigenvalues smaller than eigtol).
%           B has m rows and q columns, where q depends on the block structure
%           and the ranks of each block.  The SDP is said to be dual
%           nondegenerate if its rows span R^q, i.e. its columns are linearly
%           independent.
%           If m < q, the SDP is dual degenerate by definition, and cnddual
%           is set to infinity.  If m >= q, cnddual is set to the 2-norm
%           condition number of B.  A very large condition number
%           is a strong indication that the SDP is dual degenerate.
%
% [cnddual,B] = dualcond(A,blk,Z,eigtol)
%
% input variables:
%     - A         matrix of primal constraints
%     - blk       block structure vector
%     - Z         the dual slack
%     - eigtol    a tolerance to determine rank of Z
%
% output variables:
%     - cnddual   the condition number of the dual problem
%     - B         matrix whose condition number is cnddual (see AHO)

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 nblk = length(blk);
 [m,n2] = size(A);
%
 [lam,Q] = blkeig(Z,blk);
 I = lam < eigtol;
 if sum(I) == 0,                       % degeneracy condition not defined
    fprintf('dualcond: Z has no eigenvalues less than tolerance: B is empty\n');
    cnddual = Inf;
    fprintf('dualcond = %11.3e\n',cnddual);
    return;
 end;
%
 start = 1;                            % need to know block structure of Q1
 fin = 0;                              % for call to svec later
 next = 0;
 for i = 1:nblk
    bsize = blk(i);
    fin = fin + bsize;
    tmp = sum(I(start:fin));
    if tmp > 0,                        % block i of Q has contribution to Q1
       next = next + 1;
       Q1blk(next) = tmp;
    end;
    start = start + bsize;
 end;
%
 Q1 = Q(:,find(I));                    % e-vectors for zero eigenvalues of Z
 ncols = sum(Q1blk .* (1+Q1blk))/2;
 B = zeros(m,ncols);
 for k=1:m
    Ak = smat(A(k,:),blk);
    B(k,:) = svec(Q1'*Ak*Q1,Q1blk)';   % Q1blk defines block structure of Q1
 end;
%
 if m < size(B,2),
    cnddual = Inf;
 else
    cnddual = cond(B);
 end;
 fprintf('dualcond = %11.3e\n',cnddual);
%
% END function
