/* svec.c - mex file, v. 4.2:    replaces svec.m
 *
 * synopsis:   v = svec(A,blk,sp_blk)
 *
 * inputs:
 *    A                 an nxn block diagonal matrix
 *    blk               the block diagonal structure
 *    sp_blk (optional) nonzero if blocks are sparse, zero otherwise
 *
 * output:
 *    v                 a vector of length sum_i blk[i]*(blk[i]+1)/2
 *
 * Notes:
 *    1. if # blocks > 1, or the blocks are declared sparse via sp_blk,
 *       then A must be sparse; if not svec terminates with an error.
 *    2. if sp_blk is omitted, svec assumes that the blocks are full
 *       unless there is only one block and the matrix A is sparse (in
 *       this case svec can determine that the block is sparse; if there
 *       are more than one block it can not, since A is sparse irrelevantly
 *       of whether the blocks are).
 *    3. svec returns a sparse vector v if it detrmined that the blocks are
 *       sparse and a full vector otherwise.
 *
 * Copyright (c) 1997 by
 * F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
 * Last modified : 3/2/97
 */
#include <math.h>
#include "mex.h"

/* Input Arguments */
#define  A_IN       prhs[0]
#define  blk_IN     prhs[1]
#define  sp_blk_IN  prhs[2]

/* Output Arguments */
#define  v_OUT  plhs[0]

#if !defined(max)
#define  max(A, B)   ((A) > (B) ? (A) : (B))
#endif

#if !defined(min)
#define  min(A, B)   ((A) < (B) ? (A) : (B))
#endif

#ifdef __STDC__
static void svec(
   double  *v,
   int  *vir,
   int  *vjc,
   double  *pr,
   int  *ir,
   int  *jc,
   int  n,
   double  *blk,
   int  nblk,
   int  eltsize
)
#else
svec(v,vir,vjc,pr,ir,jc,n,blk,nblk,eltsize)
   double  *v;
   int  *vir;
   int  *vjc;
   double  *pr;
   int  *ir;
   int  *jc;
   int  n;
   double  *blk;
   int  nblk;
   int  eltsize;
#endif
{
   int i,j,start,fin,bsize,btmp,blkidx,idx,baseidx,colidx,vrowidx,viridx;
   double *vidx;
   static double r2;

   r2 = sqrt(2.0);

   if(vir != '\0') {  /* sparse blocks so v is sparse */
      vjc[0] = 0;
      baseidx = 0;
      colidx = 0;
      viridx = 0;
      fin = 0;
      for(blkidx = 0; blkidx < nblk; blkidx++) {
         bsize = blk[blkidx];
         btmp = bsize;
         for(j = colidx; j < colidx+bsize; j++) {
            start = fin;     /* indices into ip, pr for nonzero entries in */
            fin = jc[j+1];   /* j-th column of A                           */
            for(idx = start; idx < fin; idx++) {
               i = ir[idx];
               if(i >= j) {   /* lower triangle entries */
                  vrowidx = baseidx + i - j;
                  vir[viridx] = vrowidx;
                  if(i > j)
                     v[viridx] = r2*pr[idx];
                  else
                     v[viridx] = pr[idx];
                  ++viridx;
               }
            }
            baseidx += btmp;
            --btmp;
         }
         colidx += bsize;
      }
      vjc[1] = viridx;
   }
   else {             /* full blocks : v is full */
      if(nblk > 1) {  /* more than one block so A is sparse */
         baseidx = 0;
         colidx = 0;
         fin = 0;
         for(blkidx = 0; blkidx < nblk; blkidx++) {
            bsize = blk[blkidx];
            btmp = bsize;
            for(j = colidx; j < colidx+bsize; j++) {
               start = fin;         /* indices into ip, pr for nonzero */
               fin = jc[j+1];       /* entries in j-th column of A     */
               for(idx = start; idx < fin; idx++) {
                  i = ir[idx];
                  if(i >= j) {   /* lower triangle entries */
                     vrowidx = baseidx + i - j;
                     if(i > j)
                        v[vrowidx] = r2*pr[idx];
                     else
                        v[vrowidx] = pr[idx];
                  }
               }
               baseidx += btmp;
               --btmp;
            }
            colidx += bsize;
         }
      }
      else {          /* one full block only */
         vidx = v;
         start = 0;
         for(i = n; i > 0; i--) {
            memcpy(vidx,pr+start,i*eltsize);
            for(j = 1; j < i; j++)
               vidx[j] *= r2;
            start += n+1;
            vidx += i;
         }
      }
   }
   return;
}

#ifdef __STDC__
void mexFunction(
   int nlhs, Matrix *plhs[],
   int nrhs, Matrix *prhs[]
)
#else
mexFunction(nlhs,plhs,nrhs,prhs)
   int nlhs;
   Matrix *plhs[];
   int nrhs;
   Matrix *prhs[];
#endif
{
   double *pr,*v,*blk,sp_blk;
   int i,m,n,n2,nblk,eltsize,nnz;
   int *ir,*jc,*vir,*vjc;
   int is_sparse;

/* Check for proper number of arguments */
   if ((nrhs > 3) || (nrhs < 2)) {
      mexErrMsgTxt("svec requires two or three input arguments.");
   } else if (nlhs > 1) {
      mexErrMsgTxt("svec requires one output argument.");
   }

   m = mxGetM(A_IN);
   n = mxGetN(A_IN);
   if (min(m,n) == 0)
      mexErrMsgTxt("svec: A is empty.");
   if (m != n)
      mexErrMsgTxt("svec: A must be square.");
   m = mxGetM(blk_IN);
   nblk = mxGetN(blk_IN);
   if (min(m,nblk) == 0)
      mexErrMsgTxt("svec: blk is empty.");
   nblk = max(m,nblk);
   if (nblk > n)
      mexErrMsgTxt("svec: too many blocks.");
   eltsize = sizeof(double);
   if(nrhs == 3)
      sp_blk = mxGetScalar(sp_blk_IN);
   else
      sp_blk = 0;

/* consistency check */
   is_sparse = mxIsSparse(A_IN);
   if(((nblk > 1) || sp_blk) && !is_sparse)
      mexErrMsgTxt("svec: A should be sparse");
   if((nblk == 1) && (sp_blk == 0) && is_sparse)
      sp_blk = 1;

/* Assign pointers to the various input parameters */
   pr = mxGetPr(A_IN);
   blk = mxGetPr(blk_IN);
   n2 = 0;
   for(i = 0; i < nblk; i++) {
      m = blk[i];
      if (m == 0)
         mexErrMsgTxt("svec: encountered an empty block.");
      n2 += m*(m+1);
   }
   n2 /= 2;
   if(is_sparse) {
      ir = mxGetIr(A_IN);
      jc = mxGetJc(A_IN);
   } else {
      ir = '\0';
      jc = '\0';
   }

/* Create a matrix for the return argument */
   if (sp_blk) {                   /* jc[n] = # nonzero entries in A */
      nnz = (jc[n] - n)/2 + n + 1; /* estimate # nonzero entries in v */
      v_OUT = mxCreateSparse(n2,1,nnz,REAL);
      vir = mxGetIr(v_OUT);
      vjc = mxGetJc(v_OUT);
   }
   else {
      v_OUT = mxCreateFull(n2,1,REAL);
      vir = '\0';
      vjc = '\0';
   }

/* Assign pointers to the output parameter */
   v = mxGetPr(v_OUT);

/* Do the actual computations in a subroutine */
   svec(v,vir,vjc,pr,ir,jc,n,blk,nblk,eltsize);
   return;
}
