function [K, f, loc] = hifoomain(pars, order, init, obj, options)
% main routine for HIFOO: H-infinity fixed-order optimization
% intended to be called by hifoo only
% see comments in hifoo.m
% pars and init use the structure format (other formats already converted)
m = size(pars.B2,2); % number of cols of B2 (number of rows of Chat) 
p = size(pars.C2,1); % number of rows of C2 (number of cols of Bhat)
pars.nhat = order;
% function that computes h-infinity norm, complex stability radius,
% pseudospectral abscissa or spectral abscissa, depending on pars.obj
pars.fgname = 'hifooobj';
nvar = (order + m)*(order + p); % number of optimization variables
pars.nvar = nvar;
% hifoo options (no call to setdefaults)
if ~isfield(options, 'penalty') 
    options.penalty = 0;
elseif isempty(options.penalty)
    options.penalty = 0;
elseif ~(isposreal(options.penalty) | options.penalty == 0)
    error('hifoo: input "options.penalty" must be a nonnegative real scalar')
end
penalty = options.penalty; % penalty is copied to pars.penalty below
if ~isfield(options, 'barrier') 
    options.barrier = 0;
elseif isempty(options.barrier)
    options.barrier = 0;
elseif ~(isposreal(options.barrier) | options.barrier == 0)
    error('hifoo: input "options.barrier" must be a nonnegative real scalar')
end
barrier = options.barrier; % barrier is copied to pars.barrier below
if ~isfield(options, 'cpumax') % quit when cpu time in seconds exceeds this
    options.cpumax = inf;
elseif ~isposreal(options.cpumax)
    error('hifoo: input "options.cpumax" must be a positive scalar')
end
cpufinish = cputime + options.cpumax;
if ~isfield(options, 'normtol')
    options.normtol = 1e-3; % larger default than HANSO default
elseif ~isposreal(options.normtol)
    error('hifoo: input "options.normtol" must be a positive scalar')
end 
if ~isfield(options, 'evaldist') 
    options.evaldist = 1e-3; 
elseif ~isposreal(options.evaldist)
    error('hifoo: input "options.evaldist" must be a positive scalar')
end
if ~isfield(options, 'nrand') % number of starting points besides init
    options.nrand = 3;
elseif ~isposint(options.nrand)
    % nrand is 0 is not allowed: if init came from output of a hifoo run,
    % perhaps a different order, objective is likely not to be smooth
    % there and BFGS may fail immediately: some perturbation is needed
    error('hifoo: input "options.nrand" must be a positive integer')
end
nrand = options.nrand;
prtlevel = options.prtlevel; % set in hifoo
if prtlevel > 0 
    fprintf('hifoo: using options\n')
    disp(options)
end
if prtlevel > 0
    fprintf('hifoo: number of design variables is %d\n', nvar)
end
options.prtlevel = max(0, prtlevel-1); % to reduce output from HANSO
% pseudospectral abscissa code is part of HIFOO, copy epsilon info from options to pars
if obj == 'p'
    if isfield(options,'epsilon') % defines epsilon-pseudospectral abscissa
        if ~isposreal(options.epsilon)
            error('hifoo: input "options.epsilon" must be a positive real scalar')
        else
            pars.epsilon = options.epsilon; % for passing to hifooobj via PARS, not options
        end
    else
        pars.epsilon = .01; % default value, for passing to hifooobj via PARS, not options
    end
    if exist('pspa') ~= 2
        error('hifoo: "pspa" must be installed from www.cs.nyu.edu/overton/software/pspa/ when input "obj" is "p"')
    end
end
% check out what software is available to compute H-infinity norm
if obj == 'h' | obj == 'r'
    if exist('linorm') == 3 % SLICOT is the fastest choice (3 means mex file)
        slicot = 1;
        if prtlevel > 0
            fprintf('hifoo: using "linorm" from SLICOT to compute')
            if obj == 'h'
                fprintf(' H-infinity norm\n')
            else
                fprintf(' complex stability radius\n')
            end
        end
    elseif exist('lti') == 2 % Control System Toolbox is 2nd choice (2 means m-file)
        if prtlevel > 0 
            fprintf('hifoo: using Control System Toolbox to compute')
            if obj == 'h'
                fprintf(' H-infinity norm\n')
            else
                fprintf(' complex stability radius\n')
            end
            fprintf('hifoo: *** install "linorm" from SLICOT for MUCH better performance\n')
            fprintf('hifoo: see www.cs.nyu.edu/overton/software/hifoo or www.slicot.de\n')
        end
        slicot = 0;  % Control System Toolbox
    else
        error('hifoo: either SLICOT (linorm) or Control System Toolbox must be installed when input "obj" is "h" or "r"')
    end
else
    slicot = 0; % don't need H-infinity norm when obj is 's' or '+'
end
pars.slicot = slicot;  % passed to hinfspec via hanso
% check out whether QUADPROG is available
whichquadprog = which('quadprog');
if isempty(whichquadprog)
    if prtlevel > 0
        fprintf('hifoo: will not be able to run final phases of optimization since "quadprog" is not in path\n')
        fprintf('hifoo: *** install Optimization Toolbox or MOSEK (www.mosek.com) for better results\n') 
    end
    options.quadprog = -1;  % pass to hanso so it does not keep checking
elseif ~isempty(findstr('toolbox',whichquadprog)) & ~isempty(findstr('optim',whichquadprog)) 
    if prtlevel > 0
        fprintf('hifoo: using "quadprog" from Optimization Toolbox\n')
    end
    options.quadprog = 1;
elseif prtlevel > 0 & ~isempty(findstr('mosek',whichquadprog))
    if prtlevel > 0
        fprintf('hifoo: using "quadprog" from MOSEK\n')
    end
    options.quadprog = 1;
elseif prtlevel > 0
    if prtlevel > 0
        fprintf('hifoo: using an unrecognized version of "quadprog" \n')
    end
    options.quadprog = 1;
end
% check whether hanso is available
if exist('hanso') ~= 2
    error('hifoo: "hanso" must be installed from www.cs.nyu.edu/overton/software/hanso/')
end
if isempty(init)
    % no initial point provided, starting points are randomly generated
    x0 = randn(nvar,nrand);
    if prtlevel > 0
        if nrand > 1
            fprintf('hifoo: no initial point provided, using %d randomly generated starting points\n', nrand)
        else
            fprintf('hifoo: no initial point provided, using a randomly generated starting point\n')
        end
    end
else
    % set up initial point for optimization, expanding (or truncating) the 
    % given data to correspond to the desired order if necessary
    xinit = xinitsetup(m, p, order, init, obj, barrier, pars); 
    % starting points include progressively bigger perturbations of xinit
    scalepert = norm(xinit)*logspace(-3, -1, nrand); % spaced from 0.001 to 0.1
    xinitpert = randn(nvar, nrand)*diag(scalepert);
    x0 = [xinit    xinit*ones(1,nrand) + xinitpert]; 
    if prtlevel > 0
        if nrand > 1
            fprintf('hifoo: supplementing initial point with %d randomly generated perturbations\n', nrand)
        else
            fprintf('hifoo: supplementing initial point with a randomly generated perturbation\n')
        end
    end
end
if obj == 'h' | obj == 'r' | obj == '+'
    % for these objectives, first step is to get stable
    pars.objective = 's';   % to this end, minimize spectral abscissa
    pars.penalty = 0;       % without any penalty term (barrier irrelevant)
    options.fvalquit = 0;   % stopping when get stable
    options.maxit = 1000;   % run a long time if necessary
    if prtlevel > 0
        fprintf('hifoo: searching for stabilizing order %d controllers\n', order)
        if nrand > 1
            fprintf('hifoo: if stabilization time is excessive, reduce options.nrand\n')
        end
    end
    % use BFGS alone at first, as in most cases this will be sufficient
    % call it one starting point at a time, because when objective is '+',
    % want to stop when a stable point is found, while for objectives
    % 'h' and 'r', want to find several stable points for initializing
    % optimization
    for k=1:size(x0,2)
        options.x0 = x0(:,k);
        % might waste too much of the allotted cputime stabilizing many
        % start points but complicated to avoid, user can adjust options
        options.cpumax = cpufinish - cputime;
        [xB(:,k),fB(k),gB(:,k)] = bfgs(pars,options);  
        if cputime > cpufinish
            if prtlevel > 0
                fprintf('hifoo: quit stabilizing since CPU time limit exceeded\n')
            end    
            break % not return
        end
        if obj == '+' & fB(k) < 0 % only want one stable point
            break % not return
        end
    end
    % if BFGS did not find a stable point, try gradient sampling,
    % initializing with the best half of the points found by BFGS, 
    % starting with the lowest
    % (no point using local bundle, not trying to verify optimality)
    if cputime < cpufinish & min(fB) >= 0 
        if prtlevel > 1
            fprintf('hifoo: BFGS did not find any stabilizing controllers, trying gradient sampling\n')
        end
        [fBsort,indx] = sort(fB);
        indx = indx(1:ceil(length(fB)/2));
        xBsort = xB(:,indx);
        options.maxit = 100; % gradient sampling is more expensive than BFGS
        for k = 1:length(indx)
            options.x0 = xBsort(:,k);
            options.cpumax = cpufinish - cputime;
            [xGS, fGS, gGS] = gradsamp(pars,options);
            if cputime > cpufinish
                if prtlevel > 0
                    fprintf('hifoo: quit stabilizing since CPU time limit exceeded\n')
                end    
                break % not return
            end
            if fGS < 0 % settle for one stable point
                xB = xGS; fB = fGS; gB = gGS;
                break % not return
            end
        end
    end
    stabindx = find(fB < 0);
    nstabpts = length(stabindx);
    if nstabpts > 0
        if prtlevel > 0
            if min(fB) > -1e-8
                qualification = '(barely) ';
            else
                qualification = '';
            end
            if nstabpts == 1
                fprintf('hifoo: found a %sstabilizing controller', qualification)
            else
                fprintf('hifoo: found %d %sstabilizing controllers', nstabpts, qualification)
            end
            if obj ~= '+'
                fprintf(' for initializing optimization\n')
            else
                fprintf(' , quitting\n')
            end
        end
        x0 = xB(:,stabindx);  % at least one stable point was found
        f0 = fB(:,stabindx);
        foundstablepoint = 1;
    else
        if prtlevel > 0
            fprintf('hifoo: could not find a stabilizing order %d controller\n', order)
            fprintf('hifoo: returning controller with best spectral abscissa %g instead\n', min(fB))
            fprintf('hifoo: try specifying a new "init" or increasing "options.cpumax"\n')
        end
        foundstablepoint = 0;
    end
    if obj == '+'| ~foundstablepoint % nothing else to do
        % fB and xB, not f0 and x0, as f0 and x0 are empty in 2nd case
        [f, k] = min(fB); % in both cases, return one with lowest abscissa
        if obj ~= '+'
            f = inf; % but return infinite objective when obj is 'h' or 'r'
        end
        x = xB(:,k);
        K = [];
        [K.a, K.b, K.c, K.d] = getABCDhat(order, m, p, x);
        loc.dnorm = nan;
        loc.evaldist = nan;
        return
    end
end
pars.objective = obj; % optimization objective
objectivename = objname(obj, pars);
pars.objname = objectivename; % may be handy for display purposes
pars.penalty = penalty;   % penalty term on ||x||_2
if penalty > 0
    penaltystring = ' (plus penalty term)';
else
    penaltystring = '';
end
pars.barrier = barrier;   % barrier term: multiple of inverse of stability radius
if barrier > 0
    barrierstring = ' (plus barrier term)';
else
    barrierstring = '';
end
options.fvalquit = -inf; % do not quit early
options.x0 = x0;   % multiple starting points, stable if obj is 'h' or 'r'
if prtlevel > 0
    fprintf('hifoo: optimizing %s%s%s for order %d controller\n', ...
        objectivename, penaltystring, barrierstring, order)
    fprintf('hifoo: if optimization time is excessive, reduce options.cpumax and/or options.nrand\n')
end
% run HANSO from all the valid starting points, returning only best result
% HANSO is a hybrid of BFGS, local bundle and gradient sampling
options.phasemaxit = [1000, nvar + 10, 100];
% options.phasemaxit = [0, 0, 100];
options.phasenum = [size(options.x0,2), 1, 2];
options.cpumax = cpufinish - cputime; % time left after stabilization
[x,f,loc] = hanso(pars,options);
if cputime > cpufinish
    if prtlevel > 0
        fprintf('hifoo: quit optimizing since CPU time limit exceeded\n')
    end
end
if prtlevel > 0
    fprintf('hifoo: best order %d controller found has %s%s%s %g\n', ...
        order, objectivename, penaltystring, barrierstring, f)
    fprintf(' with local optimality measure: dnorm = %5.1e, evaldist = %5.1e\n',...
        loc.dnorm, loc.evaldist')
end
% return the best point found, translated into controller format
K = [];
[K.a, K.b, K.c, K.d] = getABCDhat(order, m, p, x);
% display eigenvalues of final Abig when spectral abscissa was optimized
 if prtlevel > 0 & obj == 's'
    [Abig, Bbig, Cbig, Dbig] = ...
        getABCDbig(pars.A, pars.B1, pars.B2, pars.C1, pars.C2,...
            pars.D11, pars.D12, pars.D21, order, x);
    fprintf('hifoo: final optimized eigenvalues are \n')
    disp(num2str(eig(Abig)))
end