function [K, f, loc] = hifoo(plant, varargin)
%HIFOO  H-infinity fixed-order optimization.
%   K = HIFOO(PLANT) attempts to return a static output feedback
%    controller for PLANT that locally optimizes H-infinity performance.
%   K = HIFOO(PLANT, ORDER) attempts to return an order ORDER
%    controller for PLANT that locally optimizes H-infinity performance.
%   K = HIFOO(PLANT, ORDER, 's') attempts to return an order ORDER
%    controller for PLANT that locally optimizes the spectral abscissa
%    (pushing closed-loop poles as far left as possible in complex plane).
%   [K, F, LOC] = HIFOO(PLANT, ORDER, INIT, OBJ, OPTIONS)
%    looks for an order ORDER controller [Ahat Bhat; Chat Dhat] for 
%    PLANT that locally optimizes the objective OBJ for 
%    the transfer function Cbig (sI - Abig)^{-1}) Bbig + Dbig where
%     |Abig Bbig|   |A  0 B1 |   |0 B2 | |Ahat Bhat| |0  I 0  |
%     |         | = |0  0 0  | + |I 0  | |Chat Dhat| |C2 0 D21|
%     |Cbig Dbig|   |C1 0 D11|   |0 D12|
%    starting from initial guess INIT, with options given in OPTIONS,
%    returning the controller K, objective value F and local optimality
%    certificate LOC.
%  Input Parameters:
%   PLANT must be provided first and has one of 3 formats:
%     - structure with fields 
%        A,B1,B2,C1,C2,D11,D12,D21 
%        (B can be used instead of B2, C can be used instead of C2)
%        (B1,C1,D11,D12,D21 are required only for optimizing 
%         H-infinity performance, and are ignored otherwise), OR
%     - SS object, encoding 
%        A,[B1 B2],[C1; C2],[D11 D12; D21 0] in standard format 
%        (when optimizing H-infinity performance, the index partitioning 
%         of [B1 B2] and [C1 C2] must be specified in InputGroup.U1, 
%         InputGroup.U2, OutputGroup.Y1 and OutputGroup.Y2, which are
%         set and viewed by "set" and "get" respectively)
%        (for other objectives, if the partitioning is not specified,
%         it is assumed that B1 and C1 are empty), OR
%     - string giving the COMPleib name of the plant (see www.compleib.de)
%
%   ORDER, INIT, OBJ, OPTIONS are optional and may be given in any order
%    ORDER: order of controller (the dimension of Ahat)
%     (default: 0 (static output feedback))
%    INIT: initial guess for controller, has one of 2 formats:
%     - structure with fields: a, b, c, d, specifying Ahat,Bhat,Chat,Dhat
%     - SS object
%     If the order of the initial guess is less than the desired order, 
%     the initial guess is augmented to have the desired order without 
%     increasing the objective value.  Thus this routine can be called 
%     repeatedly to get successively better controllers as the order is 
%     increased. If the order of the initial guess is greater than the
%     desired order, the initial guess is truncated arbitrarily.
%     (default: generated randomly)
%    OBJ: objective, one of the following: 
%       'h': minimize H-infinity norm of transfer function from
%             performance input to performance output (default)
%       'r': minimize inverse of complex stability radius of Abig
%       's': minimize spectral abscissa (max(real(eig))) of Abig
%       'p': minimize pseudospectral abscissa of Abig (see options.epsilon)
%       '+': stabilize Abig, but do not optimize
%       (in all cases except 'h', fields B1, C1, D11, D12, D21 are ignored)
%    OPTIONS: structure with all fields optional:
%       options.penalty:  weight for adding penalty term to objective,
%          sqrt(||K.a||^2 + ||K.b||^2 +  ||K.c||^2 + ||K.d||^2)  
%          (default 0)
%       options.barrier: for objective = 'h' only: weight for adding 
%          inverse of complex stability radius to objective, to avoid
%          solution on boundary of stability domain (default 0)
%       options.nrand: number of starting points to use in addition to init 
%          (random perturbations of increasing sizes if init is provided)
%          (must be a positive integer; default: 3)
%       options.normtol: termination tolerance (see output parameter LOC)
%          (default: 1e-3)
%       options.evaldist: evaluation distance (see output parameter LOC)
%          (default: 1e-3)
%       options.cpumax:  quit when cpu time in seconds exceeds this
%          (must be a positive number; default: inf)
%       options.prtlevel: one of 0 (no printing), 1 (minimal, default),
%          2 (includes output from HANSO), 3 (verbose)
%       options.epsilon: in case that obj is 'p', value of epsilon 
%          defining epsilon-pseudospectral abscissa objective 
%          (ignored if obj ~= 'p'; default: 0.01)
%   Output parameters
%   K: best controller found, has one of 2 formats:
%     - structure with 4 fields: a, b, c, d specifying Ahat,Bhat,Chat,Dhat 
%     - SS object 
%     the format is compatible with the format used for INIT
%     (or the format used by PLANT if INIT is not provided)
%   F: corresponding value of objective, including any penalty/barrier term 
%     (inf if no stable point was found and obj is 'h' or 'r')
%     (spectral abscissa whether or not stable point was found if obj is '+')
%   LOC: local optimality certificate, structure with 2 fields:
%       loc.dnorm: norm of a vector in the convex hull of bundled or
%          sampled gradients of the objective function (including any
%          penalty/barrier term) evaluated at and near K 
%       loc.evaldist: specifies max distance from K at which these gradients 
%          were evaluated.  
%      The smaller loc.dnorm and loc.evaldist are, the more likely 
%       it is that K is an approximate local minimizer.
%      If quadprog is not in path, loc.dnorm is the norm of the gradient 
%       at K and loc.evaldist is 0.
%      In the case of objective '+', LOC is not relevant so both fields
%       are set to nan.
%
%   Other software needed
%    Required: HANSO (Hybrid Algorithm for Non-Smooth Optimization),
%      www.cs.nyu.edu/overton/software/hanso/
%    Required when objective is 'h' or 'r': either SLICOT's LINORM  
%     (highly recommended) or MATLAB's Control System toolbox (much slower)
%    Required when objective is 'p': pseudospectral abscissa routine pspa,
%      www.cs.nyu.edu/overton/software/pspa/
%    Strongly recommended: QUADPROG from either MOSEK or MATLAB's 
%      optimization toolbox
%
%   Further documentation is at www.cs.nyu.edu/overton/software/hifoo/
%   James V. Burke, Didier Henrion, Adrian S. Lewis and Michael L. Overton
%   Send comments/bug reports to Michael Overton, overton@cs.nyu.edu,
%   with a subject header containing the string "hifoo".

%   Method:
%   For objectives '+', 'h' and 'r', begin by minimizing spectral abscissa 
%   (without penalty term), starting from init and options.nrand successively 
%   larger random perturbations of init, or options.nrand random starts 
%   if init is not provided, until a stable point is reached, or default
%   iteration limits are exceeded. For objective '+', stop.
%   For objectives 'h' and 'r', if one or more stable points were found, 
%   switch to minimizing the objective (with any penalty/barrier term), 
%   starting at the stable points.  For objectives 's' and 'p', minimize  
%   objective (with any penalty/barrier term) directly. The minimization is 
%   done by HANSO, Hybrid Algorithm for Non-Smooth Optimization, a
%   hybrid of BFGS, local bundle and gradient sampling if QUADPROG is 
%   provided and BFGS alone if it is not. Termination takes place when
%   loc.dnorm <= options.normtol with loc.evaldist approximately <= 
%   options.evaldist, or default iteration limits are exceeded, or 
%   options.cpumax CPU time is exceeded.  If the output K is not 
%   satisfactory, call HIFOO again with INIT set to K.

% first decode the optional arguments provided to varargin
[order, init, obj, options] = hifooinputcheck(varargin);
if ~isfield(options, 'prtlevel')
    options.prtlevel = 1; % other fields checked in hifoomain
end 
prtlevel = options.prtlevel; 
if prtlevel > 0 % options fields printed by hifoomain
    fprintf('HIFOO Version 1.0\n') % July 2006
end
% convert input format for plant, making all the necessary input checks
ssinput = 0;
if ischar(plant) % plant is a string giving COMPleib name
    % convert from COMPleib format to ours
    if exist('COMPleib')~= 2
        error('hifoo: when input "plant" is a string "COMPleib" must be installed; see www.compleib.de')
    end
    [A,B1,B2,C1,C2,D11,D12,D21]=COMPleib(plant);
    if isempty(A)
        error('hifoo: input "plant" is a string but is not recognized by COMPleib')
    end
    pars = [];
    pars.A = A;
    pars.B1 = B1;
    pars.B2 = B2;
    pars.C1 = C1;
    pars.C2 = C2;
    pars.D11 = D11;
    pars.D12 = D12;
    pars.D21 = D21;
    % the problems in COMPleib all have D22 = 0
elseif strcmp(class(plant), 'ss') % SS object
    [A, B, C, D] = ssdata(plant);  % standard toolbox conversion
    plantstruct = get(plant); % convert to struct
    plantstructIn = plantstruct.InputGroup;
    if ~isfield(plantstructIn, 'U1')|~isfield(plantstructIn, 'U2')
        if prtlevel > 0
            fprintf('hifoo: "plant" is an SS object but cannot find U1 and U2 fields in "InputGroup"\n')
        end
        if obj == 'h'
            error('hifoo: cannot optimize H-infinity performance when input performance channel is unspecified')
        else
            U1 = [];
            U2 = 1:size(B,2);
            if prtlevel > 0
                fprintf('hifoo: partitioning B = [B1 B2] with B1 empty\n')
            end
        end
    else
        U1 = plantstructIn.U1;
        U2 = plantstructIn.U2;
        if any(sort([U1 U2]) ~= (1:size(B,2)))
            error('hifoo: input "plant" is an SS object but U1 and U2 fields in "InputGroup" do not provide a valid input partitioning')
        else
            if prtlevel > 0
                fprintf('hifoo: "plant" is an SS object, using input partitioning in "InputGroup"\n')
            end
        end
    end
    plantstructOut = plantstruct.OutputGroup;
    if ~isfield(plantstructOut, 'Y1')|~isfield(plantstructOut, 'Y2')
        if prtlevel > 0
            fprintf('hifoo: "plant" is an SS object but cannot find Y1 and Y2 fields in "OutputGroup"\n')
        end
        if obj == 'h' 
            error('hifoo: cannot optimize H-infinity performance when output performance channel is unspecified')
        else    
            Y1 = [];
            Y2 = 1:size(C,1);
            if prtlevel > 0
                fprintf('hifoo: partitioning C = [C1; C2] with C1 empty\n')
            end
        end
    else
        Y1 = plantstructOut.Y1;
        Y2 = plantstructOut.Y2;
        if any(sort([Y1 Y2]) ~= (1:size(C,1)))
            error('hifoo: input "plant" is an SS object but Y1 and Y2 fields in "OutputGroup" do not provide a valid output partitioning')
        else
            if prtlevel > 0
                fprintf('hifoo: "plant" is an SS object, using output partitioning in "OutputGroup"\n')
            end
        end
    end
    D22 = D(Y2,U2);
    if max(max(abs(D22))) > 0
        error('hifoo: input "plant" is an SS object but its D22 part is not zero; subsequent versions of HIFOO will handle D22 nonzero')
    end
    pars = [];
    pars.A = A;
    pars.B1 = B(:,U1);
    pars.B2 = B(:,U2);
    pars.C1 = C(Y1,:);
    pars.C2 = C(Y2,:);
    pars.D11 = D(Y1,U1);
    pars.D12 = D(Y1,U2);
    pars.D21 = D(Y2,U1);
    ssinput = 1;  % used at end in determining output format
elseif isstruct(plant) % A, B2 (or B) and C2 (or C) are always required
    if isfield(plant,'A')
        pars.A = plant.A;
    elseif isfield(plant, 'a')
        pars.A = plant.a;
    else
        error('hifoo: input "plant" is a structure but it does not have a field "A"')
    end
    if isempty(pars.A)
        error('hifoo: field "A" of input "plant" is empty')
    end
    % field B is a synonym for B2, and C for C2
    if isfield(plant,'B2')
        pars.B2 = plant.B2;
    elseif isfield(plant,'B')
        pars.B2 = plant.B;
    elseif isfield(plant,'b2')
        pars.B2 = plant.b2;
    elseif isfield(plant,'b')
        pars.B2 = plant.b;
    else
        error('hifoo: input "plant" is a structure but it does not have a field "B2" or "B"')
    end
    if isempty(pars.B2)
        error('hifoo: field "B2" (or "B") of input "plant" is empty')
    end
    if isfield(plant,'C2')
        pars.C2 = plant.C2;
    elseif isfield(plant,'C')
        pars.C2 = plant.C;
    elseif isfield(plant,'c2')
        pars.C2 = plant.c2;
    elseif isfield(plant,'c')
        pars.C2 = plant.c;
    else
        error('hifoo: input "plant" is a structure but it does not have a field "C2" or "C"')
    end
    if isempty(pars.C2)
        error('hifoo: field "C2" (or "C") of input "plant" is empty')
    end
    n = length(pars.A);
    m = size(pars.B2,2);
    p = size(pars.C2,1);
    % if the objective is not 'h', any B1, C1 or Dij provided 
    % should be ignored, so set them to empty
    if obj ~= 'h' 
        if (isfield(plant, 'B1') | isfield(plant, 'C1')) & prtlevel > 0
            fprintf('hifoo: objective is not "h" so ignoring plant.B1, plant.C1, etc\n')
        end
        pars.B1 = zeros(n,0);
        pars.C1 = zeros(0,n);
        pars.D11 = zeros(0,0);
        pars.D12 = zeros(0,m);
        pars.D21 = zeros(p,0);
    else % process the other fields (only the case of objective 'h')
        if isfield(plant,'b1') & ~isfield(plant,'B1')
            plant.B1 = plant.b1;
        end
        if isfield(plant,'c1') & ~isfield(plant,'C1')
            plant.C1 = plant.c1;
        end
        if isfield(plant,'d11') & ~isfield(plant,'D11')
            plant.D11 = plant.d11;
        end
        if isfield(plant,'d12') & ~isfield(plant,'D12')
            plant.D12 = plant.d12;
        end
        if isfield(plant,'d21') & ~isfield(plant,'D21')
            plant.D21 = plant.d21;
        end
        if ~isfield(plant,'B1')|~isfield(plant,'C1')|... 
         ~isfield(plant,'D11')|~isfield(plant,'D12')|~isfield(plant,'D21')
            if prtlevel > 0 % provide a little additional information
                fprintf('hifoo: input "plant" is a structure but one of B1, C1, D11, D12 or D21 is missing\n')
            end
            error('hifoo: cannot optimize H-infinity performance when input or output performance channels are unspecified')
        elseif isempty(plant.B1)|isempty(plant.C1)|isempty(plant.D11)|...
         isempty(plant.D12)|isempty(plant.D21)
            if prtlevel > 0 % provide a little additional information
                fprintf('hifoo: input "plant" is a structure but one of B1, C1, D11, D12 or D21 is empty\n')
            end
            error('hifoo: cannot optimize H-infinity performance when input or output performance channel is missing')
        else
            pars.B1 = plant.B1;
            pars.C1 = plant.C1;
            pars.D11 = plant.D11;
            pars.D12 = plant.D12;
            pars.D21 = plant.D21;
            if isfield(plant,'D22') % if D22 is present it must be 0
                if max(max(abs(plant.D22))) > 0
                    error('hifoo: input "plant" is a structure with a nonzero D22 field; subsequent versions of HIFOO will handle D22 nonzero')
                end
            end
        end
    end
    if diff(size(pars.A)) ~= 0
        error('hifoo: field "A" of input "plant" is not square')
    elseif norm(diff([size(pars.A,1) size(pars.B1,1) size(pars.B2,1)])) > 0
        error('hifoo: row dimension mismatch in fields A, B1, B2 of input "plant"')
    elseif norm(diff([size(pars.C1,1) size(pars.D11,1) size(pars.D12,1)])) > 0
        error('hifoo: row dimension mismatch in fields C1, D11, D12 of input "plant"')
    elseif size(pars.C2,1) ~= size(pars.D21,1)
        error('hifoo: row dimension mismatch in fields C2, D21 of input "plant"')
    elseif norm(diff([size(pars.A,2) size(pars.C1,2) size(pars.C2,2)])) > 0
        error('hifoo: column dimension mismatch in fields A, C1, C2 of input "plant"')
    elseif norm(diff([size(pars.B1,2) size(pars.D11,2) size(pars.D21,2)])) > 0
        error('hifoo: column dimension mismatch in fields B1, D11, D21 of input "plant"')
    elseif size(pars.B2,2) ~= size(pars.D12,2)
        error('hifoo: column dimension mismatch in fields B2, D12 of input "plant"')
    end
else
    error('hifoo: input "plant" must be a structure or an SS object or a string')
end
n = length(pars.A);
if order > n
    error(sprintf('hifoo: order %d is greater than state space dimension %d\n', order, n))
end
% now process the init argument 
% if it was omitted, it was already set to [] by hifooinputcheck, and
% this will be checked by hifoomain
if strcmp(class(init),'ss')
    [A, B, C, D] = ssdata(init);
    init = []; % redefine init to be a structure
    init.a = A; 
    init.b = B;
    init.c = C;
    init.d = D;
    ssinput = 1; % used at end in determining output format, 
                 % overrides setting based on input "plant"
elseif isstruct(init) % default [] is not a struct
    m = size(pars.B2,2);
    p = size(pars.C2,1);
    if isfield(init,'a')
        % nothing to do
    elseif isfield(init,'A')
        init.a = init.A;
    elseif isfield(init,'Ahat')
        init.a = init.Ahat;
    else
        init.a = zeros(0,0); % an order 0 (SOF) controller
    end
    initorder = size(init.a, 1);
    if initorder ~= size(init.a, 2)
        error('hifoo: input structure "init.a" must be square')
    end
    if isfield(init,'b')
        % nothing to do
    elseif isfield(init,'B')
        init.b = init.B;
    elseif isfield(init, 'Bhat')
        init.b = init.Bhat;
    else
        init.b = zeros(0,p); % an order 0 (SOF) controller
    end
    if size(init.b, 2) ~= p
        error('hifoo: input "init.b" must have column dimension %d', p)
    elseif size(init.b, 1) ~= initorder
        error('hifoo: inputs "init.a" and "init.b" must have same row dimension')
    end
    if isfield(init,'c')
        % nothing to do
    elseif isfield(init,'C')
        init.c = init.C;
    elseif isfield(init, 'Chat')
        init.c = init.Chat;
    else
        init.c = zeros(m,0); % an order 0 (SOF) controller
    end
    if size(init.c, 1) ~= m
        error('hifoo: input "init.c" must have row dimension %d', m)
    elseif size(init.c, 2) ~= initorder
        error('hifoo: inputs "init.a" and "init.c" must have same column dimension')
    end
    if isfield(init,'d')
        % nothing to do
    elseif isfield(init,'D')
        init.d = init.D;
    elseif isfield(init,'Dhat')
        init.d = init.Dhat;
    else % this is the only required field as it cannot be empty
        error('hifoo: input structure "init" must have a field "d"')
    end
    if any(size(init.d) ~= [m p]) % for example, if init.d is empty
        error('hifoo: input "init.d" must have dimension %d by %d', m, p)
    end
    ssinput = 0; % used at end in determining output format, 
                 % overrides setting based on input "plant"
end % if init == [], ssinput value is set depending on input "plant"
[K, f, loc] = hifoomain(pars, order, init, obj, options);
if ssinput % convert to ss object format of Control Systems Toolbox
    K = ss(K.a, K.b, K.c, K.d);
end