function [f, g] = hifooobj(x, pars)
% hifoo optimization objective is one of
%   pars.objective = 's': spectral abscissa
%   pars.objective = 'r': inverse of complex stability radius
%   pars.objective = 'p': inverse of complex stability radius
%   pars.objective = 'h': H-infinity norm
% along with a penalty term on ||x||_2, controlled by pars.penalty,
% AND, in the case of the H-infinity norm, a barrier on
% the stability region, controlled by pars.barrier.
% The second is a special case of the third (with pars.barrier=0).
% The parameters define a reduced order model, with order pars.nhat.
% These parameters, normally known as Ahat, Bhat, Chat, Dhat, are encoded
% in a single parameter vector x for the purposes of optimization.
% The fixed data are defined in 
%   pars.A, pars.B1, pars.B2, pars.C1, pars.C2, pars.D11, pars.D12, pars.D21
% (it is assumed that D22 = 0 so the parameterization is linear).
% This function returns the optimization objective as well as the gradient
% with respect to the parameters. If the function turns out to be infinite,
% the gradient will be nan, which is correctly propogated by chainABCD.
% The SLICOT routine linorm is used if pars.slicot == 1; otherwise,
% the norm function in Matlab's control system toolbox is used.
A = pars.A; B1 = pars.B1; B2 = pars.B2; C1 = pars.C1; C2 = pars.C2; 
D11 = pars.D11; D12 = pars.D12; D21 = pars.D21; % assume that D22 is zero
n = size(A,1);
nhat = pars.nhat;
[Abig, Bbig, Cbig, Dbig] = ...
    getABCDbig(A, B1, B2, C1, C2, D11, D12, D21, nhat, x);
objective = pars.objective;
slicot = pars.slicot;  % 1 if slicot is available, 0 otherwise
if nargout < 2 % function value only
    if objective == 's' % spectral abscissa
        f = specabsc(Abig);
    elseif objective == 'r' % inverse of complex stability radius
        I = eye(n + nhat);
        f = hinfty(Abig, I, I, zeros(n + nhat), slicot);
    elseif objective == 'h' % most important case: H-infinity norm
        f = hinfty(Abig, Bbig, Cbig, Dbig, slicot);
    elseif objective == 'p'; % pseudospectral abscissa
        f = pseudospecabsc(Abig, pars.epsilon);
    else % cannot happen
        fprintf('hifooobj: invalid objective\n')
        f = nan;
    end
else  % function and gradient
    if objective == 's'
        [f, G] = specabsc(Abig);
        g = chainABCD(n, nhat, B2, C2, D12, D21, G, ...
          zeros(size(Bbig)), zeros(size(Cbig)), zeros(size(Dbig)));
    elseif objective == 'r'
        I = eye(n + nhat);
        [f, G] = hinfty(Abig, I, I, zeros(n + nhat), slicot);
        g = chainABCD(n, nhat, B2, C2, D12, D21, G, ...
              zeros(size(Bbig)), zeros(size(Cbig)), zeros(size(Dbig)));
    elseif objective == 'p'
        [f, G] = pseudospecabsc(Abig, pars.epsilon);
        g = chainABCD(n, nhat, B2, C2, D12, D21, G, ...
              zeros(size(Bbig)), zeros(size(Cbig)), zeros(size(Dbig)));
    elseif objective == 'h' 
        [f, GA, GB, GC, GD] = hinfty(Abig, Bbig, Cbig, Dbig, slicot);
        g = chainABCD(n, nhat, B2, C2, D12, D21, GA, GB, GC, GD);
    else % cannot happen
        fprintf('hifooobj: invalid objective\n')
        f = nan;
        g = nan*ones(pars.nvar,1);
    end
end
% the penalty term is to prevent the norm of the controller from blowing up
penalty = pars.penalty;
if penalty > 0
    xnorm = norm(x);
    if xnorm > 0
        f = f + penalty*xnorm;
        if nargout >= 2
            g = g + penalty*x/xnorm; % gradient of the 2-norm
        end
    else
        d = randn(pars.nvar);  % random subgradient in this case
        if nargout >= 2
            g = g + penalty*d/dnorm;
        end
    end
end
% the barrier term addresses the fact that the H-infinity norm does
% not necessarily increase, and in fact may decrease, as the eigenvalues 
% of Abig go to zero, unlike the inverse of the stability radius. 
% This makes the objective very difficult to optimize, as the value 
% at the stability boundary is by definition infinity.  Adding a small
% multiple of the inverse of the stability radius to the H-infinity
% objective alleviates this difficulty, as it blows up as the
% stability boundary is approached, behaving as a barrier to instability.
if objective == 'h'
    barrier = pars.barrier;
    if barrier > 0  
        I = eye(n + nhat);
        if nargout < 2
            stabradinv = hinfty(Abig, I, I, zeros(n + nhat), slicot);
            f = f + barrier*stabradinv;
        else
            I = eye(n + nhat);
            [stabradinv, G] = hinfty(Abig, I, I, zeros(n + nhat), slicot);
            f = f + barrier*stabradinv;
            g = g + barrier*chainABCD(n, nhat, B2, C2, D12, D21, G, ...
              zeros(size(Bbig)), zeros(size(Cbig)), zeros(size(Dbig)));
        end
    end
end
            