function [xfinal, ffinal, dfinal, iters] = autoreduce(pars, x0, options);
% call: function [xfinal, ffinal, dfinal, iters] = autoreduce(pars, x0, options)
% repeatedly run gradient sampling minimization, for various sampling radii
% input:   
%        pars:       parameters defining the problem, including pars.m,
%          pars.fgname:       name of m-file coding function and gradient
%          pars.fcomparename: name of m-file performing function comparisons
%        x0:         starting point
%        options:    parameters defining the algorithm, including
%          options.samprad:    vector of sampling radius values to be tried
%          options.ngrad:      number of gradients per bundle
%          options.maxit:      max number of iterations per run   
%          options.dnormtol:   vector of tolerances on norm(d) 
%          options.xbound:     quit if inf-norm(x) hits this bound
%          options.qpsolver:   'M' for Mosek QP solver,
%                  'S' for Sedumi,  'T' for opt toolbox
%          options.prtlevel:   print level, typically 1
% output: xfinal:    final x iterates, for each sampling radius
%         ffinal:    final function values
%         dfinal:    final norm(d)
%         iters:     the iteration counts
%
%  Written by M. Overton (overton@cs.nyu.edu), last revised November 2003

for choice = 1:length(options.samprad)
% run the gradient sampling method with a fixed sample radius
   [x, f, d, a, hitbound] = ...
       gradsampfixed(pars, x0, options.samprad(choice), ...
           options.dnormtol(choice), options);
   ffinal(choice) = f(length(f)); 
   dfinal(choice) = d(length(d)); % actually norm of d
   xfinal(:,choice) = x(:,size(x,2));
   iters(choice) = length(f);     % even though first entry is initial point
% quit if function appears to be unbounded below
   if hitbound 
      if options.prtlevel > 0
         disp('hit bound on x - quit minimizing')
      end
      return  % no need to set a flag, norm of x should equal the bound
   end
% get ready for next run, with lower sampling radius
   x0 = x(:,size(x,2));   % start from where previous one finished,
                          % because this is lowest function value so far
end
