#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include "Barrier.h"
#include "Coin.h"
#include "sthread.h"
#include "Table.h"

struct container {
  Table *table;
  Barrier *barrier;
};


static void mikesTest();
static void sequentialTest();
static void parallelTest();
static void *make100(void *p);
static void *eat99(void *p);

static void *philosopher(void *arg);

/*
 *------------------------------------------------------------------
 *
 * mikesTest --
 *
 *          Run some very simple sanity checks.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
static void
mikesTest()
{
  sequentialTest();
  parallelTest();
}

/*
 *------------------------------------------------------------------
 *
 * sequentialTest --
 *
 *          Simple sanity check: count should be the difference
 *          between the number of works and the number of eats.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
static 
void sequentialTest()
{
  Table *t = new Table();
  t->work();
  t->work();
  assert(t->dbgCountFull() == 2);
  t->eat();
  t->eat();
  assert(t->dbgCountFull() == 0);
  delete t;
  printf("Pass sequentialTest.\n");
}

/*
 *------------------------------------------------------------------
 *
 * parallelTest --
 *
 *         Slightly more interesting test. Create three worker
 *         threads and three eat threads. 
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
static void
parallelTest()
{
  static const int NTHREADS = 6;
  struct container container;
  container.table = new Table();
  container.barrier = new Barrier(NTHREADS);
  sthread_t *threads = (sthread_t *)malloc(sizeof(sthread_t) * NTHREADS);
  sthread_create(&threads[0], make100, &container);
  sthread_create(&threads[1], make100, &container);
  sthread_create(&threads[2], make100, &container);
  sthread_create(&threads[3], eat99, &container);
  sthread_create(&threads[4], eat99, &container);
  sthread_create(&threads[5], eat99, &container);

  container.barrier->waitUntilDone();
  assert(container.table->dbgCountFull() == 3);
  delete container.table; 
  container.table = NULL;
  delete container.barrier;
  container.barrier = NULL;
  free(threads);
  threads = NULL;
  printf("Pass parallelTest.\n");
}

/*
 *------------------------------------------------------------------
 *
 * make100 --
 *
 *         Fill 100 bowls. Use a pseudo-random yield() pattern
 *         to give us some interesting patterns to test.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
void*
make100(void *p)
{
  struct container *container;
  int ii;
  Table *t;
  Barrier *b;
  int r;

  container = (struct container *)p;
  t = container->table;
  b = container->barrier;

  for(ii = 0; ii < 100; ii++){
    t->work();
    r = sutil_random();
    if(r % 1 == 0){
      sthread_yield();
    }
  }

  b->done();
  sthread_exit();
  return NULL;
}


/*
 *------------------------------------------------------------------
 *
 * eat99 --
 *
 *         Empty 99 bowls. Use a pseudo-random yield() pattern
 *         to give us some interesting patterns to test.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
void*
eat99(void *p)
{
  struct container *container;
  int ii;
  Table *t;
  Barrier *b;
  int r;

  container = (struct container *)p;
  t = container->table;
  b = container->barrier;

  for(ii = 0; ii < 99; ii++){
    t->eat();
    r = sutil_random();
    if(r % 1 == 0){
      sthread_yield();
    }
  }
  b->done();
  sthread_exit();
  return NULL;
}


/*
 *------------------------------------------------------------------
 *
 * moreTests --
 *
 *         Put more tests here.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
static void
moreTests()
{
}

/*
 *------------------------------------------------------------------
 *
 * runPhilosophers --
 *
 *         Fork off nPhilosophers philosophers threads.
 *         Each philosopher has a simple life:
 *     while(1){
 *         flip the coin  (there is one shared coin)
 *         if(head) Table.Work()
 *         else Table.Eat()
 *         Think()
 *     }
 *
 *         These threads never return (philosophers live 
 *         long healthy lives.)
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
static void
runPhilosophers(int nPhilosophers)
{
}


/*
 *------------------------------------------------------------------
 *
 * philosopher --
 *
 *         Main philosopher thread. 
 *
 * Results:
 *      Never returns.
 *
 *------------------------------------------------------------------
 */
void *
philosopher(void *arg)
{
  return NULL; // Make compiler happy
}

/*
 *------------------------------------------------------------------
 *
 * main --
 *
 *         Run tests on table and barrier.
 *
 * Results:
 *      None.
 *
 *------------------------------------------------------------------
 */
int
main(int argc, char **argv)
{
  mikesTest();
  moreTests();  
  runPhilosophers(42);
}



