#ifndef _POSIX_PTHREAD_SEMANTICS
#define _POSIX_PTHREAD_SEMANTICS
#endif
#include "sthread.h"
#include <assert.h>
#include <pthread.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

void smutex_init(smutex_t *mutex)
{
  if(pthread_mutex_init(mutex, NULL)){
      perror("pthread_mutex_init failed");
      exit(-1);
  }    
}

void smutex_destroy(smutex_t *mutex)
{
  if(pthread_mutex_destroy(mutex)){
      perror("pthread_mutex_destroy failed");
      exit(-1);
  }    
}

void smutex_lock(smutex_t *mutex)
{
  if(pthread_mutex_lock(mutex)){
    perror("pthread_mutex_lock failed");
    exit(-1);
  }    
}

void smutex_unlock(smutex_t *mutex)
{
  if(pthread_mutex_unlock(mutex)){
    perror("pthread_mutex_unlock failed");
    exit(-1);
  }    
}



void scond_init(scond_t *cond)
{
  if(pthread_cond_init(cond, NULL)){
      perror("pthread_cond_init failed");
      exit(-1);
  }
}

void scond_destroy(scond_t *cond)
{
  if(pthread_cond_destroy(cond)){
      perror("pthread_cond_destroy failed");
      exit(-1);
  }
}

void scond_signal(scond_t *cond, smutex_t * /* mutex NOTUSED */)
{
  //
  // assert(mutex is held by this thread);
  //

  if(pthread_cond_signal(cond)){
    perror("pthread_cond_signal failed");
    exit(-1);
  }
}

void scond_broadcast(scond_t *cond, smutex_t * /* mutex NOTUSED */)
{
  //
  // assert(mutex is held by this thread);
  //
  if(pthread_cond_broadcast(cond)){
    perror("pthread_cond_broadcast failed");
    exit(-1);
  }
}

void scond_wait(scond_t *cond, smutex_t *mutex)
{
  //
  // assert(mutex is held by this thread);
  //

  if(pthread_cond_wait(cond, mutex)){
    perror("pthread_cond_wait failed");
    exit(-1);
  }
}



void sthread_create(sthread_t *thread,
		    void (*start_routine(void*)), 
		    void *argToStartRoutine)
{
  if(pthread_create(thread, NULL, start_routine, argToStartRoutine)){
      perror("pthread_create failed");
      exit(-1);
  }
}

void sthread_exit(void)
{
  pthread_exit(NULL);
}





void sthread_sleep(unsigned int seconds, unsigned int nanoseconds)
{
  struct timespec rqt;
  assert(nanoseconds < 1000000000);
  rqt.tv_sec = seconds;
  rqt.tv_nsec = nanoseconds;
  if(nanosleep(&rqt, NULL) != 0){
    perror("sleep failed. Woke up early");
    exit(-1);
  }
}

void sthread_yield()
{
  pthread_yield();
}

int _SEED1=0x0AA;
int _SEED2=0x055;
int _ROTATE1COUNT=1;
int _ROTATE1DIR=0;
int _ROTATE2COUNT=3;
int _ROTATE2DIR=1;
int _RNDMASK=0x0F0;
int _NHITS=0;
int _CRITICAL=512;          /* an empirical value that seems to work well */
int _RNDFUDGE=0;
int _ODD_OR_EVEN=0;
int _AM_ERSTEN=1;           /* set our initial use flag */
int _RND_BUSY=0;            /* initial state is free */

pthread_mutex_t  sulock = PTHREAD_MUTEX_INITIALIZER;


int _rotate(int value, int direction, int rotcount);
void _rnd(); 
int uf2int(float a);
float int2uf(int a);


long sutil_random()         /* pseudo-random number generator              */

{
    float s1,s2;            /* needed to pass final arguments */
    long val;               /* return value */

    if(pthread_mutex_lock(&sulock)){
       perror("random lock");
       exit(-1);
    }


        if (_AM_ERSTEN==1)
        {
            /* do nothing, just return the existing random number */
        }
        else    /* do the whole shebang */
        {
            if (_ODD_OR_EVEN==1)    /* if it's an odd time */
            {
                _ODD_OR_EVEN=0;     /* flag that we're even */
                _rnd();             /* get a random number and throw it away */
            }
            else                    /* we're even */
                _ODD_OR_EVEN=1;     /* make us odd for next time */
            _rnd();                 /* now get a random number and keep it */
        }
    
    _AM_ERSTEN=0;                   /* mark that we've been here */
    s1=int2uf(_SEED1);              /* get our seeds as floats */
    s2=int2uf(_SEED2);

    val = uf2int(s1+(256.*s2));  /* make the final signed integer */
   
    if (val < 0)
       val = -val;
    
    
    if(pthread_mutex_unlock(&sulock)){
       perror("random unlock");
       exit(-1);
    }

    return val;
	     
}


/*********************************

  Following routines are used to implement the random generator sutil_random() shown
  above. You do not need to care about all these routines below

***********************************/


int _rotate(int value, int direction, int rotcount)
                                    /* internal routine which rotates bits */
                                    /* on entry, a=value to be rotated,    */
                                    /* b=direction (0=left, 1=right), and  */
                                    /* c=number of times to rotate it      */
                                    /* Note:  has fudge so that rotcount=0 */
                                    /* always results in 1 rotation.  This */
                                    /* is needed to break the symmetry of  */
                                    /* the algorithm; otherwise you get    */
                                    /* fractal patterns in the output.     */
{
    int n,mybit0;
    if (rotcount==0)
        rotcount++;             /* fudge */
    if (direction==0)
    {
        for (n=0;n<rotcount;n++)
        {
            value=value*2;          /* do a left shift */
            if ((value&256)>0)      /* we had a carry into bit 8 */
            {
                value=value|1;      /* so copy into bit 0 */
                value=value&255;    /* and wipe bits 8-15 */
            }
        }
    }
    else
    {
        for (n=0;n<rotcount;n++)
        {
            mybit0=value&1;         /* get state of bit 0 */
            value=value/2;          /* do a right shift   */
            if (mybit0>0)           /* if we had a carry, */
                value=value|128;    /* put it into bit 7  */
        }
    }
    return(value);      /* at wast, at wast, the scwewwy wabbits awe done! */
}


void _rnd()         /* internal routine which does all the work */
                    /* requires that the global variables be already set up */
                    /* changes the value of the global variables! */
{
    if ((_SEED1&1)>0)                   /* set rotate2dir */
        _ROTATE2DIR=1;                  /* bit 0 was set */
    else
        _ROTATE2DIR=0;

    if ((_SEED2&8)>0)                   /* set rotate1dir */
        _ROTATE1DIR=1;                  /* bit 3 was set */
    else
        _ROTATE1DIR=0;

    _ROTATE2COUNT=(_ROTATE1COUNT^_SEED2)&7;     /* keep it 0-7 */
    _ROTATE1COUNT=(_ROTATE2COUNT&_SEED1)&7;     /* keep it 0-7 */

    _SEED1=_rotate(_SEED1,_ROTATE1DIR,_ROTATE1COUNT);   /* rotate the seed 1 bits */
    _SEED2=_rotate(_SEED2,_ROTATE2DIR,_ROTATE2COUNT);   /* rotate the seed 2 bits */

    _RNDMASK-=_SEED2;
    if (_RNDMASK<0)
        _RNDMASK+=256;          /* keep it a positive number */

    _RNDMASK-=_RNDFUDGE;
    if (_RNDMASK<0)
        _RNDMASK+=256;          /* keep it a positive number */

    _RNDMASK-=_SEED1;
    if (_RNDMASK<0)
        _RNDMASK+=256;          /* keep it a positive number */

    _SEED1=_SEED2^_RNDMASK;
    _SEED2=_SEED1&_RNDMASK;

    _RNDMASK=(255-_RNDMASK)&255;    /* 1's complement, keeping only lower 8 bits */

    _NHITS++;                       /* one more hit */
    if (_NHITS==_CRITICAL)          /* is it the magic threshold? */
    {
        _RNDFUDGE=(_RNDFUDGE+1)&255;        /* keep only lower 8 bits */
        _NHITS=0;                           /* and reset the hit parade */
    }
}


int uf2int(float a)     /* converts a float representing an unsigned  */
                        /* int into its corresponding signed int form */
                        /* Note:  0<=a<=65535 only!                   */
                        /* returns 0 if a<0. and -1 if a>65535.       */
{
    if (a<0.)
        return (0);             /* underflow error */
    if (a>65536.)
        return  (-1);           /* overflow error */
   if (a>32768)
        return ((int)(a - 65536));
    else
        return ((int)(a));
}



float int2uf(int a)             /* converts a signed int into its    */
				/* corresponding unsigned form, and  */
				/* returns the value as a float.     */
	        		/* return values range 0. to 65535.  */
{
    if (a<0)
       return (((float)(a))+65536.);
    else
       return ((float)(a));
}




