package Jet.Refres;

import java.util.*;
import Jet.Tipster.*;
import Jet.Parser.StatParser;

/**
 *  functions for computing 'Hobbs distance' for reference resolution.
 */

public class Hobbs {
	
	/**
	 *  computes the distance (number of nodes traversed) in a Hobbs search
	 *  starting from parse tree node 'm2' and searching backwards for parse
	 *  tree node 'm1'.  'sentences' is a Vector of the sentences in the
	 *  Document.
	 */
	
	public static int distance (Annotation m1, Annotation m2, Vector sentences) {
		Span span1 = m1.span();
		Span span2 = m2.span();
		// if spans are nested, reject
		if (span1.within(span2) || span2.within(span1)) {
			return 99999;
		}
		// determine associated sentences for nodes m1 and m2
		int isent1 = -1;
		int isent2 = -1;
		Annotation sent1 = null;
		Annotation sent2 = null;
		for (int i = 0; i < sentences.size(); i++) {
			Annotation sentence = (Annotation) sentences.get(i);
			Span span = sentence.span();
			if (span1.within(span)) {
				isent1 = i;
				sent1 = sentence;
			}
			if (span2.within(span)) {
				isent2 = i;
				sent2 = sentence;
			}
		}
		if (isent1 < 0 || isent2 < 0) {	
			System.out.println ("Hobbs.distance:  cannot find containing sentences.");
			return 99999;
		}
		Annotation parse1 = (Annotation) sent1.get("parse");
		Annotation parse2 = (Annotation) sent2.get("parse");
		if (parse1 == null || parse2 == null) {
				System.out.println ("Hobbs.distance:  no parse.");
				return 99999;
		}	
		if (isent1 == isent2) {
			// find lowest node dominating m1 and m2
			Annotation lcdNode = lcd(m1, m2, parse2);
			if (lcdNode == null) {
				System.out.println ("Hobbs.distance:  no LCD");
				return 99999;
			}
			int d1 = bfsCount (lcdNode, m1);
			// compute lcd child
			Annotation lcdChild = null;
			Annotation[] children = StatParser.children(lcdNode);
			if (children != null)
				for (int ichild=0; ichild<children.length; ichild++)
					if (m2.span().within(children[ichild].span()))
						lcdChild = children[ichild];
			if (lcdChild == null) {
				System.out.println ("Hobbs.distance:  error");
				return 99999;
			}
			int d2 = bfsCount (lcdChild, m2);
			return d1 + d2;
		} else {
			// compute distance from root of s1 to m1
			int d1 = bfsCount(parse1, m1);
			// compute distance from root of lastSentence to m2
			int d2 = 1000 * (isent2 - isent1 -1);
			// add factor for intermediate sentences
			int d3 = bfsCount(parse2, m2);
			// return sum
			return d1 + d2 + d3;
		}
	}
	
	// find the lowest common dominator of nodes m1 and m2
	
	private static Annotation lcd (Annotation m1, Annotation m2, Annotation p) {
		Annotation[] children = StatParser.children(p);
		if (children == null)
			return null;
		for (int ichild=0; ichild<children.length; ichild++) {
			Annotation q = children[ichild];
			if (m1.span().within(q.span()) && m2.span().within(q.span()))
				return lcd(m1, m2, q);
		}
		return p;
	}
	
	/**
	 *  returns a count of the nodes encountered in a breadth-first
	 *  traversal of the tree rooted at 'root', up to the node 'target'.
	 */
	 
	private static int bfsCount (Annotation root, Annotation target) {
		LinkedList q = new LinkedList();
		int count = 0;
		q.add(root);
		while (q.size() > 0) {
			Annotation a = (Annotation) q.removeFirst();
			if (a == target)
				return count;
			if (a.get("mention") != null)
				count++;
			Annotation[] children = StatParser.children(a);
			if (children != null)
				for (int ichild=0; ichild<children.length; ichild++) {
					if (children[ichild] != null)
						q.add(children[ichild]);
				}
		}
		System.out.println ("Hobbs.bfs:  target not underneath root.");
		System.out.println ("            target = " + target);
		System.out.println ("            root = " + root);
		return 9999;
	}
}
