#!/usr/bin/python
##################################################
#Copyright (c) 2011, David F. Fouhey
#See License.txt
##################################################
#Spatial Analysis Code
##################################################

from common import *
from random import *

TRIANGULATE_EXEC = "./triangulate"

def generateDelaunayTriangulation(matches):
    """Get a dictionary giving the Delaunay triangulation of a point set such that q in D[p] if and only if 
    there's an edge between p and q. The returned dictionary uses the original points and thus lookups can be
    done without resolving point sameness."""
    try:
        #open a pipe to the triangulation executable
        dtInFile, dtOutFile = getTempFilename(suffix=".txt"), getTempFilename(suffix=".txt")
        dtIn = file(dtInFile,"w")
        #write in the points in order
        for p in matches:
            x, y = p[:2]
            dtIn.write(str(x)+" "+str(y)+" ")
        dtIn.close()
        dtCom = "%s %s < %s > %s" % (TRIANGULATE_EXEC, str(len(matches)), dtInFile, dtOutFile)
        os.system(dtCom)
        dtOut = file(dtOutFile)
        #wait for the triangulation to be done
        tri = {}
        lineCount = 0
        #Each line will be of the form node connectedNode0 connectedNode1 ... connectedNodeN
        #Each edge will be given in duplicate form, i.e. pq and qp will both be given.
        for line in dtOut.readlines():
            lineCount += 1
            splitLine = line.split()
            if len(splitLine) == 0:
                #if the line's blank, ignore it
                continue
            #p is the node that all the following nodes are connected to
            p = int(splitLine[0])
            tri[matches[p]] = []
            for q in map(int, splitLine[1:]):
                tri[matches[p]].append(matches[q])
        os.remove(dtInFile)
        os.remove(dtOutFile)
        return tri
    except Exception as e:
        print "Something went wrong when triangulating"
        print e

def connectedComponents(tri):
    """Return the connected components in a triangulation"""
    #get the nodes of the triangulation
    clusters = [[c] for c in tri]

    #get the edges
    edges = []
    for p in tri:
        for q in tri[p]:
            edges.append((p,q))

    for p, q in edges:
        #find which clusters p and q are in
        clusterP = reduce(lambda s1, s2: s1 if p in s1 else s2, clusters, [])
        clusterQ = reduce(lambda s1, s2: s1 if q in s1 else s2, clusters, [])
        if clusterP != clusterQ:
            clusterUnion = clusterP+clusterQ
            clusters.remove(clusterQ)
            clusters = [clusterUnion if cluster == clusterP else cluster for cluster in clusters]
    return clusters

def getFragmentedDT(matches):
    """Take the delaunay triangulation, and remove edges with length greater 
    than 1 sigma"""
    tri = generateDelaunayTriangulation(matches)
    #compute all the edge lengths
    edgeLengths = []
    for p in tri:
        for q in tri:
            edgeLengths.append(ED(p[:2], q[:2]))
    mu, sigma = getMeanAndSDev(edgeLengths)
    cutoff = mu+sigma

    #make a new triangulation with only the edge lengths
    #within a standard deviation
    tri2 = {}
    for p in tri:
        tri2[p] = []
        for q in tri[p]:
            if ED(p[:2],q[:2]) < cutoff:
                tri2[p].append(q)
    
    clusters = connectedComponents(tri2)
    return clusters

def fragmentClusters(clusters, minimumSize):
    """Break a list of clusters into clusters"""
    fixedClusters = []
    for cluster in clusters:
        clustersFragmented = [c for c in getFragmentedDT(cluster) if len(c) >= minimumSize]
        fixedClusters += clustersFragmented
    return fixedClusters
        

