/*
 * Triangulation Program
 * David Fouhey
 *
 *
 * Copyright (c) 2011, David F. Fouhey
 * See ../License.txt
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>


#define epsilon 1e-6
#define ABS(x) (((x) > 0) ? (x) : (-1*(x)))
#define MIN(x,y) (((x) > (y)) ? (y) : (x))
#define MAX(x,y) (((x) > (y)) ? (x) : (y))

#define FLIP_USING_T(x,y) t = (x); (x) = (y); y = (t);

typedef struct IndexEdge_t{
    double x0, y0, x1, y1, length;
    int i0, i1;
} IndexEdge;


                                       
double getEuclideanDist(double px, double py, double qx, double qy){
    double dx = px - qx, dy = py - qy;
    return sqrt(dx*dx + dy*dy);
}

                                      
/*
 * See if the segment between p00 and p01 intersects the segment between
 * p10 and p11.
 *
 * This isn't particularly robust. There's probably a beautiful solution, 
 * involving inspecting the number of solutions to a certain linear system
 * of equations. A quick search finds no particularly short ones; I haven't
 * looked hard enough though. But, get it working, then get it beautiful.
 *
 */
int lineSegmentsIntersect(  double p00x, double p00y, double p01x, double p01y,
                            double p10x, double p10y, double p11x, double p11y){
    double slope0, slope1, b0, b1, xIntersect;
    int vertical0 = (ABS(p00x - p01x) < epsilon), vertical1 = (ABS(p10x -p11x) < epsilon);
    /*
    * But! if the two lines intersect at one point which is the endpoints, 
    * we don't want this to count as an intersection */
    int count = 0;
    /* we can unroll it, or the compiler can. Note that we'd have to set up
    * some sort of array first though, which would be as much typing work
    * as this and the comment. */
    if(getEuclideanDist(p00x, p00y, p10x, p10y) < epsilon){
        count++;
    }
    if(getEuclideanDist(p00x, p00y, p11x, p11y) < epsilon){
        count++;
    }
    if(getEuclideanDist(p01x, p01y, p10x, p10y) < epsilon){
        count++;
    }
    if(getEuclideanDist(p01x, p01y, p11x, p11y) < epsilon){
        count++;
    }
    if(count == 1){
        return 0;
    } 

    if(vertical0 && vertical1){
        /* if they're on the same vertical line, and if so, is their y-overlap
         * 2*epsilon. 2 epsilon for a degenerate case where they both meet at the top,  
         * and dx in both lines is epsilon. 
         *
         * If the two have a y overlap, then the bottom will be less than the top of the other
         */
        double b0 = MIN(p00y, p01y), t0 = MAX(p00y, p01y);
        double b1 = MIN(p10y, p11y), t1 = MAX(p10y, p11y);
        double dX = p00x - p10x;
        if((ABS(dX) < 2*epsilon) && (b0 < t1) && (b1 < t0)){
            return 1;
        }
/*        printf("Exit point 0\n"); */
        return 0; 
    }
    if(vertical1){
        /* flip variables around so we can avoid duplicating code */
        int t;
        FLIP_USING_T(p00x, p10x);
        FLIP_USING_T(p00y, p10y);
        FLIP_USING_T(p01x, p11x);
        FLIP_USING_T(p01y, p11y);
    }
    /* now if vertical 1 is true, the lines have been switched so that 
     * p00 and p01 represent the vertical line
     */
    if(vertical0 || vertical1){
        double b1, y;
        int yWithin, xWithin;
        slope1 = (p11y - p10y) / (p11x - p10x);
        b1 = (p10y - p10x*slope1);
        y = p00x*slope1 + b1;
        yWithin = (MIN(p00y, p01y) <= y) && (y <= MAX(p00y, p01y));
        xWithin = (MIN(p10x, p11x) <= p00x) && (p00x <= MAX(p10x, p11x));
        if(!(xWithin && yWithin)){
/*            printf("Exit point 1\n"); */
            return 0;
        }
        return 1;
    }
    /* more normal cases */
    slope0 = (p01y - p00y) / (p01x - p00x);
    slope1 = (p11y - p10y) / (p11x - p10x);
    /* slope0 = (p00y - b) / (p00x - 0) =>> p00x * slope0 = p00y - b ==> p00[0] *slope0 = -b + p00y
     * -p00y + p00x * slope0 = -b =>> p00y - p00x * slope0 = b */
    b0 = (p00y - p00x * slope0);
    b1 = (p10y - p10x * slope1);
    if( ABS(slope1 - slope0) < epsilon){
        if( (ABS(b1 - b0) < epsilon) && (MIN(p00x, p01x) < MAX(p10x, p11x)) && (MIN(p10x, p11x) <= MAX(p00x, p01x))){
            return 1;
        }
/*        printf("Exit point 2\n"); */
        return 0;
    }
    /* slope0*x + b0 = slope1*x + b1 =>> slope0*x - slope1*x = b1 - b0 => x*(slope0 - slope1) = b1 - b0
     * x = (b1 - b0) / (slope0 - slope1)
     */
    xIntersect = (b1 - b0) / (slope0 - slope1);
    if( (MIN(p00x, p01x) <= xIntersect) && (xIntersect <= MAX(p00x, p01x)) &&
        (MIN(p10x, p11x) <= xIntersect) && (xIntersect <= MAX(p10x, p11x)) ){
        return 1;
    }
/*
    printf("Exit point 4\n");
    printf("Slopes: %f | %f\n",slope0, slope1);
    printf("Bs: %f | %f\n",b0, b1);
    printf("xIntersect: %f\n",xIntersect);
    printf("Min x0/1: %f | %f\n",MIN(p00x, p01x), MIN(p10x,p11x));
    printf("Max x0/1: %f | %f\n",MAX(p00x, p01x), MAX(p10x,p11x));
    */
    return 0;
        
}


double determineTurn(   double px, double py, 
                        double qx, double qy,
                        double rx, double ry ){
    return (px*qy) - (px*ry) - (py*qx) + (py*rx) + (qx*ry) - (rx*qy);
}

int isLeftTurn(double value){
    return (value > epsilon);
}

int isRightTurn(double value){
    return (value < -epsilon);
}

int isColinear(double value){
    return !(isLeftTurn(value) || isRightTurn(value));
}

/*
 * Tests whether s is in the circle given by p, q, and r.
 *
 */
double doCircleTest(    double px, double py,
                        double qx, double qy,
                        double rx, double ry,
                        double sx, double sy 
                    ){
    double pz, qxp, qyp, qzp, rxp, ryp, rzp, sxp, syp, szp;
    if(isRightTurn(determineTurn(px, py, qx, qy, rx, ry))){
        return doCircleTest(px, py, rx, ry, qx, qy, sx, sy);
    }
    pz = (px*px + py*py);
    qxp = qx - px; qyp = qy - py; qzp = (qx*qx + qy*qy) - pz;
    rxp = rx - px; ryp = ry - py; rzp = (rx*rx + ry*ry) - pz;
    sxp = sx - px; syp = sy - py; szp = (sx*sx + sy*sy) - pz;
    return -1*( qxp*(ryp*szp - syp*rzp) + qyp*(rzp*sxp - szp*rxp) + qzp*(rxp*syp - sxp*ryp) );
}

int isInCircle(double value){
    return (value > epsilon);
}
  

/*
 * Adds edges in tri between p and q (noting both ways
 * in the adjacency list)
 *
 *
 * If all is well with the graph, the calls should go as 
 * follows
 *
 * addToTriangulationGraph(g, p1, p2)
 *  '->addToTriangulationGraph(g, p2, p1)
 *      '->addToTriangulationGraph(g, p1, p2)
 *          '->Finds that p2 is in p1's adjacency list, stops
 * 
 */
void addToTriangulationGraph(int **graph, int p, int q){
    int i, *vec = graph[p], vecsize, nextPlace;
    vecsize = vec[0];
    nextPlace = -1;
    for(i = 1; i < vecsize; i++){
        if(vec[i] == q){
            return;
        }
        if(vec[i] == -1){
            nextPlace = i;
            break;
        }
    }
    if(nextPlace == -1){
        /* if we can't find the next place to stick the 
         * reference for q, then the 'list' is full and
         * we need to reallocate.
         */
        nextPlace = vecsize;
        vec = realloc(vec, (sizeof(float)*vecsize*2));
        /* place q in the first spot in the new space */
        vec[nextPlace] = q;
        /* set the rest to -1 (or not used) */
        for(i = vecsize+1; i < vecsize*2; i++){
            vec[i] = -1; 
        }
        /*update the listing of how big it is */
        vec[0] = vecsize*2;
        graph[p] = vec;
    } else{
        vec[nextPlace] = q;
    }
    addToTriangulationGraph(graph, q, p);
}

void removeFromTriangulationGraph(int **graph, int p, int q){
    int i, *vec = graph[p], vecsize, itemPlace;
    itemPlace = -1;
    vecsize = vec[0];
    for(i = 1; i < vecsize; i++){
        if(vec[i] == q){
            itemPlace = i;
            break;
        }
    }
    if(itemPlace == -1){
        return;
    }
    vec[itemPlace] = -1;
    removeFromTriangulationGraph(graph, q, p);
}

int inTriangulationGraph(int **graph, int p, int q){
    int i, *vecp = graph[p];
    int psize = vecp[0];
    for(i = 1; i < psize; i++){
        if(vecp[i] == q){
            return 1;
        }
    }
    return 0;
}


int indexEdgeComparator( const void *ev0, const void *ev1){
    IndexEdge *e0 = *((IndexEdge **) ev0), *e1 = *((IndexEdge **) ev1); 
    if(e0->length < e1->length){
        return -1;
    }
    return 1;
}

int **computeTriangulation(int n, double *pointsX, double *pointsY){
    /* Compute a delaunay triangulation of the points (x_i,y_i), where each
     * x_i is in X, etc.
     *
     * This computes a triangulation by making an arbitary triangulation, and
     * then modifying it to be a Delaunay one by flipping edges within 
     * quadrilaterals. This procedure is guaranteed to converge.
     */
    int i, j, k;
    int candidateEdgeSize = n*(n-1)/2;
    double **candidateEdges = malloc(sizeof(double *)*(candidateEdgeSize));
    int **candidateEdgeIndices = malloc(sizeof(int *)*(candidateEdgeSize));
    IndexEdge **toSort = malloc(sizeof(IndexEdge *)*(candidateEdgeSize));
    double **edges = malloc(sizeof(double *)*(3*n-6));
    /* make it slightly bigger than the maximum number of edges */
    int stackCapacity = 4*n-6;
    int **considerStack = malloc(sizeof(int *)*stackCapacity);
    int stackCount; 
    int **tri = malloc(sizeof(int *)*n);
    int triSize;
    k = 0;

    /* Allocate everything */
    for(i = 0; i < n; i++){
        for(j = i; j < n; j++){
            if(j == i){
                continue;
            }
            /* allocate them here, for no particular reason */
            candidateEdges[k] = malloc(sizeof(double)*4); 
            candidateEdgeIndices[k] = malloc(sizeof(int)*2);

            toSort[k] = malloc(sizeof(IndexEdge));
            toSort[k]->x0 = pointsX[i];
            toSort[k]->y0 = pointsY[i];
            toSort[k]->x1 = pointsX[j];
            toSort[k]->y1 = pointsY[j];
            toSort[k]->length = getEuclideanDist(pointsX[i], pointsY[i], pointsX[j], pointsY[j]);
            toSort[k]->i0 = i;
            toSort[k]->i1 = j;

            k += 1;                
        }
        /* set up the triangulation graph as well */
        tri[i] = malloc(sizeof(int)*4);
        tri[i][0] = 4;
        for(j = 1; j < 4; j++){
            tri[i][j] = -1;
        }
    }

    /* Sort the edges */    
    qsort(toSort, candidateEdgeSize, sizeof(IndexEdge *), indexEdgeComparator);

    /* stackCount is how many confirmed edges we have */
    stackCount = 0;
    /* i goes through our candidate edges;
     * add edges while we can. */
    for(i = 0; i < candidateEdgeSize; i++){
        int intersects = 0;
        candidateEdgeIndices[i][0] = toSort[i]->i0;
        candidateEdgeIndices[i][1] = toSort[i]->i1;
        candidateEdges[i][0] = toSort[i]->x0; candidateEdges[i][1] = toSort[i]->y0;
        candidateEdges[i][2] = toSort[i]->x1; candidateEdges[i][3] = toSort[i]->y1;
        for(j = 0; j < stackCount; j++){
            if(lineSegmentsIntersect(   candidateEdges[i][0], candidateEdges[i][1], 
                                        candidateEdges[i][2], candidateEdges[i][3],
                                        edges[j][0], edges[j][1],
                                        edges[j][2], edges[j][3] )){
                intersects = 1;
                break;
            }
        }
        if(intersects == 0){
            edges[stackCount] = malloc(sizeof(double)*4);
            edges[stackCount][0] = candidateEdges[i][0]; edges[stackCount][1] = candidateEdges[i][1];
            edges[stackCount][2] = candidateEdges[i][2]; edges[stackCount][3] = candidateEdges[i][3];

            addToTriangulationGraph(tri, candidateEdgeIndices[i][0], candidateEdgeIndices[i][1]);

            considerStack[stackCount] = malloc(sizeof(int)*2);
            considerStack[stackCount][0] = candidateEdgeIndices[i][0];
            considerStack[stackCount][1] = candidateEdgeIndices[i][1];
            stackCount++;                
        }
    }
    
    for(i = stackCount; i < stackCapacity; i++){
        considerStack[i] = malloc(sizeof(int)*2);
    }
    
    free(candidateEdges);
    free(candidateEdgeIndices);
    triSize = stackCount;

    /* Flip edges while they violate the Delaunay property; keep a stack of 
     * edges that we need to check. */
    while(stackCount > 0){
        int itemsToUpdate[4];
        int adjSize, *adj;
        int k0 = considerStack[stackCount-1][0], k1 = considerStack[stackCount-1][1];
        int bestGuessLeftIndex, bestGuessRightIndex;
        double bestGuessLeftDist, bestGuessRightDist;
        double a;
        stackCount--;
        
        if(!(inTriangulationGraph(tri,k0, k1))){
            continue;
        }
        

        /* compute the intersection of the adjacency lists of k0 and k1 */
        adj = malloc(sizeof(int)*(tri[k0][0]));
        for(i = 0; i < tri[k0][0]; i++){
            adj[i] = -1;
        }
        adjSize = 0;
        for(i = 0; i < tri[k0][0]; i++){
            int j, found = 0, search = tri[k0][i];
            for(j = 1; j < tri[k1][0]; j++){
                if(tri[k1][j] == search){
                    found = 1;
                    break;
                }
            }
            if(found){
                adj[adjSize] = search;
                adjSize++;
            }
        }
        
        bestGuessLeftIndex = bestGuessRightIndex = -1;
        bestGuessLeftDist = bestGuessRightDist = 1e20;

        a = getEuclideanDist(pointsX[k0], pointsY[k0], pointsX[k1], pointsY[k1]);
        /* go through the adjacency intersection, and find for points to the left
         * and right of the line, which ones minimize the distance to the line, which
         * we can find quickly by getting the area of the triangle formed */
        for(i = 0; i < adjSize; i++){
            int item = adj[i];
            /* don't duplicate code, just use common code
             * and merely add a level of indirection */
            int *toUpdateIndex = &bestGuessRightIndex;
            double *toUpdateDist = &bestGuessRightDist;
            double b, c, s, triangleAreaSquared;
            if(isLeftTurn(determineTurn(pointsX[k0], pointsY[k0], 
                                        pointsX[k1], pointsY[k1], 
                                        pointsX[item], pointsY[item]))){
                toUpdateIndex = &bestGuessLeftIndex;
                toUpdateDist = &bestGuessLeftDist;
            }
            b = getEuclideanDist(pointsX[k0], pointsY[k0],
                                 pointsX[item], pointsY[item]);
            c = getEuclideanDist(pointsX[k1], pointsY[k1],
                                 pointsX[item], pointsY[item]);
            s = (a+b+c)/2;
            triangleAreaSquared = s*(s-a)*(s-b)*(s-b);
            if(triangleAreaSquared < *toUpdateDist){
                *toUpdateIndex = item;
                *toUpdateDist = triangleAreaSquared;
            }
        }

        /* if the edge is on the convex hull, we won't necessarily have a point on 
         * both the left or right of the edge */
        if((bestGuessLeftIndex == -1) || (bestGuessRightIndex == -1)){
            continue;
        }

        /* if the present configuration works fine, don't do anything */
        if(!(isInCircle(doCircleTest(   pointsX[k0], pointsY[k0], pointsX[k1], pointsY[k1],
                                        pointsX[bestGuessLeftIndex], pointsY[bestGuessLeftIndex],
                                        pointsX[bestGuessRightIndex], pointsY[bestGuessRightIndex])))){
            continue;
        }

        /* ok, so the edge is bad, but can it be flipped? */
        if(!lineSegmentsIntersect(   pointsX[k0], pointsY[k0], pointsX[k1], pointsY[k1],
                                    pointsX[bestGuessLeftIndex], pointsY[bestGuessLeftIndex],
                                    pointsX[bestGuessRightIndex], pointsY[bestGuessRightIndex])){
            continue;
        }

        /* ok, so they can be flipped, and should be flipped. So do it. */
    
        removeFromTriangulationGraph(tri, k0, k1);
        addToTriangulationGraph(tri, bestGuessLeftIndex, bestGuessRightIndex);


        /* this isn't clean; however, it's cleaner than some sort of subroutine 
         * with 7 pointers */
        itemsToUpdate[0] = k0;
        itemsToUpdate[1] = k1;
        itemsToUpdate[2] = bestGuessLeftIndex;
        itemsToUpdate[3] = bestGuessRightIndex;
        /* now, update the stack */
        for(i = 0; i < 4; i++){
            int item = itemsToUpdate[i];
            int *vec = tri[item];
            for(j = 1; j < vec[0]; j++){
                int oldStackCapacity;
                if(vec[j] == -1){
                    continue;
                }
                if(stackCount == stackCapacity){
                    /* realloc time! */
                    oldStackCapacity = stackCapacity;
                    stackCapacity = stackCapacity * 3 / 2;
                    considerStack = realloc(considerStack, sizeof(int *)*stackCapacity);
                    for(k = oldStackCapacity; k < stackCapacity; k++){
                        considerStack[k] = malloc(sizeof(int)*2);
                    }
                }
                considerStack[stackCount][0] = item;
                considerStack[stackCount][1] = vec[j];
                stackCount++;
            }
        }
    }
    return tri;    
}


int main(int argc, char *argv[]){
    int pointCount, i; 
    double *pointsX, *pointsY;
    int **tri;

    if(argc != 2){
        fprintf(stderr,"%s pointCount\n", argv[0]);
        fprintf(stderr,"Input stream of the form: p1x p1y p2x p2y ... pnx pny\n"); 
        return 1;
    }
    pointCount = atoi(argv[1]);
    pointsX = malloc(sizeof(double)*pointCount);
    pointsY = malloc(sizeof(double)*pointCount);
    for(i = 0; i < pointCount; i++){
        int found;
        found = fscanf(stdin,"%lg",pointsX+i);
        if(found != 1){
            fprintf(stderr,"Something went wrong parsing the point input\n");
            return 1;
        }
        found = fscanf(stdin,"%lg",pointsY+i);
        if(found != 1){
            fprintf(stderr,"Something went wrong parsing the point input\n");
            return 1;
        }
    }

    tri = computeTriangulation(pointCount, pointsX, pointsY);
    /* output the triangulation */
    for(i = 0; i < pointCount; i++){
        int j;
        printf("%d ",i); 
        for(j = 1; j < tri[i][0]; j++){
            if(tri[i][j] != -1){
                printf("%d ", tri[i][j]); 
            }
        }
        printf("\n"); 
    }
    return 0;
}
