function [normalData, cache, detections] = run3DP(I, data, options, cache)
    %David Fouhey, Abhinav Gupta, Martial Hebert
    %Data-Driven 3D Primitives For Single Image Understanding 
    %ICCV 2013
    %
    %Inference-only code
    %
    %Inputs:
    %       I           -- an image
    %       programData -- program data structure, loaded via loadProgramData()
    %       options     -- an options structure controlling output. 
    %                      The options available are described below
    %       cache       -- a cache for dense transfer (optional). This prevents
    %                      the repeated loading of full-scene surface normals
    %
    %Outputs:
    %
    %       normalData  -- a structure containing the results. This is
    %                      described below
    %       cache       -- an updated cache structure
    %       detections  -- a matrix of the detections. Each row is of the form
    %                      [id,score,minX,minY,maxX,maxY]
    %                      where id is the id of the primitive, score is the
    %                      calibrated detection score, and minX, minY, maxX,
    %                      and maxY describe the bounding box of the detection.
    %
    %
    %normalData has some subset of the following fields, depending on the 
    %options used:
    %
    %  Sparse Prediction:
    %
    %   normalData.sparseMaps 
    %    -a N-entry cell array containing [HxWx3] normal maps
    %   normalData.sparseMasks
    %    -a N-entry cell array containing [HxW] masks indicating whether each
    %     pixel was predicted
    %
    %  Dense Prediction:
    %
    %   normalData.denseMaps
    %    -a N-entry cell array containing [HxWx3] normal maps
    %   normalData.denseConfidences
    %    -a N-entry cell array containing [HxW] confidence maps containing
    %     the sum of calibrated detection scores of the detections overlapping
    %     the pixel
    %
    %  Manhattan World Prediction:
    %
    %   normalData.rectifiedSparseMaps
    %    -the same as normalData.sparseMaps, but snapped to the estimated vps
    %   normalData.rectifiedDenseMaps
    %    -the same as normalData.denseMaps, but snapped to the estimated vps
    %   normalData.vp
    %    -the estimated vanishing point
    %   normalData.f
    %    -the estimated focal length
    %
    %
    %options takes the following options
    %
    %   The most important parameters are:
    %
    %   options.sparseThresholds (default [])
    %       - a Nx1 vector giving the sparse thresholds to use (in [0,1])
    %   options.denseThresholds (default [])
    %       - a Nx1 vector giving the dense thresholds to use
    %   options.makeRectifiedCopies (default false)
    %       - True/False: whether to estimate vps and return vanishing-point
    %         aligned normals
    %   options.verbosity
    %       - Whether to show status updates as the algorithm proceeds
    %   
    %   The remaining parameters control:
    %       
    %       The spatial prior on primitive transfer:
    %           (options.priorStrength, options.priorMinRank)
    %
    %       A threshold on overly large detections
    %           (options.maxPatchSizeFrac)
    %
    %       An option for transferring only consistent surface normals
    %       in a patch. This is turned off by default.
    %           (options.doMaskedTransfer, options.patchSelfConsistencyThresh)
    %           
    %       The Gaussian (akin to alpha blending) used for dense normal transfer
    %           (options.globalTransferBW)
    %
    %       The prior strength (how many calibrated detections it counts as)
    %           (options.priorStrength)
    if nargin < 3
        options = struct();
    end

    if nargin < 4
        cache = makeCache();
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%
    %% Input cleaning
    %%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if length(size(I)) == 1
        I = repmat(I,[1,1,3]);
    end 

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% 
    %% Option decoding
    %% 
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %decode the options

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% General control over output
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %what sparse thresholds to predict at
    if ~isfield(options,'sparseThresholds')
        options.sparseThresholds = [];
    end
   
    %the dense thresholds
    %slightly higher than the CV'd optimal ones on NYU, but
    %this is faster
    if ~isfield(options,'denseThresholds')
        options.denseThresholds = [0.65];
    end

    %whether to make copies that are rectified
    if ~isfield(options,'makeRectifiedCopies')
        options.makeRectifiedCopies = false;
    end

    if ~isfield(options,'focalLength')
        options.focalLength = -1;
    end

    %output verbosity
    if ~isfield(options,'verbosity')
        options.verbosity = 1;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% General algorithmic parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %the sigma of the GMM prior for weighting
    if ~isfield(options,'priorStrenth')
        options.priorStrength = 0.25;
    end

    %the minimum rank before rejecting a patch
    if ~isfield(options,'priorMinRank')
        options.priorMinRank = 0.5;
    end

    %threshold for rejecting large detections
    if ~isfield(options,'maxPatchSizeFrac')
        options.maxPatchSizeFrac = 0.4;
    end

    %whether to do masked transfer
    if ~isfield(options,'doMaskedTransfer')
        options.doMaskedTransfer = false;
    end

    %option for masked transfer. Determines whether a pixel is consisent 
    %enough; only transfer pixels with all instance angles within X radians 
    %of the median patch
    if ~isfield(options,'patchSelfConsistencyThresh')
        options.patchSelfConsistencyThresh = 0.5;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Dense transfer parameters
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    %the bandwidth of the transfer (i.e., how quickly influence fades from 
    %the detection), relative to image size
    if ~isfield(options,'globalTransferBW')
        options.globalTransferBW = 1.0 / 16;
    end

    %how many detections the prior counts for. Note that each patch
    %transfers five detections. The default prior is 1/2 a detection.
    if ~isfield(options,'priorStrength')
        options.priorStrength = 2.5;
    end

    %the normal data resize factor (you don't need full sized context images)
    %so you store them at 4x smaller so they fit into memory
    %Do not change this unless you change the normals you're using.
    if ~isfield(options,'normalResizeFactor')
        options.normalResizeFactor = 4;
    end

    %the source of the normals
    if ~isfield(options,'normalSource')
        options.normalSource = [getResourcePath() '/normalData/'];
    end


    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Get the detections
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    tTotal = tic;

    %get the detections
    detectionParams = struct('selectTopN', false, 'useDecisionThresh', true, 'overlap', 0.4, 'fixedDecisionThresh', -1.1);
    imageStruct = makeImageStruct(I);
    if options.verbosity > 0
        fprintf('Detecting\n');
    end
    tDetect = tic;
    detections = flattenDetections(rawDetectPresence(data.detectors, imageStruct, true, detectionParams));
    timeDetect = toc(tDetect);
    if options.verbosity > 0
        fprintf('Done; %.2f secs\n',timeDetect);
    end

    normalData.sparseMaps = cell(numel(options.sparseThresholds),1);
    normalData.sparseMasks = cell(numel(options.sparseThresholds),1);
    normalData.denseMaps = cell(numel(options.denseThresholds),1);
    normalData.denseConfidences = cell(numel(options.denseThresholds),1);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Get the sparse interpretations
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    if options.verbosity > 0 || numel(options.sparseThresholds)
        fprintf('Creating sparse interpretations\n');
    end
    tSparse = tic;
    [normalData.sparseMaps, normalData.sparseMasks] = getSparseResultsFromDetections(I, detections, data, options);
    timeSparse = toc(tSparse);
    if options.verbosity > 0 || numel(options.sparseThresholds)
        fprintf('Done: %.2f secs\n',timeSparse);
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Get the dense interpretations
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    if options.verbosity > 0 || numel(options.denseThresholds)
        fprintf('Creating dense interpretations\n');
    end
    tDense = tic;
    [normalData.denseMaps, normalData.denseConfidences, cache] = getDenseResultsFromDetections(I, detections, data, options, cache);
    timeDense = toc(tDense);
    if options.verbosity > 0 || numel(options.denseThresholds)
        fprintf('Done: %.2f secs\n',timeDense);
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Get rectified copies, if necessary
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if options.makeRectifiedCopies
        if options.verbosity > 0
            fprintf('Rectifying\nEstimating VPs\n');
        end
        tVPEst = tic;


        %which vp estimator to use
        if(1)
            [vp,f] = getVPHedau(I);
            vp = cat(2,vp{:});
        else
            [vp,f] = getVPLee(I);
        end

        if options.focalLength > 0
            %if we're using an overriding focal length
            f = options.focalLength;
        end

        normalData.vp = vp;
        normalData.f = f;
        
        timeVPEst = toc(tVPEst);
        if options.verbosity > 0
            fprintf('Done: %.2f secs\n',timeVPEst);
        end

        normalData.rectifiedSparseMaps = cell(size(normalData.sparseMaps));
        normalData.rectifiedDenseMaps = cell(size(normalData.denseMaps));


        fprintf('Rectifying\n');
        tRectify = tic;
        for i=1:numel(normalData.sparseMaps)
            normalData.rectifiedSparseMaps{i} = getRectified(vp, f, normalData.sparseMaps{i});
        end
        for i=1:numel(normalData.denseMaps)
            normalData.rectifiedDenseMaps{i} = getRectified(vp, f, normalData.denseMaps{i});
        end
        timeRectify = toc(tRectify);
        fprintf('Done: %.2f secs\n',timeRectify); 
    end

    timeTotal = toc(tTotal);
    if options.verbosity > 0
        fprintf('All done: %.2f secs\n',timeTotal);
    end

end 
