function [normalMaps, confidences, cache] = getDenseResultsFromDetections(I, detections, data, options, cache)
    %David Fouhey, Abhinav Gupta, Martial Hebert
    %Data-Driven 3D Primitives For Single Image Understanding 
    %ICCV 2013
    %
    %Inference-only code


    %Do the dense transfer, given an image I, detections, the program data, and program options.
    %Cache surface normals for images in cache.
    %
    %DO NOT CALL DIRECTLY -- call through run3DP


    h = size(I,1); w = size(I,2);

    threshes = options.denseThresholds;

    %compute the cutoffs for the various precisions
    cutoffs = zeros(numel(data.precisionVSVMScore),numel(threshes));
    for i=1:numel(data.precisionVSVMScore)
        for j=1:numel(threshes)
            cutoffs(i,j) = computePrecisionCutoff(data.precisionVSVMScore{i},threshes(j));
        end
    end

    %sort the detections
    [scoresSorted, scoreSort] = sort(detections(:,2));


    %for every threshold accumulate a normal map and confidence/normalizer
    normalMaps = {};
    confidences = {};
    for threshI=1:numel(threshes)


        toTransfer = zeros(size(scoreSort,1),1);
        priorRank = zeros(size(detections,1),1);

        for ii=1:numel(scoreSort)
            i = scoreSort(ii);
            %unpack the row to more sensible names
            clusterId = detections(i,1); score = detections(i,2);
            minX = detections(i,3); maxX = detections(i,5);
            minY = detections(i,4); maxY = detections(i,6);

            %check if we should let the detector fire at this threshold
            if(score < cutoffs(clusterId, threshI))
                continue;
            end

            xferPatchWidth = maxX  - minX; xferPatchHeight = maxY - minY;
            xferSize = [xferPatchHeight, xferPatchWidth];

            %reject big clusters
            if ((xferPatchWidth > w*options.maxPatchSizeFrac) || ...
                (xferPatchHeight > h*options.maxPatchSizeFrac))
                continue;
            end

            priorRankMap = computePriorRankImage(data.clusterData.priors{clusterId}, ...
                                                options.priorStrength, h, w);
            
            meanX = fix(mean([minX,maxX])); meanY = fix(mean([minY,maxY]));
            priorRank(i) = priorRankMap(meanY,meanX);

            if priorRankMap(meanY,meanX) < options.priorMinRank
                continue;
            end

            %keep track of how many images we have to transfer
            toTransfer(i) = numel(data.clusterData.bboxes{clusterId});

        end


        numToTransfer = sum(toTransfer);
        %the accumulated sum-of-normals
        transferAccums = zeros(h,w,3);
        %and the normalizer
        transferWeight = zeros(h,w,3);

        %inject the prior by just adding it with the specified strength
        for c=1:3
            transferAccums(:,:,c) = imresize_old(data.prior.normalPrior(:,:,c),[h,w]);
            transferWeight(:,:,c) = options.priorStrength;
        end

        transferred = 0;
        for i=1:numel(scoreSort)

            if(toTransfer(i) == 0)
                continue;
            end

            %unpack the detection result again
            clusterId = detections(i,1); scoreId = detections(i,2);
            dstMinX = detections(i,3); dstMaxX = detections(i,5);
            dstMinY = detections(i,4); dstMaxY = detections(i,6);

            wparams = data.weibullCalibrations.weibullParams{clusterId};
            calibratedScore = wblcdf(scoreId+wparams.bias,wparams.a,wparams.b);


            %for every instance, do the transfer
            for j=1:toTransfer(i)
                srcMinX = data.clusterData.bboxes{clusterId}{j}(1);
                srcMinY = data.clusterData.bboxes{clusterId}{j}(2);
                srcMaxX = data.clusterData.bboxes{clusterId}{j}(3);
                srcMaxY = data.clusterData.bboxes{clusterId}{j}(4);
                
                %load the normals for the primitive
                toLoadNormalName = [options.normalSource '/' strrep(data.clusterData.imageNames{clusterId}{j},'rgb','nm') '.mat'];
                if cache.normalTransferCache.isKey(toLoadNormalName)
                    if options.verbosity > 1
                        fprintf('Cache hit\n');
                    end
                    normalData = cache.normalTransferCache(toLoadNormalName);
                else
                    normalData = load(toLoadNormalName);
                    cache.normalTransferCache(toLoadNormalName) = normalData;
                end

                %the normals
                %note the double cast -- they might be stored as singles to save space
                normalConcat = {double(normalData.nx),double(normalData.ny),double(normalData.nz)};

                %compute how much to adjust the bandwidth of the negative exponential
                transferScale = (srcMaxX-srcMinX) / (dstMaxX - dstMinX);
                transferBW = options.globalTransferBW*w/transferScale;

                for c=1:3
                    %mask invalid values
                    normalConcat{c}(~normalData.depthValid) = NaN;

                    %figure out the warp parameters
                    %ax+b
                    if options.verbosity > 1
                        tWarp = tic;
                    end
                    xa = (dstMaxX - dstMinX) / (srcMaxX - srcMinX);
                    ya = (dstMaxY - dstMinY) / (srcMaxY - srcMinY);
                    xb = dstMaxX - xa*srcMaxX;
                    yb = dstMaxY - ya*srcMaxY;
                    %X source / X target
                    [XS,YS] = meshgrid(1:size(normalData.nx,2),1:size(normalData.nx,1));
                    [XT,YT] = meshgrid(1:w,1:h);
                    warpedNormals = interp2(XS*xa*options.normalResizeFactor+xb, ...
                                            YS*ya*options.normalResizeFactor+yb, ...
                                            normalConcat{c}, XT, YT);
                    if options.verbosity > 1
                        toc(tWarp);
                    end
                    %keep track of which ones come out as NaN
                    validNormalMask = ~isnan(warpedNormals);
                    %but set them to 0
                    warpedNormals(~validNormalMask) = 0;

                    %make a weight image
                    weightImage = zeros(h,w);
                    weightImage(dstMinY:dstMaxY,dstMinX:dstMaxX) = 1;
                    %do a distance transform
                    weightImage = bwdist(weightImage);
                    weightImage = exp(-((weightImage.^2) / (transferBW.^2)));

                    weightImage(dstMinY:dstMaxY,dstMinX:dstMaxX) = weightImage(dstMinY:dstMaxY,dstMinX:dstMaxX) * 2;

                    %but give 0 credence to invalid normals
                    weightImage = weightImage .* validNormalMask;

                    weightImage = weightImage .* calibratedScore;
                    weightImage = weightImage .* ((priorRank(i) - options.priorMinRank)/(1.0 - options.priorMinRank));

                    transferAccums(:,:,c) = transferAccums(:,:,c) + weightImage .* warpedNormals;
                    transferWeight(:,:,c) = transferWeight(:,:,c) + weightImage;

                end
            end
            transferred = transferred + toTransfer(i);

        end

        %finally normalize everything properly
        normalMap = transferAccums ./ (transferWeight + eps);
        normalMapDiv = sum(normalMap.^2,3).^0.5;
        normalMap = normalMap ./ repmat(normalMapDiv + eps, [1, 1, 3]);

        %store it
        normalMaps{threshI} = normalMap;
        confidences{threshI} = transferWeight(:,:,1);

    end
end
