//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: Real.h
//    class Real
//

#ifndef _REAL_H
#define _REAL_H

#include "BigFloat.h"

//  struct RealBase
//    used to distinguish Real from its descendents

struct RealBase
{
  RealBase(int = 0) {}
};

//  forward reference

class RealInt;
class RealLong;
class RealDouble;
class RealBigInt;
class RealBigFloat;
class RealRational;

//  class Real
//    envelope

class Real
{
  Real*    rep;
  
  unsigned refCount;
  
protected:
  
  //  constructor
  //    used for descendents
  
  Real(RealBase);
  
  //  error detection
  
  void error(const char*) const;
  
public:
  
  //  constructors
  
  Real();
  Real(int);
  Real(long);
  Real(double);
  Real(BigInt);
  Real(BigFloat);
  Real(Rational);
  
  Real(const Real&);
  
  //  the destructor
  
  virtual ~Real();
  
  //  assignment
  
  Real& operator =(const Real&);
  
  //  approximation
  
  virtual Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  virtual Real operator -() const;
  
  //  addition
  
  virtual Real operator +(const Real&) const;
  virtual Real addInt(const RealInt&) const;
  virtual Real addLong(const RealLong&) const;
  virtual Real addDouble(const RealDouble&) const;
  virtual Real addBigInt(const RealBigInt&) const;
  virtual Real addBigFloat(const RealBigFloat&) const;
  virtual Real addRational(const RealRational&) const;
  
  //  subtraction
  
  virtual Real operator -(const Real&) const;
  virtual Real subInt(const RealInt&) const;
  virtual Real subLong(const RealLong&) const;
  virtual Real subDouble(const RealDouble&) const;
  virtual Real subBigInt(const RealBigInt&) const;
  virtual Real subBigFloat(const RealBigFloat&) const;
  virtual Real subRational(const RealRational&) const;
  
  //  multiplication
  
  virtual Real operator *(const Real&) const;
  virtual Real mulInt(const RealInt&) const;
  virtual Real mulLong(const RealLong&) const;
  virtual Real mulDouble(const RealDouble&) const;
  virtual Real mulBigInt(const RealBigInt&) const;
  virtual Real mulBigFloat(const RealBigFloat&) const;
  virtual Real mulRational(const RealRational&) const;
  
  //  division
  
  virtual Real operator /(const Real&) const;
  virtual Real div(const Real&, const extULong&) const;
  virtual Real divInt(const RealInt&, const extULong&) const;
  virtual Real divLong(const RealLong&, const extULong&) const;
  virtual Real divDouble(const RealDouble&, const extULong&) const;
  virtual Real divBigInt(const RealBigInt&, const extULong&) const;
  virtual Real divBigFloat(const RealBigFloat&, const extULong&) const;
  virtual Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  virtual Real sqrt(const extLong&) const;
  friend Real sqrt(const Real&);
  
  //  equality
  
  virtual int operator ==(const Real&) const;
  virtual int eqlInt(const RealInt&) const;
  virtual int eqlLong(const RealLong&) const;
  virtual int eqlDouble(const RealDouble&) const;
  virtual int eqlBigInt(const RealBigInt&) const;
  virtual int eqlBigFloat(const RealBigFloat&) const;
  virtual int eqlRational(const RealRational&) const;
  
  //  anti-equality
  
  int operator !=(const Real&) const;
  
  //  smaller than
  
  virtual int operator <(const Real&) const;
  virtual int grtInt(const RealInt&) const;
  virtual int grtLong(const RealLong&) const;
  virtual int grtDouble(const RealDouble&) const;
  virtual int grtBigInt(const RealBigInt&) const;
  virtual int grtBigFloat(const RealBigFloat&) const;
  virtual int grtRational(const RealRational&) const;
  
  //  other inequalities
  
  int operator <=(const Real&) const;
  int operator  >(const Real&) const;
  int operator >=(const Real&) const;
  
  //  arithmetic and assignment opeartors
  
  Real& operator +=(const Real&);
  Real& operator -=(const Real&);
  Real& operator *=(const Real&);
  Real& operator /=(const Real&);
  
  //  builtin functions
  
  virtual int     isExact() const;
  virtual extLong lMSB() const;
  virtual extLong MSB() const;
  virtual extLong flrLgErr() const;
  virtual extLong clLgErr() const;
  virtual int     isZeroIn() const;
  
  virtual unsigned long degree() const;
  virtual unsigned long length() const;
  virtual unsigned long height() const;
  
  virtual int sgn() const;
  friend int sgn(const Real&);
  
  //  sream
  
  virtual ostream& operator <<(ostream&) const;
  friend ostream& operator <<(ostream&, const Real&);
};

//  class RealInt
//    machine integer

class RealInt : public Real
{
friend class RealLong;
friend class RealDouble;
friend class RealBigInt;
friend class RealBigFloat;
friend class RealRational;
  
  int ker;
  
public:
  
  //  constructor
  
  RealInt(const int&);
  
  //  the destructor
  
  ~RealInt();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

//  class RealLong
//    machine long integer

class RealLong : public Real
{
friend class RealInt;
friend class RealDouble;
friend class RealBigInt;
friend class RealBigFloat;
friend class RealRational;
  
  long ker;
  
public:
  
  //  constructor
  
  RealLong(const long&);
  
  //  the destructor
  
  ~RealLong();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

//  class RealDouble
//    machine double precious floating point number

class RealDouble : public Real
{
friend class RealInt;
friend class RealLong;
friend class RealBigInt;
friend class RealBigFloat;
friend class RealRational;
  
  double ker;
  
public:
  
  //  constructor
  
  RealDouble(const double& d);
  
  //  the destructor
  
  ~RealDouble();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

//  class RealBigInt
//    arbitrary length integer

class RealBigInt : public Real
{
friend class RealInt;
friend class RealLong;
friend class RealDouble;
friend class RealBigFloat;
friend class RealRational;
  
  BigInt ker;
  
public:
  
  //  constructor
  
  RealBigInt(const BigInt& I);
  
  //  the destructor
  
  ~RealBigInt();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

//  class RealBigFloat
//    BigFloat

class RealBigFloat : public Real
{
friend class RealInt;
friend class RealLong;
friend class RealDouble;
friend class RealBigInt;
friend class RealRational;
  
  BigFloat ker;
  
public:
  
  //  constructor
  
  RealBigFloat(const BigFloat& B);
  
  //  the destructor
  
  ~RealBigFloat();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

//  class RealRational
//    rational number

class RealRational : public Real
{
friend class RealInt;
friend class RealLong;
friend class RealDouble;
friend class RealBigInt;
friend class RealBigFloat;
  
  Rational ker;
  
public:
  
  //  constructor
  
  RealRational(const Rational& R);
  
  //  the destructor
  
  ~RealRational();
  
  //  approximation
  
  Real approx(const extULong&, const extLong&) const;
  
  //  unary minus
  
  Real operator -() const;
  
  //  addition
  
  Real operator +(const Real&) const;
  Real addInt(const RealInt&) const;
  Real addLong(const RealLong&) const;
  Real addDouble(const RealDouble&) const;
  Real addBigInt(const RealBigInt&) const;
  Real addBigFloat(const RealBigFloat&) const;
  Real addRational(const RealRational&) const;
  
  //  subtraction
  
  Real operator -(const Real&) const;
  Real subInt(const RealInt&) const;
  Real subLong(const RealLong&) const;
  Real subDouble(const RealDouble&) const;
  Real subBigInt(const RealBigInt&) const;
  Real subBigFloat(const RealBigFloat&) const;
  Real subRational(const RealRational&) const;
  
  //  multiplication
  
  Real operator *(const Real&) const;
  Real mulInt(const RealInt&) const;
  Real mulLong(const RealLong&) const;
  Real mulDouble(const RealDouble&) const;
  Real mulBigInt(const RealBigInt&) const;
  Real mulBigFloat(const RealBigFloat&) const;
  Real mulRational(const RealRational&) const;
  
  //  division
  
  Real div(const Real&, const extULong&) const;
  Real divInt(const RealInt&, const extULong&) const;
  Real divLong(const RealLong&, const extULong&) const;
  Real divDouble(const RealDouble&, const extULong&) const;
  Real divBigInt(const RealBigInt&, const extULong&) const;
  Real divBigFloat(const RealBigFloat&, const extULong&) const;
  Real divRational(const RealRational&, const extULong&) const;
  
  //  squareroot
  
  Real sqrt(const extLong&) const;
  
  //  equality
  
  int operator ==(const Real&) const;
  int eqlInt(const RealInt&) const;
  int eqlLong(const RealLong&) const;
  int eqlDouble(const RealDouble&) const;
  int eqlBigInt(const RealBigInt&) const;
  int eqlBigFloat(const RealBigFloat&) const;
  int eqlRational(const RealRational&) const;
  
  //  smaller-than
  
  int operator <(const Real&) const;
  int grtInt(const RealInt&) const;
  int grtLong(const RealLong&) const;
  int grtDouble(const RealDouble&) const;
  int grtBigInt(const RealBigInt&) const;
  int grtBigFloat(const RealBigFloat&) const;
  int grtRational(const RealRational&) const;
  
  //  builtin functions
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  unsigned long degree() const;
  unsigned long length() const;
  unsigned long height() const;
  
  int sgn() const;
  
  //  stream
  
  ostream& operator <<(ostream&) const;
};

#endif

