//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: Real.cc
//    class Real
//

#include <iostream.h>
#include <limits.h>
#include "Real.h"

//  floor log base 2 of abs(x)
//  convention lg(0) = 0

extern long flrLg(long);
extern long flrLg(unsigned long);
extern long flrLg(const BigInt&);

//  infty and tiny

static long halfIntMax = INT_MAX /2;
static long halfIntMin = INT_MIN /2;

static long halfLongMax = LONG_MAX /2;
static long halfLongMin = LONG_MIN /2;

static const BigInt anonymBigInt(0);

//  class Real

//  constructors
  
Real :: Real(RealBase)
{
  rep      = this;
  refCount = 0;
}
  
Real :: Real() 
{
  rep = new RealInt(0);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(int i)
{
  rep = new RealInt(i);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(long l)
{
  rep = new RealLong(l);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(double d)
{
  rep = new RealDouble(d);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(BigInt I)
{
  rep = new RealBigInt(I);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(BigFloat B)
{
  rep = new RealBigFloat(B);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(Rational R)
{
  rep = new RealRational(R);
  rep->refCount++;
  refCount = 1;
}

Real :: Real(const Real& x)
{
  rep = x.rep;
  rep->refCount++;
  refCount = 1;
}
  
//  the destructor

Real :: ~Real()
{
  if (rep != this && --rep->refCount == 0)
    delete rep;
}
  
//  assignment

Real& Real :: operator =(const Real& x)
{
  if (--rep->refCount == 0)
    delete rep;
  
  rep = x.rep;
  rep->refCount++;
  
  return *this;
}
  
//  approximation

Real Real :: approx(const extULong& r, const extLong& a) const
{
  return rep->approx(r, a);
}

//  unary minus
  
Real Real :: operator -() const
{
  return rep->operator -();
}

//  addition

Real Real :: operator +(const Real& x) const
{
  return rep->operator +(x);
}

Real Real :: addInt(const RealInt& i) const
{
  return rep->addInt(i);
}

Real Real :: addLong(const RealLong& l) const
{
  return rep->addLong(l);
}

Real Real :: addDouble(const RealDouble& d) const
{
  return rep->addDouble(d);
}

Real Real :: addBigInt(const RealBigInt& I) const
{
  return rep->addBigInt(I);
}

Real Real :: addBigFloat(const RealBigFloat& B) const
{
  return rep->addBigFloat(B);
}

Real Real :: addRational(const RealRational& R) const
{
  return rep->addRational(R);
}

//  subtraction

Real Real :: operator -(const Real& x) const
{
  return rep->operator -(x);
}

Real Real :: subInt(const RealInt& i) const
{
  return rep->subInt(i);
}

Real Real :: subLong(const RealLong& l) const
{
  return rep->subLong(l);
}

Real Real :: subDouble(const RealDouble& d) const
{
  return rep->subDouble(d);
}

Real Real :: subBigInt(const RealBigInt& I) const
{
  return rep->subBigInt(I);
}

Real Real :: subBigFloat(const RealBigFloat& B) const
{
  return rep->subBigFloat(B);
}

Real Real :: subRational(const RealRational& R) const
{
  return rep->subRational(R);
}

//  multiplication

Real Real :: operator *(const Real& x) const
{
  return rep->operator *(x);
}

Real Real :: mulInt(const RealInt& i) const
{
  return rep->mulInt(i);
}

Real Real :: mulLong(const RealLong& l) const
{
  return rep->mulLong(l);
}

Real Real :: mulDouble(const RealDouble& d) const
{
  return rep->mulDouble(d);
}

Real Real :: mulBigInt(const RealBigInt& I) const
{
  return rep->mulBigInt(I);
}

Real Real :: mulBigFloat(const RealBigFloat& B) const
{
  return rep->mulBigFloat(B);
}

Real Real :: mulRational(const RealRational& R) const
{
  return rep->mulRational(R);
}

//  division

Real Real :: operator /(const Real& x) const
{
  return rep->div(x, defRelPrec);
}

Real Real :: div(const Real& x, const extULong& r) const
{
  return rep->div(x, r);
}

Real Real :: divInt(const RealInt& i, const extULong& r) const
{
  return rep->divInt(i, r);
}

Real Real :: divLong(const RealLong& l, const extULong& r) const
{
  return rep->divLong(l, r);
}

Real Real :: divDouble(const RealDouble& d, const extULong& r) const
{
  return rep->divDouble(d, r);
}

Real Real :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return rep->divBigInt(I, r);
}

Real
Real :: divBigFloat(const RealBigFloat& B, const extULong& r) const
{
  return rep->divBigFloat(B, r);
}

Real Real :: divRational(const RealRational& R, const extULong& r) const
{
  return rep->divRational(R, r);
}

//  squareroot

Real Real :: sqrt(const extLong& a) const
{
  return rep->sqrt(a);
}

Real sqrt(const Real& x)
{
  return x.sqrt(defAbsPrec);
}

//  equality

int Real :: operator ==(const Real& x) const
{
  return rep->operator ==(x);
}

int Real :: eqlInt(const RealInt& i) const
{
  return rep->eqlInt(i);
}

int Real :: eqlLong(const RealLong& l) const
{
  return rep->eqlLong(l);
}

int Real :: eqlDouble(const RealDouble& d) const
{
  return rep->eqlDouble(d);
}

int Real :: eqlBigInt(const RealBigInt& I) const
{
  return rep->eqlBigInt(I);
}

int Real :: eqlBigFloat(const RealBigFloat& B) const
{
  return rep->eqlBigFloat(B);
}

int Real :: eqlRational(const RealRational& R) const
{
  return rep->eqlRational(R);
}

//  anti-equality

int Real :: operator !=(const Real& x) const
{
  return !rep->operator ==(x);
}

//  smaller than

int Real :: operator <(const Real& x) const
{
  return rep->operator <(x);
}

int Real :: grtInt(const RealInt& i) const
{
  return rep->grtInt(i);
}

int Real :: grtLong(const RealLong& l) const
{
  return rep->grtLong(l);
}

int Real :: grtDouble(const RealDouble& d) const
{
  return rep->grtDouble(d);
}

int Real :: grtBigInt(const RealBigInt& I) const
{
  return rep->grtBigInt(I);
}

int Real :: grtBigFloat(const RealBigFloat& B) const
{
  return rep->grtBigFloat(B);
}

int Real :: grtRational(const RealRational& R) const
{
  return rep->grtRational(R);
}

//  other inequalities

int Real :: operator <=(const Real& x) const
{
  return rep->operator <(x) || rep->operator ==(x);
}

int Real :: operator  >(const Real& x) const
{
  return !rep->operator <(x) && !rep->operator ==(x);
}

int Real :: operator >=(const Real& x) const
{
  return !rep->operator <(x);
}

//  arithmetic and assignment opeartors

Real& Real :: operator +=(const Real& x)
{
  Real t = rep->operator +(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

Real& Real :: operator -=(const Real& x)
{
  Real t = rep->operator -(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

Real& Real :: operator *=(const Real& x)
{
  Real t = rep->operator *(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

Real& Real :: operator /=(const Real& x)
{
  Real t = rep->div(x, defRelPrec);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

//  builtin functions

int Real :: isExact() const
{
  return rep->isExact();
}

extLong Real :: lMSB() const
{
  return rep->lMSB();
}

extLong Real :: MSB() const
{
  return rep->MSB();
}

extLong Real :: flrLgErr() const
{
  return rep->flrLgErr();
}

extLong Real :: clLgErr() const
{
  return rep->clLgErr();
}

int Real :: isZeroIn() const
{
  return rep->isZeroIn();
}

unsigned long Real :: degree() const
{
  return rep->degree();
}

unsigned long Real :: length() const
{
  return rep->length();
}

unsigned long Real :: height() const
{
  return rep->height();
}

int Real :: sgn() const
{
  return rep->sgn();
}

int sgn(const Real& x)
{
  return x.sgn();
}

//  stream

ostream& Real :: operator <<(ostream& o) const
{
  rep->operator <<(o);
  
  return o;
}

ostream& operator <<(ostream& o, const Real& x)
{
  x.operator <<(o);
  
  return o;
}

//  error detection

void Real :: error(const char* msg) const
{
  //  output error message to stderr
  
  cerr << "Real Error : " << msg << endl;
  
  //  unusual termination
  
  exit(1);
}

//  class RealInt

//  constructor

RealInt :: RealInt(const int& i) : Real(RealBase()), ker(i)
{}

//  the destructor

RealInt :: ~RealInt()
{}
  
//  approximation

Real RealInt :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(BigInt(ker), r, a);
  
  return Real(x);
}

//  unary minus
  
Real RealInt :: operator -() const
{
  if (ker < - INT_MAX)
    return Real(- long(ker));
  else
    return Real(- ker);
}

//  addition

Real RealInt :: operator +(const Real& x) const
{
  return x.addInt(*this);
}

Real RealInt :: addInt(const RealInt& i) const
{
  if (i.ker > halfIntMax && ker > halfIntMax
      ||
      i.ker < halfIntMin && ker < halfIntMin)
    return Real(BigInt(i.ker) + BigInt(ker));
  else
    return Real(i.ker + ker);
}

Real RealInt :: addLong(const RealLong& l) const
{
  if (l.ker > halfLongMax && long(ker) > halfLongMax
      ||
      l.ker < halfLongMin && long(ker) < halfLongMin)
    return Real(BigInt(l.ker) + BigInt(ker));
  else
    return Real(l.ker + long(ker));
}

Real RealInt :: addDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator +(BigFloat(ker)));
}

Real RealInt :: addBigInt(const RealBigInt& I) const
{
  return Real(I.ker + BigInt(ker));
}

Real RealInt :: addBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator +(BigFloat(ker)));
}

Real RealInt :: addRational(const RealRational& R) const
{
  return Real(R.ker + Rational(ker));
}

//  subtraction

Real RealInt :: operator -(const Real& x) const
{
  return x.subInt(*this);
}

Real RealInt :: subInt(const RealInt& i) const
{
  if (i.ker < halfIntMin && ker > halfIntMax
      ||
      i.ker > halfIntMax && ker < halfIntMin)
    return Real(BigInt(i.ker) - BigInt(ker));
  else
    return Real(i.ker - ker);
}

Real RealInt :: subLong(const RealLong& l) const
{
  if (l.ker < halfLongMin && long(ker) > halfLongMax
      ||
      l.ker > halfLongMax && long(ker) < halfLongMin)
    return Real(BigInt(l.ker) - BigInt(ker));
  else
    return Real(l.ker - long(ker));
}

Real RealInt :: subDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator -(BigFloat(ker)));
}

Real RealInt :: subBigInt(const RealBigInt& I) const
{
  return Real(I.ker - BigInt(ker));
}

Real RealInt :: subBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator -(BigFloat(ker)));
}

Real RealInt :: subRational(const RealRational& R) const
{
  return Real(R.ker - Rational(ker));
}

//  multiplication

Real RealInt :: operator *(const Real& x) const
{
  return x.mulInt(*this);
}

Real RealInt :: mulInt(const RealInt& i) const
{
  if (flrLg(long(i.ker)) + flrLg(long(ker)) < WORD_BIT - 2)
    return Real(i.ker * ker);
  else
    return Real(BigInt(i.ker) * BigInt(ker));
}

Real RealInt :: mulLong(const RealLong& l) const
{
  if (flrLg(l.ker) + flrLg(long(ker)) < LONG_BIT - 2)
    return Real(l.ker * long(ker));
  else
    return Real(BigInt(l.ker) * BigInt(ker));
}

Real RealInt :: mulDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator *(BigFloat(ker)));
}

Real RealInt :: mulBigInt(const RealBigInt& I) const
{
  return Real(I.ker * BigInt(ker));
}

Real RealInt :: mulBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator *(BigFloat(ker)));
}

Real RealInt :: mulRational(const RealRational& R) const
{
  return Real(Rational(R.ker.numerator() * ker, R.ker.denominator()));
}

//  division

Real RealInt :: div(const Real& x, const extULong& r) const
{
  return x.divInt(*this, r);
}

Real RealInt :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(Rational(i.ker, ker));
}

Real RealInt :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(Rational(l.ker, BigInt(ker)));
}

Real RealInt :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(BigFloat(d.ker).div(BigFloat(ker), r));
}

Real RealInt :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(Rational(I.ker, BigInt(ker)));
}

Real RealInt :: divBigFloat(const RealBigFloat& B, const extULong& r) const
{
  return Real(B.ker.div(BigFloat(ker), r));
}

Real RealInt :: divRational(const RealRational& R, const extULong& r) const
{
  return Real(Rational(R.ker.numerator(), R.ker.denominator() * ker));
}

//  squareroot

Real RealInt :: sqrt(const extLong& a) const
{
  return Real(BigFloat(ker).sqrt(a));
}

//  equality

int RealInt :: operator ==(const Real& x) const
{
  return x.eqlInt(*this);
}

int RealInt :: eqlInt(const RealInt& i) const
{
  return i.ker == ker;
}

int RealInt :: eqlLong(const RealLong& l) const
{
  return l.ker == long(ker);
}

int RealInt :: eqlDouble(const RealDouble& d) const
{
  return d.ker == double(ker);
}

int RealInt :: eqlBigInt(const RealBigInt& I) const
{
  return I.ker == BigInt(ker);
}

int RealInt :: eqlBigFloat(const RealBigFloat& B) const
{
  return B.ker == BigFloat(ker);
}

int RealInt :: eqlRational(const RealRational& R) const
{
  return R.ker == Rational(ker);
}

//  smaller-than

int RealInt :: operator <(const Real& x) const
{
  return x.grtInt(*this);
}

int RealInt :: grtInt(const RealInt& i) const
{
  return i.ker < ker;
}

int RealInt :: grtLong(const RealLong& l) const
{
  return l.ker < long(ker);
}

int RealInt :: grtDouble(const RealDouble& d) const
{
  return d.ker < double(ker);
}

int RealInt :: grtBigInt(const RealBigInt& I) const
{
  return I.ker < BigInt(ker);
}

int RealInt :: grtBigFloat(const RealBigFloat& B) const
{
  return B.ker < BigFloat(ker);
}

int RealInt :: grtRational(const RealRational& R) const
{
  return R.ker < Rational(ker);
}

//  builtin functions

int RealInt :: isExact() const
{
  return 1;
}

extLong RealInt :: lMSB() const
{
  if (ker)
    return extLong(flrLg(long(ker)));
  else
    return extLong(tinyLong);
}

extLong RealInt :: MSB() const
{
  if (ker)
    return extLong(flrLg(long(ker)));
  else
    return extLong(tinyLong);
}

extLong RealInt :: flrLgErr() const
{
  return extLong(tinyLong);
}

extLong RealInt :: clLgErr() const
{
  return extLong(tinyLong);
}

int RealInt :: isZeroIn() const
{
  return ker == 0;
}

unsigned long RealInt :: degree() const
{
  return 1;
}

unsigned long RealInt :: length() const
{
  long l = flrLg(long(ker));
  
  return l ? l + 1 : 2;
}

unsigned long RealInt :: height() const
{
  long l = flrLg(long(ker));
  
  return l ? l + 1 : 2;
}

int RealInt :: sgn() const
{
  return ker > 0 ? 1 : !ker ? 0 : - 1;
}

//  stream

ostream& RealInt :: operator <<(ostream& o) const
{
  o << "Real(int " << ker << ')';
  
  return o;
}

//  class RealLong

//  constructor

RealLong :: RealLong(const long& l) : Real(RealBase()), ker(l)
{}

//  the destructor

RealLong :: ~RealLong()
{}
  
//  approximation

Real RealLong :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(BigInt(ker), r, a);
  
  return Real(x);
}

//  unary minus
  
Real RealLong :: operator -() const
{
  if (ker < - LONG_MAX)
    return Real(- BigInt(ker));
  else
    return Real(- ker);
}

//  addition

Real RealLong :: operator +(const Real& x) const
{
  return x.addLong(*this);
}

Real RealLong :: addInt(const RealInt& i) const
{
  if (long(i.ker) > halfLongMax && ker > halfLongMax
      ||
      long(i.ker) < halfLongMin && ker < halfLongMin)
    return Real(BigInt(i.ker) + BigInt(ker));
  else
    return Real(long(i.ker) + ker);
}

Real RealLong :: addLong(const RealLong& l) const
{
  if (l.ker > halfLongMax && ker > halfLongMax
      ||
      l.ker < halfLongMin && ker < halfLongMin)
    return Real(BigInt(l.ker) + BigInt(ker));
  else
    return Real(l.ker + ker);
}

Real RealLong :: addDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator +(BigFloat(ker)));
}

Real RealLong :: addBigInt(const RealBigInt& I) const
{
  return Real(I.ker + BigInt(ker));
}

Real RealLong :: addBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator +(BigFloat(ker)));
}

Real RealLong :: addRational(const RealRational& R) const
{
  return Real(R.ker + Rational(ker));
}

//  subtraction

Real RealLong :: operator -(const Real& x) const
{
  return x.subLong(*this);
}

Real RealLong :: subInt(const RealInt& i) const
{
  if (long(i.ker) < halfLongMin && ker > halfLongMax
      ||
      long(i.ker) > halfLongMax && ker < halfLongMin)
    return Real(BigInt(i.ker) - BigInt(ker));
  else
    return Real(long(i.ker) - ker);
}

Real RealLong :: subLong(const RealLong& l) const
{
  if (l.ker < halfLongMin && ker > halfLongMax
      ||
      l.ker > halfLongMax && ker < halfLongMin)
    return Real(BigInt(l.ker) - BigInt(ker));
  else
    return Real(l.ker - ker);
}

Real RealLong :: subDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator -(BigFloat(ker)));
}

Real RealLong :: subBigInt(const RealBigInt& I) const
{
  return Real(I.ker - BigInt(ker));
}

Real RealLong :: subBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator -(BigFloat(ker)));
}

Real RealLong :: subRational(const RealRational& R) const
{
  return Real(R.ker - Rational(ker));
}

//  multiplication

Real RealLong :: operator *(const Real& x) const
{
  return x.mulLong(*this);
}

Real RealLong :: mulInt(const RealInt& i) const
{
  if (flrLg(long(i.ker)) + flrLg(ker) < LONG_BIT - 2)
    return Real(long(i.ker) * ker);
  else
    return Real(BigInt(i.ker) * BigInt(ker));
}

Real RealLong :: mulLong(const RealLong& l) const
{
  if (flrLg(l.ker) + flrLg(ker) < LONG_BIT - 2)
    return Real(l.ker * ker);
  else
    return Real(BigInt(l.ker) * BigInt(ker));
}

Real RealLong :: mulDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator *(BigFloat(ker)));
}

Real RealLong :: mulBigInt(const RealBigInt& I) const
{
  return Real(I.ker * BigInt(ker));
}

Real RealLong :: mulBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator *(BigFloat(ker)));
}

Real RealLong :: mulRational(const RealRational& R) const
{
  return Real(Rational(R.ker.numerator() * ker, R.ker.denominator()));
}

//  division

Real RealLong :: div(const Real& x, const extULong& r) const
{
  return x.divLong(*this, r);
}

Real RealLong :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(Rational(BigInt(i.ker), ker));
}

Real RealLong :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(Rational(l.ker, ker));
}

Real RealLong :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(BigFloat(d.ker).div(BigFloat(ker), r));
}

Real RealLong :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(Rational(I.ker, BigInt(ker)));
}

Real RealLong :: divBigFloat(const RealBigFloat& B, const extULong& r) const
{
  return Real(B.ker.div(BigFloat(ker), r));
}

Real RealLong :: divRational(const RealRational& R, const extULong& r) const
{
  return Real(Rational(R.ker.numerator(), R.ker.denominator() * ker));
}

//  squareroot

Real RealLong :: sqrt(const extLong& a) const
{
  return Real(BigFloat(ker).sqrt(a));
}

//  equality

int RealLong :: operator ==(const Real& x) const
{
  return x.eqlLong(*this);
}

int RealLong :: eqlInt(const RealInt& i) const
{
  return long(i.ker) == ker;
}

int RealLong :: eqlLong(const RealLong& l) const
{
  return l.ker == ker;
}

int RealLong :: eqlDouble(const RealDouble& d) const
{
  return d.ker == double(ker);
}

int RealLong :: eqlBigInt(const RealBigInt& I) const
{
  return I.ker == BigInt(ker);
}

int RealLong :: eqlBigFloat(const RealBigFloat& B) const
{
  return B.ker == BigFloat(ker);
}

int RealLong :: eqlRational(const RealRational& R) const
{
  return R.ker == Rational(ker);
}

//  smaller-than

int RealLong :: operator <(const Real& x) const
{
  return x.grtLong(*this);
}

int RealLong :: grtInt(const RealInt& i) const
{
  return long(i.ker) < ker;
}

int RealLong :: grtLong(const RealLong& l) const
{
  return l.ker < ker;
}

int RealLong :: grtDouble(const RealDouble& d) const
{
  return d.ker < double(ker);
}

int RealLong :: grtBigInt(const RealBigInt& I) const
{
  return I.ker < BigInt(ker);
}

int RealLong :: grtBigFloat(const RealBigFloat& B) const
{
  return B.ker < BigFloat(ker);
}

int RealLong :: grtRational(const RealRational& R) const
{
  return R.ker < Rational(ker);
}

//  builtin functions

int RealLong :: isExact() const
{
  return 1;
}

extLong RealLong :: lMSB() const
{
  if (ker)
    return extLong(flrLg(ker));
  else
    return extLong(tinyLong);
}

extLong RealLong :: MSB() const
{
  if (ker)
    return extLong(flrLg(ker));
  else
    return extLong(tinyLong);
}

extLong RealLong :: flrLgErr() const
{
  return extLong(tinyLong);
}

extLong RealLong :: clLgErr() const
{
  return extLong(tinyLong);
}

int RealLong :: isZeroIn() const
{
  return ker == 0;
}

unsigned long RealLong :: degree() const
{
  return 1;
}

unsigned long RealLong :: length() const
{
  long l = flrLg(ker);
  
  return l ? l + 1 : 2;
}

unsigned long RealLong :: height() const
{
  long l = flrLg(ker);
  
  return l ? l + 1 : 2;
}

int RealLong :: sgn() const
{
  return ker > 0 ? 1 : !ker ? 0 : - 1;
}

//  stream

ostream& RealLong :: operator <<(ostream& o) const
{
  o << "Real(long " << ker << ')';
  
  return o;
}

//  class RealDouble

//  constructor

RealDouble :: RealDouble(const double& d) : Real(RealBase()), ker(d)
{}

//  the destructor

RealDouble :: ~RealDouble()
{}
  
//  approximation

Real RealDouble :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(Rational(ker), r, a);
  
  return Real(x);
}

//  unary minus
  
Real RealDouble :: operator -() const
{
  return Real(- ker);
}

//  addition

Real RealDouble :: operator +(const Real& x) const
{
  return x.addDouble(*this);
}

Real RealDouble :: addInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator +(BigFloat(ker)));
}

Real RealDouble :: addLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator +(BigFloat(ker)));
}

Real RealDouble :: addDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator +(BigFloat(ker)));
}

Real RealDouble :: addBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator +(BigFloat(ker)));
}

Real RealDouble :: addBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator +(BigFloat(ker)));
}

Real RealDouble :: addRational(const RealRational& R) const
{
  return Real(R.ker + Rational(ker));
}

//  subtraction

Real RealDouble :: operator -(const Real& x) const
{
  return x.subDouble(*this);
}

Real RealDouble :: subInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator -(BigFloat(ker)));
}

Real RealDouble :: subLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator -(BigFloat(ker)));
}

Real RealDouble :: subDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator -(BigFloat(ker)));
}

Real RealDouble :: subBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator -(BigFloat(ker)));
}

Real RealDouble :: subBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator -(BigFloat(ker)));
}

Real RealDouble :: subRational(const RealRational& R) const
{
  return Real(R.ker - Rational(ker));
}

//  multiplication

Real RealDouble :: operator *(const Real& x) const
{
  return x.mulDouble(*this);
}

Real RealDouble :: mulInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator *(BigFloat(ker)));
}

Real RealDouble :: mulLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator *(BigFloat(ker)));
}

Real RealDouble :: mulDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator *(BigFloat(ker)));
}

Real RealDouble :: mulBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator *(BigFloat(ker)));
}

Real RealDouble :: mulBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator *(BigFloat(ker)));
}

Real RealDouble :: mulRational(const RealRational& R) const
{
  return Real(R.ker * Rational(ker));
}

//  division

Real RealDouble :: div(const Real& x, const extULong& r) const
{
  return x.divDouble(*this, r);
}

Real RealDouble :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(BigFloat(i.ker).div(BigFloat(ker), r));
}

Real RealDouble :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(BigFloat(l.ker).div(BigFloat(ker), r));
}

Real RealDouble :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(BigFloat(d.ker).div(BigFloat(ker), r));
}

Real RealDouble :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(BigFloat(I.ker).div(BigFloat(ker), r));
}

Real RealDouble :: divBigFloat(const RealBigFloat& B, const extULong& r) const
{
  return Real(B.ker.div(BigFloat(ker), r));
}

Real RealDouble :: divRational(const RealRational& R, const extULong& r) const
{
  return Real(R.ker / Rational(ker));
}

//  squareroot

Real RealDouble :: sqrt(const extLong& a) const
{
  return Real(BigFloat(ker).sqrt(a));
}

//  equality

int RealDouble :: operator ==(const Real& x) const
{
  return x.eqlDouble(*this);
}

int RealDouble :: eqlInt(const RealInt& i) const
{
  return double(i.ker) == ker;
}

int RealDouble :: eqlLong(const RealLong& l) const
{
  return double(l.ker) == ker;
}

int RealDouble :: eqlDouble(const RealDouble& d) const
{
  return d.ker == ker;
}

int RealDouble :: eqlBigInt(const RealBigInt& I) const
{
  return BigFloat(I.ker) == BigFloat(ker);
}

int RealDouble :: eqlBigFloat(const RealBigFloat& B) const
{
  return B.ker == BigFloat(ker);
}

int RealDouble :: eqlRational(const RealRational& R) const
{
  return R.ker == Rational(ker);
}

//  smaller-than

int RealDouble :: operator <(const Real& x) const
{
  return x.grtDouble(*this);
}

int RealDouble :: grtInt(const RealInt& i) const
{
  return double(i.ker) < ker;
}

int RealDouble :: grtLong(const RealLong& l) const
{
  return double(l.ker) < ker;
}

int RealDouble :: grtDouble(const RealDouble& d) const
{
  return d.ker < ker;
}

int RealDouble :: grtBigInt(const RealBigInt& I) const
{
  return BigFloat(I.ker) < BigFloat(ker);
}

int RealDouble :: grtBigFloat(const RealBigFloat& B) const
{
  return B.ker < BigFloat(ker);
}

int RealDouble :: grtRational(const RealRational& R) const
{
  return R.ker < Rational(ker);
}

//  builtin functions

int RealDouble :: isExact() const
{
  return 1;
}

extLong RealDouble :: lMSB() const
{
  return extLong(BigFloat(ker).MSB());
}

extLong RealDouble :: MSB() const
{
  return extLong(BigFloat(ker).MSB());
}

extLong RealDouble :: flrLgErr() const
{
  return extLong(tinyLong);
}

extLong RealDouble :: clLgErr() const
{
  return extLong(tinyLong);
}

int RealDouble :: isZeroIn() const
{
  return ker == 0.0;
}

unsigned long RealDouble :: degree() const
{
  return 1;
}

unsigned long RealDouble :: length() const
{
  Rational R  = Rational(ker);
  long     ln = lg(R.numerator());
  long     ld = lg(R.denominator());
  long     l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

unsigned long RealDouble :: height() const
{
  Rational R  = Rational(ker);
  long     ln = lg(R.numerator());
  long     ld = lg(R.denominator());
  long     l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

int RealDouble :: sgn() const
{
  return ker > 0.0 ? 1 : ker == 0.0 ? 0 : - 1;
}

//  stream

ostream& RealDouble :: operator <<(ostream& o) const
{
  o << "Real(double " << ker << ')';
  
  return o;
}

//  class RealBigInt

//  constructor

RealBigInt :: RealBigInt(const BigInt& I) : Real(RealBase()), ker(I)
{}

//  the destructor

RealBigInt :: ~RealBigInt()
{}
  
//  approximation

Real RealBigInt :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(ker, r, a);
  
  return Real(x);
}

//  unary minus
  
Real RealBigInt :: operator -() const
{
  return Real(- ker);
}

//  addition

Real RealBigInt :: operator +(const Real& x) const
{
  return x.addBigInt(*this);
}

Real RealBigInt :: addInt(const RealInt& i) const
{
  return Real(BigInt(i.ker) + ker);
}

Real RealBigInt :: addLong(const RealLong& l) const
{
  return Real(BigInt(l.ker) + ker);
}

Real RealBigInt :: addDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator +(BigFloat(ker)));
}

Real RealBigInt :: addBigInt(const RealBigInt& I) const
{
  return Real(I.ker + ker);
}

Real RealBigInt :: addBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator +(BigFloat(ker)));
}

Real RealBigInt :: addRational(const RealRational& R) const
{
  return Real(R.ker + Rational(ker));
}

//  subtraction

Real RealBigInt :: operator -(const Real& x) const
{
  return x.subBigInt(*this);
}

Real RealBigInt :: subInt(const RealInt& i) const
{
  return Real(BigInt(i.ker) - ker);
}

Real RealBigInt :: subLong(const RealLong& l) const
{
  return Real(BigInt(l.ker) - ker);
}

Real RealBigInt :: subDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator -(BigFloat(ker)));
}

Real RealBigInt :: subBigInt(const RealBigInt& I) const
{
  return Real(I.ker - ker);
}

Real RealBigInt :: subBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator -(BigFloat(ker)));
}

Real RealBigInt :: subRational(const RealRational& R) const
{
  return Real(R.ker - Rational(ker));
}

//  multiplication

Real RealBigInt :: operator *(const Real& x) const
{
  return x.mulBigInt(*this);
}

Real RealBigInt :: mulInt(const RealInt& i) const
{
  return Real(BigInt(i.ker) * ker);
}

Real RealBigInt :: mulLong(const RealLong& l) const
{
  return Real(BigInt(l.ker) * ker);
}

Real RealBigInt :: mulDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator *(BigFloat(ker)));
}

Real RealBigInt :: mulBigInt(const RealBigInt& I) const
{
  return Real(I.ker * ker);
}

Real RealBigInt :: mulBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator *(BigFloat(ker)));
}

Real RealBigInt :: mulRational(const RealRational& R) const
{
  return Real(Rational(R.ker.numerator() * ker, R.ker.denominator()));
}

//  division

Real RealBigInt :: div(const Real& x, const extULong& r) const
{
  return x.divBigInt(*this, r);
}

Real RealBigInt :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(Rational(BigInt(i.ker), ker));
}

Real RealBigInt :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(Rational(BigInt(l.ker), ker));
}

Real RealBigInt :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(BigFloat(d.ker).div(BigFloat(ker), r));
}

Real RealBigInt :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(Rational(I.ker, ker));
}

Real RealBigInt :: divBigFloat(const RealBigFloat& B, const extULong& r) const
{
  return Real(B.ker.div(BigFloat(ker), r));
}

Real RealBigInt :: divRational(const RealRational& R, const extULong& r) const
{
  return Real(Rational(R.ker.numerator(), R.ker.denominator() * ker));
}

//  squareroot

Real RealBigInt :: sqrt(const extLong& a) const
{
  return Real(BigFloat(ker).sqrt(a));
}

//  equality

int RealBigInt :: operator ==(const Real& x) const
{
  return x.eqlBigInt(*this);
}

int RealBigInt :: eqlInt(const RealInt& i) const
{
  return BigInt(i.ker) == ker;
}

int RealBigInt :: eqlLong(const RealLong& l) const
{
  return BigInt(l.ker) == ker;
}

int RealBigInt :: eqlDouble(const RealDouble& d) const
{
  return BigFloat(d.ker) == BigFloat(ker);
}

int RealBigInt :: eqlBigInt(const RealBigInt& I) const
{
  return I.ker == ker;
}

int RealBigInt :: eqlBigFloat(const RealBigFloat& B) const
{
  return B.ker == BigFloat(ker);
}

int RealBigInt :: eqlRational(const RealRational& R) const
{
  return R.ker == Rational(ker);
}

//  smaller-than

int RealBigInt :: operator <(const Real& x) const
{
  return x.grtBigInt(*this);
}

int RealBigInt :: grtInt(const RealInt& i) const
{
  return BigInt(i.ker) < ker;
}

int RealBigInt :: grtLong(const RealLong& l) const
{
  return BigInt(l.ker) < ker;
}

int RealBigInt :: grtDouble(const RealDouble& d) const
{
  return BigFloat(d.ker) < BigFloat(ker);
}

int RealBigInt :: grtBigInt(const RealBigInt& I) const
{
  return I.ker < ker;
}

int RealBigInt :: grtBigFloat(const RealBigFloat& B) const
{
  return B.ker < BigFloat(ker);
}

int RealBigInt :: grtRational(const RealRational& R) const
{
  return R.ker < Rational(ker);
}

//  builtin functions

int RealBigInt :: isExact() const
{
  return 1;
}

extLong RealBigInt :: lMSB() const
{
  if (sign(ker))
    return extLong(lg(ker));
  else
    return extLong(tinyLong);
}

extLong RealBigInt :: MSB() const
{
  if (sign(ker))
    return extLong(lg(ker));
  else
    return extLong(tinyLong);
}

extLong RealBigInt :: flrLgErr() const
{
  return extLong(tinyLong);
}

extLong RealBigInt :: clLgErr() const
{
  return extLong(tinyLong);
}

int RealBigInt :: isZeroIn() const
{
  return sign(ker) == 0;
}

unsigned long RealBigInt :: degree() const
{
  return 1;
}

unsigned long RealBigInt :: length() const
{
  long l = flrLg(ker);
  
  return l ? l + 1 : 2;
}

unsigned long RealBigInt :: height() const
{
  long l = flrLg(ker);
  
  return l ? l + 1 : 2;
}

int RealBigInt :: sgn() const
{
  return sign(ker);
}

//  stream

ostream& RealBigInt :: operator <<(ostream& o) const
{
  o << "Real(BigInt " << ker << ')';
  
  return o;
}

//  class RealBigFloat

//  constructor

RealBigFloat :: RealBigFloat(const BigFloat& B) : Real(RealBase()), ker(B)
{}

//  the destructor

RealBigFloat :: ~RealBigFloat()
{}
  
//  approximation

Real RealBigFloat :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(ker, r, a);
    
  return Real(x);
}

//  unary minus
  
Real RealBigFloat :: operator -() const
{
  return Real(- ker);
}

//  addition

Real RealBigFloat :: operator +(const Real& x) const
{
  return x.addBigFloat(*this);
}

Real RealBigFloat :: addInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator +(ker));
}

Real RealBigFloat :: addLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator +(ker));
}

Real RealBigFloat :: addDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator +(ker));
}

Real RealBigFloat :: addBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator +(ker));
}

Real RealBigFloat :: addBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator +(ker));
}

Real RealBigFloat :: addRational(const RealRational& R) const
{
  if (ker.isExact())
    return Real(R.ker + Rational(ker));
  else
  {
    BigFloat x;
    
    x.approx(R.ker, inftyLong, - ker.flrLgErr());
    
    return Real(x.operator +(ker));
  }
}

//  subtraction

Real RealBigFloat :: operator -(const Real& x) const
{
  return x.subBigFloat(*this);
}

Real RealBigFloat :: subInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator -(ker));
}

Real RealBigFloat :: subLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator -(ker));
}

Real RealBigFloat :: subDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator -(ker));
}

Real RealBigFloat :: subBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator -(ker));
}

Real RealBigFloat :: subBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator -(ker));
}

Real RealBigFloat :: subRational(const RealRational& R) const
{
  if (ker.isExact())
    return Real(R.ker - Rational(ker));
  else
  {
    BigFloat x;
    
    x.approx(R.ker, inftyLong, - ker.flrLgErr());
    
    return Real(x.operator -(ker));
  }
}

//  multiplication

Real RealBigFloat :: operator *(const Real& x) const
{
  return x.mulBigFloat(*this);
}

Real RealBigFloat :: mulInt(const RealInt& i) const
{
  return Real(BigFloat(i.ker).operator *(ker));
}

Real RealBigFloat :: mulLong(const RealLong& l) const
{
  return Real(BigFloat(l.ker).operator *(ker));
}

Real RealBigFloat :: mulDouble(const RealDouble& d) const
{
  return Real(BigFloat(d.ker).operator *(ker));
}

Real RealBigFloat :: mulBigInt(const RealBigInt& I) const
{
  return Real(BigFloat(I.ker).operator *(ker));
}

Real RealBigFloat :: mulBigFloat(const RealBigFloat& B) const
{
  return Real(B.ker.operator *(ker));
}

Real RealBigFloat :: mulRational(const RealRational& R) const
{
  if (ker.isExact())
    return Real(R.ker * Rational(ker));
  else
  {
    BigFloat x;
    
    x.approx(R.ker, extULong(ker.MSB() - ker.flrLgErr()) + 1, inftyLong);
    
    return Real(x.operator *(ker));
  }
}

//  division

Real RealBigFloat :: div(const Real& x, const extULong& r) const
{
  return x.divBigFloat(*this, r);
}

Real RealBigFloat :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(BigFloat(i.ker).div(ker, r));
}

Real RealBigFloat :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(BigFloat(l.ker).div(ker, r));
}

Real RealBigFloat :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(BigFloat(d.ker).div(ker, r));
}

Real RealBigFloat :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(BigFloat(I.ker).div(ker, r));
}

Real RealBigFloat :: divBigFloat(const RealBigFloat& B,
				 const extULong& r) const
{
  return Real(BigFloat(B.ker).div(ker, r));
}

Real RealBigFloat :: divRational(const RealRational& R,
				 const extULong& r) const
{
  if (ker.isExact())
    return Real(R.ker / Rational(ker));
  else
  {
    BigFloat x;
    
    x.approx(R.ker, extULong(ker.MSB() - ker.flrLgErr()) + 1, inftyLong);
    
    return Real(x.div(ker, r));
  }
}

//  squareroot

Real RealBigFloat :: sqrt(const extLong& a) const
{
  return Real(ker.sqrt(a));
}

//  equality

int RealBigFloat :: operator ==(const Real& x) const
{
  return x.eqlBigFloat(*this);
}

int RealBigFloat :: eqlInt(const RealInt& i) const
{
  return BigFloat(i.ker) == ker;
}

int RealBigFloat :: eqlLong(const RealLong& l) const
{
  return BigFloat(l.ker) == ker;
}

int RealBigFloat :: eqlDouble(const RealDouble& d) const
{
  return BigFloat(d.ker) == ker;
}

int RealBigFloat :: eqlBigInt(const RealBigInt& I) const
{
  return BigFloat(I.ker) == ker;
}

int RealBigFloat :: eqlBigFloat(const RealBigFloat& B) const
{
  return B.ker == ker;
}

int RealBigFloat :: eqlRational(const RealRational& R) const
{
  return R.ker == Rational(ker);
}

//  smaller-than

int RealBigFloat :: operator <(const Real& x) const
{
  return x.grtBigFloat(*this);
}

int RealBigFloat :: grtInt(const RealInt& i) const
{
  return BigFloat(i.ker) < ker;
}

int RealBigFloat :: grtLong(const RealLong& l) const
{
  return BigFloat(l.ker) < ker;
}

int RealBigFloat :: grtDouble(const RealDouble& d) const
{
  return BigFloat(d.ker) < ker;
}

int RealBigFloat :: grtBigInt(const RealBigInt& I) const
{
  return BigFloat(I.ker) < ker;
}

int RealBigFloat :: grtBigFloat(const RealBigFloat& B) const
{
  return B.ker < ker;
}

int RealBigFloat :: grtRational(const RealRational& R) const
{
  return R.ker < Rational(ker);
}

//  builtin functions

int RealBigFloat :: isExact() const
{
  return ker.isExact();
}

extLong RealBigFloat :: lMSB() const
{
  return ker.lMSB();
}

extLong RealBigFloat :: MSB() const
{
  return ker.MSB();
}

extLong RealBigFloat :: flrLgErr() const
{
  return ker.flrLgErr();
}

extLong RealBigFloat :: clLgErr() const
{
  return ker.clLgErr();
}

int RealBigFloat :: isZeroIn() const
{
  return ker.isZeroIn();
}

unsigned long RealBigFloat :: degree() const
{
  return 1;
}

unsigned long RealBigFloat :: length() const
{
  Rational R  = Rational(ker);
  long     ln = lg(R.numerator());
  long     ld = lg(R.denominator());
  long     l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

unsigned long RealBigFloat :: height() const
{
  Rational R  = Rational(ker);
  long     ln = lg(R.numerator());
  long     ld = lg(R.denominator());
  long     l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

int RealBigFloat :: sgn() const
{
  return ker.sign();
}

//  stream

ostream& RealBigFloat :: operator <<(ostream& o) const
{
  o << "Real(BigFloat " << ker << ')';
  
  return o;
}

//  class RealRational

//  constructor

RealRational :: RealRational(const Rational& R) : Real(RealBase()), ker(R)
{}

//  the destructor

RealRational :: ~RealRational()
{}
  
//  approximation

Real RealRational :: approx(const extULong& r, const extLong& a) const
{
  BigFloat x;
  
  x.approx(ker, r, a);
  
  return Real(x);
}

//  unary minus
  
Real RealRational :: operator -() const
{
  return Real(- ker);
}

//  addition

Real RealRational :: operator +(const Real& x) const
{
  return x.addRational(*this);
}

Real RealRational :: addInt(const RealInt& i) const
{
  return Real(Rational(i.ker) + ker);
}

Real RealRational :: addLong(const RealLong& l) const
{
  return Real(Rational(l.ker) + ker);
}

Real RealRational :: addDouble(const RealDouble& d) const
{
  return Real(Rational(d.ker) + ker);
}

Real RealRational :: addBigInt(const RealBigInt& I) const
{
  return Real(Rational(I.ker) + ker);
}

Real RealRational :: addBigFloat(const RealBigFloat& B) const
{
  if (B.ker.isExact())
    return Real(Rational(B.ker) + ker);
  else
  {
    BigFloat x;
    
    x.approx(ker, inftyLong, - B.ker.flrLgErr());
    
    return Real(B.ker.operator +(x));
  }
}

Real RealRational :: addRational(const RealRational& R) const
{
  return Real(R.ker + ker);
}

//  subtraction

Real RealRational :: operator -(const Real& x) const
{
  return x.subRational(*this);
}

Real RealRational :: subInt(const RealInt& i) const
{
  return Real(Rational(i.ker) - ker);
}

Real RealRational :: subLong(const RealLong& l) const
{
  return Real(Rational(l.ker) - ker);
}

Real RealRational :: subDouble(const RealDouble& d) const
{
  return Real(Rational(d.ker) - ker);
}

Real RealRational :: subBigInt(const RealBigInt& I) const
{
  return Real(Rational(I.ker) - ker);
}

Real RealRational :: subBigFloat(const RealBigFloat& B) const
{
  if (B.ker.isExact())
    return Real(Rational(B.ker) - ker);
  else
  {
    BigFloat x;
    
    x.approx(ker, inftyLong, - B.ker.flrLgErr());
    
    return Real(B.ker.operator -(x));
  }
}

Real RealRational :: subRational(const RealRational& R) const
{
  return Real(R.ker - ker);
}

//  multiplication

Real RealRational :: operator *(const Real& x) const
{
  return x.mulRational(*this);
}

Real RealRational :: mulInt(const RealInt& i) const
{
  return Real(Rational(BigInt(i.ker) * ker.numerator(), ker.denominator()));
}

Real RealRational :: mulLong(const RealLong& l) const
{
  return Real(Rational(l.ker * ker.numerator(), ker.denominator()));
}

Real RealRational :: mulDouble(const RealDouble& d) const
{
  return Real(Rational(d.ker) * ker);
}

Real RealRational :: mulBigInt(const RealBigInt& I) const
{
  return Real(Rational(I.ker * ker.numerator(), ker.denominator()));
}

Real RealRational :: mulBigFloat(const RealBigFloat& B) const
{
  if (B.ker.isExact())
    return Real(Rational(B.ker) * ker);
  else
  {
    BigFloat x;
    
    x.approx(ker, extULong(B.ker.MSB() - B.ker.flrLgErr()) + 1, inftyLong);
    
    return Real(B.ker.operator *(x));
  }
}

Real RealRational :: mulRational(const RealRational& R) const
{
  return Real(R.ker * ker);
}

//  division

Real RealRational :: div(const Real& x, const extULong& r) const
{
  return x.divRational(*this, r);
}

Real RealRational :: divInt(const RealInt& i, const extULong& r) const
{
  return Real(Rational(BigInt(i.ker) * ker.denominator(), ker.numerator()));
}

Real RealRational :: divLong(const RealLong& l, const extULong& r) const
{
  return Real(Rational(l.ker * ker.denominator(), ker.numerator()));
}

Real RealRational :: divDouble(const RealDouble& d, const extULong& r) const
{
  return Real(Rational(d.ker) / ker);
}

Real RealRational :: divBigInt(const RealBigInt& I, const extULong& r) const
{
  return Real(Rational(I.ker * ker.denominator(), ker.numerator()));
}

Real RealRational :: divBigFloat(const RealBigFloat& B,
				 const extULong& r) const
{
  if (B.ker.isExact())
    return Real(Rational(B.ker) / ker);
  else
  {
    BigFloat x;
    
    x.approx(ker, extULong(B.ker.MSB() - B.ker.flrLgErr()) + 1, inftyLong);
    
    return Real(B.ker.div(x, r));
  }
}

Real RealRational :: divRational(const RealRational& R,
				 const extULong& r) const
{
  return Real(R.ker / ker);
}

//  squareroot

Real RealRational :: sqrt(const extLong& a) const
{
  BigFloat x;
  
  x.approx(ker, inftyLong, 2 * defAbsPrec + 8);
  
  return Real(x.sqrt(a));
}

//  equality

int RealRational :: operator ==(const Real& x) const
{
  return x.eqlRational(*this);
}

int RealRational :: eqlInt(const RealInt& i) const
{
  return Rational(i.ker) == ker;
}

int RealRational :: eqlLong(const RealLong& l) const
{
  return Rational(l.ker) == ker;
}

int RealRational :: eqlDouble(const RealDouble& d) const
{
  return Rational(d.ker) == ker;
}

int RealRational :: eqlBigInt(const RealBigInt& I) const
{
  return Rational(I.ker) == ker;
}

int RealRational :: eqlBigFloat(const RealBigFloat& B) const
{
  return Rational(B.ker) == ker;
}

int RealRational :: eqlRational(const RealRational& R) const
{
  return R.ker == ker;
}

//  smaller-than

int RealRational :: operator <(const Real& x) const
{
  return x.grtRational(*this);
}

int RealRational :: grtInt(const RealInt& i) const
{
  return Rational(i.ker) < ker;
}

int RealRational :: grtLong(const RealLong& l) const
{
  return Rational(l.ker) < ker;
}

int RealRational :: grtDouble(const RealDouble& d) const
{
  return Rational(d.ker) < ker;
}

int RealRational :: grtBigInt(const RealBigInt& I) const
{
  return Rational(I.ker) < ker;
}

int RealRational :: grtBigFloat(const RealBigFloat& B) const
{
  return Rational(B.ker) < ker;
}

int RealRational :: grtRational(const RealRational& R) const
{
  return R.ker < ker;
}

//  builtin functions

int RealRational :: isExact() const
{
  return 1;
}

extLong RealRational :: lMSB() const
{
  if (sign(ker))
    return extLong(lg(ker.numerator()) - lg(ker.denominator()));
  else
    return extLong(tinyLong);
}

extLong RealRational :: MSB() const
{
  if (sign(ker))
    return extLong(lg(ker.numerator()) - lg(ker.denominator()));
  else
    return extLong(tinyLong);
}

extLong RealRational :: flrLgErr() const
{
  return extLong(tinyLong);
}

extLong RealRational :: clLgErr() const
{
  return extLong(tinyLong);
}

int RealRational :: isZeroIn() const
{
  return sign(ker) == 0;
}

unsigned long RealRational :: degree() const
{
  return 1;
}

unsigned long RealRational :: length() const
{
  long ln = lg(ker.numerator());
  long ld = lg(ker.denominator());
  long l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

unsigned long RealRational :: height() const
{
  long ln = lg(ker.numerator());
  long ld = lg(ker.denominator());
  long l  = ln > ld ? ln : ld; 
  
  return l + 1;
}

int RealRational :: sgn() const
{
  return sign(ker);
}

//  stream

ostream& RealRational :: operator <<(ostream& o) const
{
  o << "Real(Rational " << ker << ')';
  
  return o;
}

