//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: Expr.h
//    class Expr
//

#ifndef _EXPR_H
#define _EXPR_H

#include "Real.h"
#include "ExprUtil.h"

//  struct ExprBase
//    used to distinguish Expr from its descendents

struct ExprBase
{
  ExprBase() {}
};

//  forward reference

class ExprRep;
class UnaryOpRep;
class BinOpRep;

//  class Expr
//    envelop

class Expr
{
  //  comparison
  
  int compare();
  
protected:
  
  ExprRep* rep;
  
  //  constructors
  
  Expr(ExprBase);
  
  Expr(UnaryOpRep*);
  Expr(BinOpRep*);
  
  //  stream
  
  ostream& put(ostream&) const;
  
  //  error detection
  
  void error(const char*) const;
  
public:
  
  //  constructors
  
  Expr();
  Expr(Real);
  
  Expr(int);
  Expr(long);
  Expr(double);
  Expr(BigInt);
  Expr(BigFloat);
  Expr(Rational);
  
  Expr(const Expr&);
  
  //  the destructor
  
  virtual ~Expr();
  
  //  approximation
  
  Real approx(const extULong& = defRelPrec, const extLong& = defAbsPrec);
  
  //  assignment
  
  Expr& operator =(Real);
  
  Expr& operator =(int);
  Expr& operator =(long);
  Expr& operator =(double);
  Expr& operator =(BigInt);
  Expr& operator =(BigFloat);
  Expr& operator =(Rational);
  
  Expr& operator =(const Expr&);
  
  Expr& reset();
  Expr& reset(Real);
  
  Expr& reset(int);
  Expr& reset(long);
  Expr& reset(double);
  Expr& reset(BigInt);
  Expr& reset(BigFloat);
  Expr& reset(Rational);
  
  Expr& reset(const Expr&);
  
  Expr& duplicate(const Expr&);
  
  //  unary minus
  
  Expr operator -() const;
  
  //  addition
  
  Expr operator +(const Expr&) const;
  friend Expr operator +(Real, const Expr&);
  friend Expr operator +(const Expr&, Real);
  
  //  subtraction
  
  Expr operator -(const Expr&) const;
  friend Expr operator -(Real, const Expr&);
  friend Expr operator -(const Expr&, Real);
  
  //  multiplication
  
  Expr operator *(const Expr&) const;
  friend Expr operator *(Real, const Expr&);
  friend Expr operator *(const Expr&, Real);
  
  //  division
  
  Expr operator /(const Expr&) const;
  friend Expr operator /(Real, const Expr&);
  friend Expr operator /(const Expr&, Real);
  
  //  squareroot
  
  Expr sqrt() const;
  friend Expr sqrt(const Expr&);
  
  //  comparison
  
  int operator ==(const Expr&) const;
  int operator !=(const Expr&) const;
  int operator <(const Expr&) const;
  int operator <=(const Expr&) const;
  int operator >(const Expr&) const;
  int operator >=(const Expr&) const;
  
  //  power
  
  Expr pow(unsigned long) const;
  friend Expr pow(const Expr&, unsigned long);
  
  //  stream
  
  virtual ostream& print(ostream&, long = defPrtDgt);
  friend ostream& operator <<(ostream&, const Expr&);
};

//  class ExprRep

class ExprRep : public Expr
{
friend class Expr;
friend class ParamRep;
friend class UnaryOpRep;
friend class BinOpRep;
  
private:
  
  unsigned refCount;
  
protected:
  
  //  exact value
  
  extLong uMSB;
  extLong lMSB;
  int     sign;
  
  unsigned long degree;
  unsigned long length;
//  unsigned long height;
  
  unsigned exStamp;
  
  //  approximate value
  
  Real     appValue;
  extULong relPrecision;
  extLong  absPrecision;
  
  unsigned appStamp;
  
  //  constructors
  
  ExprRep();
  
  //  the destructor
  
  virtual ~ExprRep();
  
  //  letter-envelope
  
  int      isRepThis() const;
  unsigned incRefCount();
  unsigned decRefCount();
  
  //  assignment
  
  virtual ExprRep* assign(Real);
  virtual ExprRep* assign(ExprRep*);
  
  virtual ExprRep* duplicate(ExprDict* d);
  
  //  exact value
  
  virtual void evalExValue();
  
  virtual void reEvalExValue();
  
  extLong getUMSB() const;
  extLong getLMSB() const;
  int     getSign() const;
  
  unsigned long getDegree() const;
  unsigned long getLength() const;
  unsigned long getHeight() const;
  
  unsigned getExStamp() const;
  void     putExStamp(unsigned);
  
  //  approximation
  
  virtual void evalAppValue(const extULong&, const extLong&);
  virtual Real incEvalAppValue(const extULong&, const extLong&);
  
  virtual void reEvalAppValue(const extULong&, const extLong&);
  virtual Real incReEvalAppValue(const extULong&, const extLong&);
  
  Real getAppValue() const;
  
  unsigned getAppStamp() const;
  void     putAppStamp(unsigned);
  
  //  stream
  
  ostream& put(ostream&);
  
public:
  
  //  stream
  
  virtual ostream& print(ostream&, long);
  virtual ostream& putDebug(ostream&) const;
};

//  class ParamRep
//    parameter

class ParamRep : public ExprRep
{
  Real exValue;
  
  unsigned hasExValue;
  
protected:
  
  //  assignment
  
  ExprRep* assign(Real);
  ExprRep* assign(ExprRep*);
  
  ExprRep* duplicate(ExprDict*);
  
  //  exact value
  
  void evalExValue();
  
  //  approximation
  
  void evalAppValue(const extULong&, const extLong&);
  Real incEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream& o) const;
  
public:
  
  //  constructor
  
  ParamRep();
  ParamRep(Real);
  
  //  the destructor
  
  ~ParamRep();
};

//  class UnaryOpRep
//    unary operator

class UnaryOpRep : public ExprRep
{
friend class ParamRep;
friend class BinOpRep;
  
protected:
  
  ExprRep* child;
  
  unsigned srcExStamp;
  
  unsigned srcAppStamp;
  
  //  constructor
  
  UnaryOpRep(ExprRep*);
  
  //  the destructor
  
  virtual ~UnaryOpRep();
  
  //  assignment
  
  ExprRep* assign(Real);
  ExprRep* assign(ExprRep*);
  
  ExprRep* duplicate(ExprDict*);
  
  virtual ExprRep* duplicateRep(ExprRep*);
  
  //  exact value
  
  void evalExValue();
  
  virtual void reEvalExValue();
  
  void evalChildExValue() const;
  
  extLong getChildUMSB() const;
  extLong getChildLMSB() const;
  int     getChildSign() const;
  
  unsigned long getChildDegree() const;
  unsigned long getChildLength() const;
//  unsigned long getChildHeight() const;
  
  unsigned getChildExStamp() const;
  
  //  approximation
  
  void evalAppValue(const extULong&, const extLong&);
  Real incEvalAppValue(const extULong&, const extLong&);
  
  virtual void reEvalAppValue(const extULong&, const extLong&);
  virtual Real incReEvalAppValue(const extULong&, const extLong&);
  
  void evalChildAppValue(const extULong&, const extLong&) const;
  
  Real getChildAppValue() const;
  
  unsigned getChildAppStamp() const;
  
  //  stream
  
  virtual ostream& putDebug(ostream&) const;
};

//  class NegRep
//    unary minus operator

class NegRep : public UnaryOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream&) const;
  
public:
  
  //  constructor
  
  NegRep(ExprRep*);
  
  //  the destructor
  
  ~NegRep();
};

//  class SqrtRep
//    squareroot

class SqrtRep : public UnaryOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream&) const;
  
public:
  
  //  constructor
  
  SqrtRep(ExprRep*);
  
  //  the destructor
  
  ~SqrtRep();
};

//  class BinaryOpRep
//    binary operator

class BinOpRep : public ExprRep
{
protected:
  
  ExprRep* frstChild;
  ExprRep* scndChild;
  
  unsigned frstSrcExStamp;
  unsigned scndSrcExStamp;
  
  unsigned frstSrcAppStamp;
  unsigned scndSrcAppStamp;
  
  //  constructor
  
  BinOpRep(ExprRep*, ExprRep*);
  
  //  the destructor
  
  virtual ~BinOpRep();
  
  //  assignment
  
  ExprRep* assign(Real);
  ExprRep* assign(ExprRep*);
  
  ExprRep* duplicate(ExprDict*);
  
  virtual ExprRep* duplicateRep(ExprRep*, ExprRep*);
  
  //  exact value
  
  void evalExValue();
  
  virtual void reEvalExValue();
  
  void evalFrstChildExValue() const;
  void evalScndChildExValue() const;
  
  extLong getFrstChildUMSB() const;
  extLong getScndChildUMSB() const;
  extLong getFrstChildLMSB() const;
  extLong getScndChildLMSB() const;
  int     getFrstChildSign() const;
  int     getScndChildSign() const;
  
  unsigned long getFrstChildDegree() const;
  unsigned long getScndChildDegree() const;
  unsigned long getFrstChildLength() const;
  unsigned long getScndChildLength() const;
//  unsigned long getFrstChildHeight() const;
//  unsigned long getScndChildHeight() const;
  
  unsigned getFrstChildExStamp() const;
  unsigned getScndChildExStamp() const;
  
  //  approximation
  
  void evalAppValue(const extULong&, const extLong&);
  Real incEvalAppValue(const extULong&, const extLong&);
  
  virtual void reEvalAppValue(const extULong&, const extLong&);
  virtual Real incReEvalAppValue(const extULong&, const extLong&);
  
  void evalFrstChildAppValue(const extULong&, const extLong&) const;
  void evalScndChildAppValue(const extULong&, const extLong&) const;
  
  Real getFrstChildAppValue() const;
  Real getScndChildAppValue() const;
  
  unsigned getFrstChildAppStamp() const;
  unsigned getScndChildAppStamp() const;
  
  //  stream
  
  virtual ostream& putDebug(ostream&) const;
};

//  class AddRep
//    addition

class AddRep : public BinOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*, ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream&) const;
  
public:
  
  //  constructor
  
  AddRep(ExprRep*, ExprRep*);
  
  //  the destructor
  
  ~AddRep();
};

//  class SubRep
//    subtraction

class SubRep : public BinOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*, ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream& o) const;
  
public:
  
  //  constructor
  
  SubRep(ExprRep*, ExprRep*);
  
  //  the destructor
  
  ~SubRep();
};

//  class MultRep
//    multiplication

class MultRep : public BinOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*, ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream&) const;
  
public:
  
  //  constructor
  
  MultRep(ExprRep*, ExprRep*);
  
  //  the destructor
  
  ~MultRep ();
};

//  class DivRep
//    Division

class DivRep : public BinOpRep
{
protected:
  
  //  assignment
  
  ExprRep* duplicateRep(ExprRep*, ExprRep*);
  
  //  exact value
  
  void reEvalExValue();
  
  //  approximation
  
  void reEvalAppValue(const extULong&, const extLong&);
  Real incReEvalAppValue(const extULong&, const extLong&);
  
  //  stream
  
  ostream& putDebug(ostream&) const;
  
public:
  
  //  constructor
  
  DivRep(ExprRep*, ExprRep*);
  
  //  the destructor
  
  ~DivRep ();
};

#endif

