//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: Expr.cc
//    class Expr
//

#include <iostream.h>
#include "Expr.h"

//  floor log base 2 of abs(x)
//  convention lg(0) = 0

extern long flrLg(long);
extern long flrLg(unsigned long);
extern long flrLg(const BigInt&);

//  class Expr

//  constructors

Expr :: Expr(ExprBase)
{}

Expr :: Expr(UnaryOpRep* p)
{
  rep = p;
  rep->incRefCount();
}

Expr :: Expr(BinOpRep* p)
{
  rep = p;
  rep->incRefCount();
}

Expr :: Expr()
{
  rep = new ParamRep();
  rep->incRefCount();
}

Expr :: Expr(Real r)
{
  rep = new ParamRep(r);
  rep->incRefCount();
}

Expr :: Expr(int i)
{
  rep = new ParamRep(Real(i));
  rep->incRefCount();
}

Expr :: Expr(long l)
{
  rep = new ParamRep(Real(l));
  rep->incRefCount();
}

Expr :: Expr(double d)
{
  rep = new ParamRep(Real(d));
  rep->incRefCount();
}

Expr :: Expr(BigInt I)
{
  rep = new ParamRep(Real(I));
  rep->incRefCount();
}

Expr :: Expr(BigFloat B)
{
  rep = new ParamRep(Real(B));
  rep->incRefCount();
}

Expr :: Expr(Rational R)
{
  rep = new ParamRep(Real(R));
  rep->incRefCount();
}

Expr :: Expr(const Expr& e)
{
  rep = new ParamRep(0);
  
  ExprRep* p = rep->assign(e.rep);
  
  if (p != rep)
  {
    if (!rep->decRefCount())
      delete rep;
    
    rep = p;
  }
  
  rep->incRefCount();
}

//  the destructor

Expr :: ~Expr()
{
  if (rep != this && !rep->decRefCount())
    delete rep;
}

//  approximation

Real Expr :: approx(const extULong& relPrec, const extLong& absPrec)
{
  rep->evalExValue();
  rep->evalAppValue(relPrec, absPrec);
  
  return rep->getAppValue();
}

//  assignment

Expr& Expr :: operator =(Real r)
{
  ExprRep* p = rep->assign(r);
  
  if (p != rep)
  {
    if (!rep->decRefCount())
      delete rep;
    
    rep = p;
    rep->incRefCount();
  }
  
  return *this;
}

Expr& Expr :: operator =(int i)
{
  return operator =(Real(i));
}

Expr& Expr :: operator =(long l)
{
  return operator =(Real(l));
}

Expr& Expr :: operator =(double d)
{
  return operator =(Real(d));
}

Expr& Expr :: operator =(BigInt I)
{
  return operator =(Real(I));
}

Expr& Expr :: operator =(BigFloat B)
{
  return operator =(Real(B));
}

Expr& Expr :: operator =(Rational R)
{
  return operator =(Real(R));
}

Expr& Expr :: operator =(const Expr& e)
{
  ExprRep* p = rep->assign(e.rep);
  
  if (p != rep)
  {
    if (!rep->decRefCount())
      delete rep;
    
    rep = p;
    rep->incRefCount();
  }
  
  return *this;
}

Expr& Expr :: reset()
{
  rep->decRefCount();
  rep = new ParamRep();
  rep->incRefCount();
  
  return *this;
}

Expr& Expr :: reset(Real r)
{
  rep->decRefCount();
  rep = new ParamRep(r);
  rep->incRefCount();
  
  return *this;
}

Expr& Expr :: reset(int i)
{
  return reset(Real(i));
}

Expr& Expr :: reset(long l)
{
  return reset(Real(l));
}

Expr& Expr :: reset(double d)
{
  return reset(Real(d));
}

Expr& Expr :: reset(BigInt I)
{
  return reset(Real(I));
}

Expr& Expr :: reset(BigFloat B)
{
  return reset(Real(B));
}

Expr& Expr :: reset(Rational R)
{
  return reset(Real(R));
}

Expr& Expr :: reset(const Expr& e)
{
  rep->decRefCount();
  rep = e.rep;
  rep->incRefCount();
  
  return *this;
}

Expr& Expr :: duplicate(const Expr& e)
{
  if (!rep->decRefCount())
    delete rep;
  
  ExprDict* d = new ExprDict;
  
  rep = e.rep->duplicate(d);
  rep->incRefCount();
  
  delete d;
  
  return *this;
}

//  unary-minus

Expr Expr :: operator -() const
{
  return Expr(new NegRep(rep));
}

//  addition

Expr Expr :: operator +(const Expr& e) const
{
  return Expr(new AddRep(rep, e.rep));
}

Expr operator +(Real r, const Expr& e)
{
  return Expr(new AddRep(new ParamRep(r), e.rep));
}

Expr operator +(const Expr& e, Real r)
{
  return Expr(new AddRep(e.rep, new ParamRep(r)));
}

//  subtraction

Expr Expr :: operator -(const Expr& e) const
{
  return Expr(new SubRep(rep, e.rep));
}

Expr operator -(Real r, const Expr& e)
{
  return Expr(new SubRep(new ParamRep(r), e.rep));
}

Expr operator -(const Expr& e, Real r)
{
  return Expr(new SubRep(e.rep, new ParamRep(r)));
}

//  multiplication

Expr Expr :: operator *(const Expr& e) const
{
  return Expr(new MultRep(rep, e.rep));
}

Expr operator *(Real r, const Expr& e)
{
  return Expr(new MultRep(new ParamRep(r), e.rep));
}

Expr operator *(const Expr& e, Real r)
{
  return Expr(new MultRep(e.rep, new ParamRep(r)));
}

//  division

Expr Expr :: operator /(const Expr& e) const
{
  return Expr(new DivRep(rep, e.rep));
}

Expr operator /(Real r, const Expr& e)
{
  return Expr(new DivRep(new ParamRep(r), e.rep));
}

Expr operator /(const Expr& e, Real r)
{
  return Expr(new DivRep(e.rep, new ParamRep(r)));
}

//  squareroot

Expr Expr :: sqrt() const
{
  return Expr(new SqrtRep(rep));
}

Expr sqrt(const Expr& e)
{
  return e.sqrt();
}

//  comparisons

int Expr :: compare()
{
  rep->evalExValue();
  
  return rep->getSign();
}

int Expr :: operator ==(const Expr& e) const
{
  return !operator -(e).compare();
}

int Expr :: operator !=(const Expr& e) const
{
  return operator -(e).compare() != 0;
}

int Expr :: operator <(const Expr& e) const
{
  return operator -(e).compare() < 0;
}

int Expr :: operator <=(const Expr& e) const
{
  return operator -(e).compare() <= 0;
}

int Expr :: operator >(const Expr& e) const
{
  return operator -(e).compare() > 0;
}

int Expr :: operator >=(const Expr& e) const
{
  return operator -(e).compare() >= 0;
}

//  power

Expr Expr :: pow(unsigned long power) const
{
  if (power == 0)
    return Expr(1);
  else if (power == 1)
    return Expr(new MultRep(rep, new ParamRep(1)));
  else
  {
    unsigned long i, j, k;
    
    unsigned long height = flrLg(power);
    unsigned long b      = 1 << height;
    
    unsigned long floors = 0;
    i = b >> 1;
    
    while (i > 0)
    {
      if (power & i)
        floors++;
      
      i >>= 1;
    }
    
    Expr** bins = new Expr* [height + 1];
    bins[0] = rep;
    
    for (i = 0; i < height-1; i++)
      bins[i+1] = new MultRep(bins[i]->rep, bins[i]->rep);
    
    if (!floors)
      return Expr(new MultRep(bins[i]->rep, bins[i]->rep));
    else
      bins[i+1] = new MultRep(bins[i]->rep, bins[i]->rep);
    
    Expr* r = new Expr;
    
    i = 0;
    j = 1;
    k = 0;
    
    while (i < height && k <= floors)
    {
      if (power & j)
      {
        if (!k)
          r->rep = bins[i]->rep;
        else
          r->rep = new MultRep(r->rep, bins[i]->rep);
        
        k++;
      }
      
      i++;
      j <<= 1;
    }
    
    return Expr(new MultRep(r->rep, bins[height]->rep));
  }
}

Expr pow(const Expr& e, unsigned long p)
{
  return e.pow(p);
}

//  stream

ostream& Expr :: put(ostream& o) const
{
  rep->put(o);
  
  return o;
}

ostream& Expr :: print(ostream& o, long l)
{
  rep->print(o, l);
//  rep->putDebug(o);
  
  return o;
}

ostream& operator <<(ostream& o, const Expr& e)
{
  e.put(o);
//  e.rep->putDebug(o);
  
  return o;
}

//  error detection

void Expr :: error(const char* msg) const
{
  //  output error message to stderr
  
  cerr << "Expr Error : " << msg << endl;
  
  //  unusual termination
  
  exit(1);
}

//  class ExprRep

//  constructor

ExprRep :: ExprRep() : Expr(ExprBase())
{
  rep      = this;
  refCount = 0;
  
//  uMSB = tinyLong;
//  lMSB = tinyLong;
//  sign = 0;
//  
//  degree = 0;
//  length = 0;
//  height = 0;
//  
  exStamp = 0;
  
//  appValue     = 0;
//  relPrecision = extULong(0);
//  absPrecision = tinyLong;
//  
  appStamp = 0;
}

//  the destructor

ExprRep :: ~ExprRep()
{}

//  letter-envelope

int ExprRep :: isRepThis() const
{
  return rep == this;
}

unsigned ExprRep :: incRefCount()
{
  return ++refCount;
}

unsigned ExprRep :: decRefCount()
{
  return --refCount;
}

//  assignment

ExprRep* ExprRep :: assign(Real)
{
  return 0;
}

ExprRep* ExprRep :: assign(ExprRep*)
{
  return 0;
}

ExprRep* ExprRep :: duplicate(ExprDict* d)
{
  return 0;
}

//  exact value

void ExprRep :: evalExValue()
{}

void ExprRep :: reEvalExValue()
{}

extLong ExprRep :: getUMSB() const
{
  if (rep == this)
    return uMSB;
  else
    return rep->getUMSB();
}

extLong ExprRep :: getLMSB() const
{
  if (rep == this)
    return lMSB;
  else
    return rep->getLMSB();
}

int ExprRep :: getSign() const
{
  if (rep == this)
    return sign;
  else
    return rep->getSign();
}

unsigned long ExprRep :: getDegree() const
{
  if (rep == this)
    return degree;
  else
    return rep->getDegree();
}

unsigned long ExprRep :: getLength() const
{
  if (rep == this)
    return length;
  else
    return rep->getLength();
}

//unsigned long ExprRep :: getHeight() const
//{
//  if (rep == this)
//   return height;
//  else
//    return rep->getHeight();
//}
//
unsigned ExprRep :: getExStamp() const
{
  if (rep == this)
    return exStamp;
  else
    return rep->getExStamp();
}

void ExprRep :: putExStamp(unsigned s)
{
  if (rep == this)
    exStamp = s;
  else
    rep->putExStamp(s);
}

//  approximation

void ExprRep :: evalAppValue(const extULong&, const extLong&)
{}

Real ExprRep :: incEvalAppValue(const extULong&, const extLong&)
{
  return Real(0);
}

void ExprRep :: reEvalAppValue(const extULong&, const extLong&)
{}

Real ExprRep :: incReEvalAppValue(const extULong&, const extLong&)
{
  return Real(0);
}

Real ExprRep :: getAppValue() const
{
  if (rep == this)
    return appValue;
  else
    return rep->getAppValue();
}

unsigned ExprRep :: getAppStamp() const
{
  if (rep == this)
    return appStamp;
  else
    return rep->getAppStamp();
}

void ExprRep :: putAppStamp(unsigned s)
{
  if (rep == this)
    appStamp = s;
  else
    rep->putAppStamp(s);
}

//  stream

ostream& ExprRep :: put(ostream& o)
{
  if (rep == this)
  {
    evalExValue();
    
    if (sign)
    {
      evalAppValue(defRelPrec, defAbsPrec);
      
      //o << "Expr(Parameter(" << appValue << "))";
      o << appValue;
    }
    else
      o << " zero ";
    
    return o;
  }
  else
    return rep->put(o);
}

static double lgTenP = 3.321928094887364;

ostream& ExprRep :: print(ostream& o, long d)
{
  if (rep == this)
  {
    evalExValue();
    
    if (sign)
    {
      extULong rr = 1;
      extULong ra = uMSB + (long)ceil((d + 2) * lgTenP) + 3;
      extULong r  = rr > ra ? rr : ra;
      
      extLong ar = - lMSB + 1;
      extLong aa = (long)ceil((d + 2) * lgTenP) + 2;
      extLong a  = ar > aa ? ar : aa;
      
      evalAppValue(r, a);
      
      long currDefPrtDgt = defPrtDgt;
      
      defPrtDgt = d;
//      o << "Expr(Parameter(";
      o << appValue;
//      o << "))";
      defPrtDgt = currDefPrtDgt;
    }
    else
      o << " zero ";
    
    return o;
  }
  else
    return rep->print(o, d);
}

ostream& ExprRep :: putDebug(ostream& o) const
{
  o << " ExprRep() " << endl << flush;
  
  return o;
}

//  class ParamRep

//  constructor

ParamRep :: ParamRep() : ExprRep(), exValue(Real())
{
  hasExValue = 0;
  
//  uMSB = lMSB = tinyLong;
//  sign = 0;
//  
//  degree = 0;
//  length = tinyLong;
//  height = tinyLong;
//  
//  exStamp = 0;
//  
//  appStamp = 0;
}

ParamRep :: ParamRep(Real r) : ExprRep(), exValue(r)
{
  hasExValue = 1;
  
//  uMSB = lMSB = exValue.MSB();
//  sign = exValue.sgn();
//  
//  degree = exValue.degree();
//  length = exValue.length();
//  height = exValue.height();
//  
//  exStamp = 0;
//  
//  appStamp = 0;
}

//  the destructor

ParamRep :: ~ParamRep()
{}

//  assignment

ExprRep* ParamRep :: assign(Real r)
{
  exValue = r;
  
  hasExValue = 1;
  
//  uMSB = lMSB = exValue.MSB();
//  sign = exValue.sgn();
//  
//  degree = exValue.degree();
//  length = exValue.length();
//  height = exValue.height();
//  
  exStamp += exStamp & 1;
  
  appStamp += appStamp & 1;
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  return rep = this;
}

ExprRep* ParamRep :: assign(ExprRep* p)
{
  unsigned pe = p->getExStamp();
  unsigned pa = p->getAppStamp();
  
  if (pe <= exStamp || pa <= appStamp)
  {
    unsigned long s = exStamp > appStamp ? exStamp : appStamp;
    
    if (pe > pa)
    {
      p->putExStamp(s + (s & 1) + (pa & 1) + pe - pa);
      p->putAppStamp(s + (s & 1) + (pa & 1));
    }
    else
    {
      p->putExStamp(s + (s & 1) + (pe & 1));
      p->putAppStamp(s + (s & 1) + (pe & 1) + pa - pe);
    }
  }
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  rep = p;
  rep->incRefCount();
  
  return this;
}

ExprRep* ParamRep :: duplicate(ExprDict* d)
{
  ExprRep* p;
  
  if (rep == this)
  {
    if (refCount == 1)
      p = new ParamRep(exValue);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
        d->insert(new ExprDictItem(this, p = new ParamRep(exValue)));
      else
        p = i->getInf();
    }
  }
  else
  {
    ExprRep* r = rep->duplicate(d);
    
    if (refCount == 1)
      p = new ParamRep(0);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
        d->insert(new ExprDictItem(this, p = new ParamRep(0)));
      else
        p = i->getInf();
    }
    
    p->rep = r;
    r->incRefCount();
  }
  
  return p;
}

//  exact value

void ParamRep :: evalExValue()
{
  if (rep == this)
  {
    if (hasExValue)
    {
      if (~ exStamp & 1)
      {
        uMSB = lMSB = exValue.MSB();
        sign = exValue.sgn();
        
        degree = exValue.degree();
        length = exValue.length();
//	height = exValue.height();
        
        exStamp++;
      }
    }
    else
      error("parameter has no exact value.");
  }
  else
    rep->evalExValue();
}

//  approximation

void ParamRep :: evalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  if (rep == this)
  {
    if (hasExValue)
    {
      if (~ appStamp & 1 || relPrecision < relPrec || absPrecision < absPrec)
      {
        appValue = exValue.approx(relPrec, absPrec);
	
        appStamp += (appStamp & 1 ? 2 : 1);
        
        if (relPrecision < relPrec)
          relPrecision = relPrec;
        
        if (absPrecision < absPrec)
          absPrecision = absPrec;
      }
    }
    else
      error("parameter has no exact value.");
  }
  else
    rep->evalAppValue(relPrec, absPrec);
}

Real ParamRep :: incEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  if (rep == this)
  {
    relPrecision += incRelPrec;
    absPrecision += incAbsPrec;
    
    Real currValue = appValue;
    
    appValue = exValue.approx(relPrecision, absPrecision);
    
    appStamp += (appStamp & 1 ? 2 : 1);
    
    return appValue - currValue;
  }
  else
    return rep->incEvalAppValue(incRelPrec, incAbsPrec);
}

//  stream

ostream& ParamRep :: putDebug(ostream& o) const
{
  o << "ParamRep(" << exValue << ")";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class UnaryOpRep

//  constructor

UnaryOpRep :: UnaryOpRep(ExprRep* c) : ExprRep(), child(c)
{
  if (c)
    child->incRefCount();
  
  srcExStamp = 0;
  
  srcAppStamp = 0;
  
//  exStamp = 0;
//  
//  appStamp = 0;
}

//  the destructor

UnaryOpRep :: ~UnaryOpRep()
{
  if (child && !child->decRefCount())
    delete child;
}

//  assignment

ExprRep* UnaryOpRep :: assign(Real r)
{
  ExprRep* p = new ParamRep(r);
  
  p->putExStamp(exStamp + (exStamp & 1));
  
  p->putAppStamp(appStamp + (appStamp & 1));
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  if (!child->decRefCount())
    delete child;
  
  child = 0;
  
  rep = p;
  rep->incRefCount();
  
  return p;
}

ExprRep* UnaryOpRep :: assign(ExprRep* p)
{
  unsigned pe = p->getExStamp();
  unsigned pa = p->getAppStamp();
  
  if (pe <= exStamp || pa <= appStamp)
  {
    unsigned long s = exStamp > appStamp ? exStamp : appStamp;
    
    if (pe > pa)
    {
      p->putExStamp(s + (s & 1) + (pa & 1) + pe - pa);
      p->putAppStamp(s + (s & 1) + (pa & 1));
    }
    else
    {
      p->putExStamp(s + (s & 1) + (pe & 1));
      p->putAppStamp(s + (s & 1) + (pe & 1) + pa - pe);
    }
  }
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  if (!child->decRefCount())
    delete child;
  
  child = 0;
  
  rep = p;
  rep->incRefCount();
  
  return this;
}

ExprRep* UnaryOpRep :: duplicate(ExprDict* d)
{
  ExprRep* p;
  
  if (rep == this)
  {
    ExprRep* c = child->duplicate(d);
    
    if (refCount == 1)
      p = duplicateRep(c);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
        d->insert(new ExprDictItem(this, p = duplicateRep(c)));
      else
        p = i->getInf();
    }
  }
  else
  {
    ExprRep* r = rep->duplicate(d);
    
    if (refCount == 1)
      p = duplicateRep(0);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
        d->insert(new ExprDictItem(this, p = duplicateRep(0)));
      else
        p = i->getInf();
    }
    
    p->rep = r;
    r->incRefCount();
  }
  
  return p;
}

ExprRep* UnaryOpRep :: duplicateRep(ExprRep*)
{
  return 0;
}

//  exact value

void UnaryOpRep :: evalExValue()
{
  if (rep == this)
  {
    evalChildExValue();
    
    unsigned childExStamp = getChildExStamp();
    
    if (srcExStamp < childExStamp)
    {
      srcExStamp = childExStamp;
      
      reEvalExValue();
    }
  }
  else
    rep->evalExValue();
}

void UnaryOpRep :: reEvalExValue()
{}

void UnaryOpRep :: evalChildExValue() const
{
  child->evalExValue();
}

extLong UnaryOpRep :: getChildUMSB() const
{
  return child->getUMSB();
}

extLong UnaryOpRep :: getChildLMSB() const
{
  return child->getLMSB();
}

int UnaryOpRep :: getChildSign() const
{
  return child->getSign();
}

unsigned long UnaryOpRep :: getChildDegree() const
{
  return child->getDegree();
}

unsigned long UnaryOpRep :: getChildLength() const
{
  return child->getLength();
}

//unsigned long UnaryOpRep :: getChildHeight() const
//{
//  return child->getHeight();
//}
//
unsigned UnaryOpRep :: getChildExStamp() const
{
  return child->getExStamp();
}

//  approximation

void UnaryOpRep :: evalAppValue(const extULong& relPrec,
				const extLong& absPrec)
{
  if (rep == this)
  {
    if (exStamp > appStamp || relPrecision < relPrec || absPrecision < absPrec)
    {
//      if (exStamp > appStamp)
        reEvalAppValue(relPrec, absPrec);
//      else
//        incEvalAppValue(relPrec - relPrecision, absPrec - absPrecision);
      
      if (relPrecision < relPrec)
        relPrecision = relPrec;
      
      if (absPrecision < absPrec)
        absPrecision = absPrec;
    }
  }
  else
    rep->evalAppValue(relPrec, absPrec);
}

Real UnaryOpRep :: incEvalAppValue(const extULong& incRelPrec,
				   const extLong& incAbsPrec)
{
  if (rep == this)
  {
    Real incValue = incReEvalAppValue(incRelPrec, incAbsPrec);
    
    appValue     += incValue;
    relPrecision += incRelPrec;
    absPrecision += incAbsPrec;
    
    return incValue;
  }
  else
    return rep->incEvalAppValue(incRelPrec, incAbsPrec);
}

void UnaryOpRep :: reEvalAppValue(const extULong&, const extLong&)
{}

Real UnaryOpRep :: incReEvalAppValue(const extULong&, const extLong&)
{
  return Real(0);
}

void UnaryOpRep :: evalChildAppValue(const extULong& relPrec,
				     const extLong& absPrec) const
{
  child->evalAppValue(relPrec, absPrec);
}

Real UnaryOpRep :: getChildAppValue() const
{
  return child->getAppValue();
}

unsigned UnaryOpRep :: getChildAppStamp() const
{
  return child->getAppStamp();
}

//  stream

ostream& UnaryOpRep :: putDebug(ostream& o) const
{
  if (child)
  {
    o << "UnaryOp(";
    child->putDebug(o);
    o << ")";
  }
  else
    o << "UnaryOp(NULL)";
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class NegRep

//  constructor

NegRep :: NegRep(ExprRep* c) : UnaryOpRep(c)
{
//  if (c)
//  {
//    degree = getChildDegree();
//    length = getChildLength();
//    height = getChildHeight();
//  }
}

//  the destructor

NegRep :: ~NegRep()
{}

//  assignement

ExprRep* NegRep :: duplicateRep(ExprRep* p)
{
  return new NegRep(p);
}

//  exact value

void NegRep :: reEvalExValue()
{
  uMSB = lMSB = getChildLMSB();
  sign = - getChildSign();
  
  degree = getChildDegree();
  length = getChildLength();
//  height = getChildHeight();
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void NegRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  evalChildAppValue(relPrec, absPrec);
  
  unsigned childAppStamp = getChildAppStamp();
  
  if (srcAppStamp < childAppStamp)
  {
    srcAppStamp = childAppStamp;
    
    appValue = - getChildAppValue();
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real NegRep :: incReEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& NegRep :: putDebug(ostream& o) const
{
  if (child)
  {
    o << "-(";
    child->putDebug(o);
    o << ")";
  }
  else
    o << "-NULL";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class SqrtRep

//  constructor

SqrtRep :: SqrtRep(ExprRep* c) : UnaryOpRep(c)
{
//  if (c)
//  {
//    unsigned long childDegree = getChildDegree();
//    
//    degree = childDegree * childDegree;
//    length = getChildLength();
//    height = getChildHeight();
//  }
}

//  the destructor

SqrtRep :: ~SqrtRep()
{}

//  assignment

ExprRep* SqrtRep :: duplicateRep(ExprRep* p)
{
  return new SqrtRep(p);
}

//  approximation

void SqrtRep :: reEvalExValue()
{
  extLong childUMSB = getChildUMSB();
  
  uMSB =
    childUMSB.isInfty() ?
      extLong(inftyLong) : extLong(childUMSB.asLong() >> 1);
  
  extLong childLMSB = getChildLMSB();
  
  lMSB =
    childLMSB.isInfty() ?
      extLong(inftyLong) : extLong(childLMSB.asLong() >> 1);
  
  sign = getChildSign();
  
  if (sign < 0)
    error("squareroot is called with negative operand.");
  
  unsigned long childDegree = getChildDegree();
  
  degree = childDegree * childDegree;
  length = getChildLength();
//  height = getChildHeight();
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void SqrtRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  extULong r = relPrec + relPrec + 8;
  extLong  a = absPrec + absPrec + 8;
  
  evalChildAppValue(r, a);
  
  unsigned childAppStamp = getChildAppStamp();
  
  if (srcAppStamp < childAppStamp)
  {
    srcAppStamp = childAppStamp;
    
    extLong pr = - lMSB + r;
    extLong p  = pr < a ? pr : a;
    
    appValue = getChildAppValue().sqrt(p);
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real SqrtRep :: incReEvalAppValue(const extULong& incRelPrec,
				  const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& SqrtRep :: putDebug(ostream& o) const
{
  if (child)
  {
    o << "sqrt(";
    child->putDebug(o);
    o << ")";
  }
  else
    o << "sqrt(NULL)";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class BinOpRep

//  constructor

BinOpRep :: BinOpRep(ExprRep* f, ExprRep* s)
  : ExprRep(), frstChild(f), scndChild(s)
{
  if (f && s)
  {
    frstChild->incRefCount();
    scndChild->incRefCount();
  }
  
  frstSrcExStamp = 0;
  scndSrcExStamp = 0;
  
  frstSrcAppStamp = 0;
  scndSrcAppStamp = 0;
}

//  the destructor

BinOpRep :: ~BinOpRep()
{
  if (frstChild && !frstChild->decRefCount())
    delete frstChild;
  
  if (scndChild && !scndChild->decRefCount())
    delete scndChild;
}

//  assignment

ExprRep* BinOpRep :: assign(Real r)
{
  ExprRep* p = new ParamRep(r);
  
  p->putExStamp(exStamp + (exStamp & 1));
  
  p->putAppStamp(appStamp + (appStamp & 1));
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  if (!frstChild->decRefCount())
    delete frstChild;
  
  frstChild = 0;
  
  if (!scndChild->decRefCount())
    delete scndChild;
  
  scndChild = 0;
  
  rep = p;
  rep->incRefCount();
  
  return p;
}

ExprRep* BinOpRep :: assign(ExprRep* p)
{
  unsigned pe = p->getExStamp();
  unsigned pa = p->getAppStamp();
  
  if (pe <= exStamp || pa <= appStamp)
  {
    unsigned long s = exStamp > appStamp ? exStamp : appStamp;
    
    if (pe > pa)
    {
      p->putExStamp(s + (s & 1) + (pa & 1) + pe - pa);
      p->putAppStamp(s + (s & 1) + (pa & 1));
    }
    else
    {
      p->putExStamp(s + (s & 1) + (pe & 1));
      p->putAppStamp(s + (s & 1) + (pe & 1) + pa - pe);
    }
  }
  
  if (!isRepThis() && !rep->decRefCount())
    delete rep;
  
  if (!frstChild->decRefCount())
    delete frstChild;
  
  frstChild = 0;
  
  if (!scndChild->decRefCount())
    delete scndChild;
  
  scndChild = 0;
  
  rep = p;
  rep->incRefCount();
  
  return this;
}

ExprRep* BinOpRep :: duplicate(ExprDict* d)
{
  ExprRep* p;
  
  if (rep == this)
  {
    ExprRep* f = frstChild->duplicate(d);
    ExprRep* s = scndChild->duplicate(d);
    
    if (refCount == 1)
      p = duplicateRep(f, s);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
	d->insert(new ExprDictItem(this, p = duplicateRep(f, s)));
      else
	p = i->getInf();
    }
  }
  else
  {
    ExprRep* r = rep->duplicate(d);
    
    if (refCount == 1)
      p = duplicateRep(0, 0);
    else
    {
      ExprDictItem* i = d->lookUp(this);
      
      if (!i)
	d->insert(new ExprDictItem(this, p = duplicateRep(0, 0)));
      else
	p = i->getInf();
    }
    
    p->rep = r;
    r->incRefCount();
  }
  
  return p;
}

ExprRep* BinOpRep :: duplicateRep(ExprRep*, ExprRep*)
{
  return 0;
}

//  exact value

void BinOpRep :: evalExValue()
{
  if (rep == this)
  {
    evalFrstChildExValue();
    evalScndChildExValue();
    
    unsigned frstChildExStamp = getFrstChildExStamp();    
    unsigned scndChildExStamp = getScndChildExStamp();
    
    if (frstSrcExStamp < frstChildExStamp
	|| scndSrcExStamp < scndChildExStamp)
    {
      frstSrcExStamp = frstChildExStamp;
      scndSrcExStamp = scndChildExStamp;
      
      reEvalExValue();
    }
  }
  else
    rep->evalExValue();
}

void BinOpRep :: reEvalExValue()
{}

void BinOpRep :: evalFrstChildExValue() const
{
  frstChild->evalExValue();
}

void BinOpRep :: evalScndChildExValue() const
{
  scndChild->evalExValue();
}

extLong BinOpRep :: getFrstChildUMSB() const
{
  return frstChild->getUMSB();
}

extLong BinOpRep :: getScndChildUMSB() const
{
  return scndChild->getUMSB();
}

extLong BinOpRep :: getFrstChildLMSB() const
{
  return frstChild->getLMSB();
}

extLong BinOpRep :: getScndChildLMSB() const
{
  return scndChild->getLMSB();
}

int BinOpRep :: getFrstChildSign() const
{
  return frstChild->getSign();
}

int BinOpRep :: getScndChildSign() const
{
  return scndChild->getSign();
}

unsigned long BinOpRep :: getFrstChildDegree() const
{
  return frstChild->getDegree();
}

unsigned long BinOpRep :: getScndChildDegree() const
{
  return scndChild->getDegree();
}

unsigned long BinOpRep :: getFrstChildLength() const
{
  return frstChild->getLength();
}

unsigned long BinOpRep :: getScndChildLength() const
{
  return scndChild->getLength();
}

//unsigned long BinOpRep :: getFrstChildHeight() const
//{
//  return frstChild->getHeight();
//}
//
//unsigned long BinOpRep :: getScndChildHeight() const
//{
//  return scndChild->getHeight();
//}
//
unsigned BinOpRep :: getFrstChildExStamp() const
{
  return frstChild->getExStamp();
}

unsigned BinOpRep :: getScndChildExStamp() const
{
  return scndChild->getExStamp();
}

//  approximation

void BinOpRep :: evalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  if (rep == this)
  {
    if (exStamp > appStamp || relPrecision < relPrec || absPrecision < absPrec)
    {
//      if (exStamp > appStamp)
	reEvalAppValue(relPrec, absPrec);
//      else
//	incEvalAppValue(relPrec - relPrecision, absPrec - absPrecision);
      
      if (relPrecision < relPrec)
	relPrecision = relPrec;
      
      if (absPrecision < absPrec)
	absPrecision = absPrec;
    }
  }
  else
    rep->evalAppValue(relPrec, absPrec);
}

Real BinOpRep :: incEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  if (rep == this)
  {
    Real incValue = incReEvalAppValue(incRelPrec, incAbsPrec);
    
    appValue     += incValue;
    relPrecision += incRelPrec;
    absPrecision += incAbsPrec;
    
    return incValue;
  }
  else
    return rep->incEvalAppValue(incRelPrec, incAbsPrec);
}

void BinOpRep :: reEvalAppValue(const extULong&, const extLong&)
{}

Real BinOpRep :: incReEvalAppValue(const extULong&, const extLong&)
{
  return Real(0);
}

void BinOpRep :: evalFrstChildAppValue(const extULong& relPrec,
				       const extLong& absPrec) const
{
  frstChild->evalAppValue(relPrec, absPrec);
}

void BinOpRep :: evalScndChildAppValue(const extULong& relPrec,
				       const extLong& absPrec) const
{
  scndChild->evalAppValue(relPrec, absPrec);
}

Real BinOpRep :: getFrstChildAppValue() const
{
  return frstChild->getAppValue();
}

Real BinOpRep :: getScndChildAppValue() const
{
  return scndChild->getAppValue();
}

unsigned BinOpRep :: getFrstChildAppStamp() const
{
  return frstChild->getAppStamp();
}

unsigned BinOpRep :: getScndChildAppStamp() const
{
  return scndChild->getAppStamp();
}

//  stream

ostream& BinOpRep :: putDebug(ostream& o) const
{
  o << "BinOp(";
  
  if (frstChild)
    frstChild->putDebug(o);
  else
    o << "NULL";
  
  o << " , ";
  
  if (scndChild)
  {
    scndChild->putDebug(o);
    o << ")";
  }
  else
    o << "NULL)";
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class AddRep

//  constructor

AddRep :: AddRep(ExprRep* f, ExprRep* s) : BinOpRep(f, s)
{
//  if (f && s)
//  {
//    unsigned long df    = getFrstChildDegree();
//    unsigned long ds    = getScndChildDegree();
//    unsigned long md    = df < ds ? df : ds;
//    unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//    unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//    unsigned long mdld  = dflds < dsldf ? dflds : dsldf;
//    unsigned long lf    = getFrstChildLength();
//    unsigned long ls    = getScndChildLength();
//    unsigned long hf    = getFrstChildHeight();
//    unsigned long hs    = getScndChildHeight();
//    
//    degree = df * ds;
//    length = df * ls + ds * lf + degree + md + df + ds - 1;
//    height = df * hs + ds * hf + degree + mdld + 2 * (df + ds) - 1;
//  }
}

//  the destructor

AddRep :: ~AddRep()
{}

//  assignment

ExprRep* AddRep :: duplicateRep(ExprRep* f, ExprRep* s)
{
  return new AddRep(f, s);
}

//  exact value

void AddRep :: reEvalExValue()
{
  int sf = getFrstChildSign();
  int ss = getScndChildSign();
  
  unsigned long df    = getFrstChildDegree();
  unsigned long ds    = getScndChildDegree();
  unsigned long md    = df < ds ? df : ds;
//  unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//  unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//  unsigned long mdld  = dflds < dsldf ? dflds : dsldf;
  unsigned long lf    = getFrstChildLength();
  unsigned long ls    = getScndChildLength();
//  unsigned long hf    = getFrstChildHeight();
//  unsigned long hs    = getScndChildHeight();
  
  degree = df * ds;
  length = df * ls + ds * lf + degree + md + df + ds - 1;
//  height = df * hs + ds * hf + degree + mdld + 2 * (df + ds) - 1;
  
  if (sf)
  {
    extLong lf = getFrstChildLMSB();
    extLong uf = getFrstChildUMSB();
    
    if (ss)
    {
      extLong ls = getScndChildLMSB();
      extLong l  = lf > ls ? lf : ls;
      
      extLong us = getScndChildUMSB();
      extLong u  = uf > us ? uf : us;
      
      if (sf * ss > 0)
      {
	uMSB = u + 1;
	lMSB = l;
	sign = sf;
      }
      else
	//  sf * ss < 0
      {
	uMSB = u;
	
	if (lf >= us + 2)
	{
	  lMSB = lf - 1;
	  sign = sf;
	}
	else if (ls >= uf + 2)
	{
	  lMSB = ls - 1;
	  sign = ss;
	}
	else
	{
//	  extULong h  = 2;
//	  extULong rf = uf - l + 5;
//	  extULong rs = us - l + 5;
//	  
//	  evalFrstChildAppValue(rf, inftyLong);
//	  evalScndChildAppValue(rs, inftyLong);
//	  Real newValue = getFrstChildAppValue() + getScndChildAppValue();
//	  
//	  while (newValue.isZeroIn()
//		 && newValue.clLgErr() + extLong(length) + 2 > 0)
//	  {
//	    rf += h;
//	    rs += h;
//	    
//	    evalFrstChildAppValue(rf, inftyLong);
//	    evalScndChildAppValue(rs, inftyLong);
//	    newValue = getFrstChildAppValue() + getScndChildAppValue();
//	    
//	    h += h;
//	  }
//	  
	  extULong rf = uf + extLong(length) + 6;
	  extULong rs = uf + extLong(length) + 6;
	  extLong  a  = extLong(length) + 5;
	  
	  evalFrstChildAppValue(rf, inftyLong);
	  evalScndChildAppValue(rs, inftyLong);
	  
	  Real newValue = getFrstChildAppValue() + getScndChildAppValue();
	  
	  if (!newValue.isZeroIn())
	  {
	    lMSB = newValue.lMSB();
	    sign = newValue.sgn();
	  }
	  else
	  {
	    lMSB = tinyLong;
	    sign = 0;
	  }
	}
      }
    }
    else
      //  ss == 0
    {
      uMSB = uf;
      lMSB = lf;
      sign = sf;
    }
  }
  else
    //  sf  == 0
    if (ss)
    {
      uMSB = getScndChildUMSB();
      lMSB = getScndChildLMSB();
      sign = ss;
    }
    else
      //  ss == 0
    {
      uMSB = lMSB = tinyLong;
      sign = 0;
    }
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void AddRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  extULong rf = getFrstChildUMSB() - lMSB + relPrec + 4;
  extULong rs = getScndChildUMSB() - lMSB + relPrec + 4;
  extLong  a  = absPrec + 3;
  
  evalFrstChildAppValue(rf, a);
  evalScndChildAppValue(rs, a);
  
  unsigned frstChildAppStamp = getFrstChildAppStamp();
  unsigned scndChildAppStamp = getScndChildAppStamp();
  
  if (frstSrcAppStamp < frstChildAppStamp
      || scndSrcAppStamp < scndChildAppStamp)
  {
    frstSrcAppStamp = frstChildAppStamp;
    scndSrcAppStamp = scndChildAppStamp;
    
    appValue = getFrstChildAppValue() + getScndChildAppValue();
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real AddRep :: incReEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& AddRep :: putDebug(ostream& o) const
{
  if (frstChild)
    frstChild->putDebug(o);
  else
    o << "NULL";
  
  o << " + ";
  
  if (scndChild)
    scndChild->putDebug(o);
  else
    o << "NULL";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class SubRep

//  constructor

SubRep :: SubRep(ExprRep* f, ExprRep* s) : BinOpRep(f, s)
{
//  if (f && s)
//  {
//    unsigned long df    = getFrstChildDegree();
//    unsigned long ds    = getScndChildDegree();
//    unsigned long md    = df < ds ? df : ds;
//    unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//    unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//    unsigned long mdld  = dflds < dsldf ? dflds : dsldf;
//    unsigned long lf    = getFrstChildLength();
//    unsigned long ls    = getScndChildLength();
//    unsigned long hf    = getFrstChildHeight();
//    unsigned long hs    = getScndChildHeight();
//    
//    degree = df * ds;
//    length = df * ls + ds * lf + degree + md + df + ds - 1;
//    height = df * hs + ds * hf + degree + mdld + 2 * (df + ds) - 1;
//  }
}

//  the Destructor

SubRep :: ~SubRep()
{}

//  assignment

ExprRep* SubRep :: duplicateRep(ExprRep* f, ExprRep* s)
{
  return new SubRep(f, s);
}

//  exact value

void SubRep :: reEvalExValue()
{
  int sf = getFrstChildSign();
  int ss = getScndChildSign();
  
  unsigned long df    = getFrstChildDegree();
  unsigned long ds    = getScndChildDegree();
  unsigned long md    = df < ds ? df : ds;
//  unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//  unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//  unsigned long mdld  = dflds < dsldf ? dflds : dsldf;
  unsigned long lf    = getFrstChildLength();
  unsigned long ls    = getScndChildLength();
//  unsigned long hf    = getFrstChildHeight();
//  unsigned long hs    = getScndChildHeight();
  
  degree = df * ds;
  length = df * ls + ds * lf + degree + md + df + ds - 1;
//  height = df * hs + ds * hf + degree + mdld + 2 * (df + ds) - 1;
  
  if (sf)
  {
    extLong lf = getFrstChildLMSB();
    extLong uf = getFrstChildUMSB();
    
    if (ss)
    {
      extLong ls = getScndChildLMSB();
      extLong l  = lf > ls ? lf : ls;
      
      extLong us = getScndChildUMSB();
      extLong u  = uf > us ? uf : us;
      
      if (sf * ss < 0)
      {
	uMSB = u + 1;
	lMSB = l;
	sign = sf;
      }
      else
	//  sf * ss > 0
      {
	uMSB = u;
	
	if (lf >= us + 2)
	{
	  lMSB = lf - 1;
	  sign = sf;
	}
	else if (ls >= uf + 2)
	{
	  lMSB = ls - 1;
	  sign = - ss;
	}
	else
	{
//	  extULong h  = 1;
//	  extULong rf = uf - l + 5;
//	  extULong rs = us - l + 5;
//	  
//	  evalFrstChildAppValue(rf, inftyLong);
//	  evalScndChildAppValue(rs, inftyLong);
//	  Real newValue = getFrstChildAppValue() - getScndChildAppValue();
//	  
//	  while (newValue.isZeroIn()
//		 && newValue.clLgErr() + extLong(length) + 2 > 0)
//	  {
//	    rf += h;
//	    rs += h;
//	    
//	    evalFrstChildAppValue(rf, inftyLong);
//	    evalScndChildAppValue(rs, inftyLong);
//	    newValue = getFrstChildAppValue() - getScndChildAppValue();
//	    
//	    h += h;
//	  }
//	  
	  extULong rf = uf + extLong(length) + 6;
	  extULong rs = uf + extLong(length) + 6;
	  extLong  a  = extLong(length) + 5;
	  
	  evalFrstChildAppValue(rf, inftyLong);
	  evalScndChildAppValue(rs, inftyLong);
	  
	  Real newValue = getFrstChildAppValue() - getScndChildAppValue();
	  
	  if (!newValue.isZeroIn())
	  {
	    lMSB = newValue.lMSB();
	    sign = newValue.sgn();
	  }
	  else
	  {
	    lMSB = extLong(tinyLong);
	    sign = 0;
	  }
	}
      }
    }
    else
      //  ss == 0
    {
      uMSB = uf;
      lMSB = lf;
      sign = sf;
    }
  }
  else
    //  sf  == 0
    if (ss)
    {
      uMSB = getScndChildUMSB();
      lMSB = getScndChildLMSB();
      sign = - ss;
    }
    else
      //  ss == 0
    {
      uMSB = lMSB = tinyLong;
      sign = 0;
    } 
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void SubRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  extULong rf = getFrstChildUMSB() - lMSB + relPrec + 4;
  extULong rs = getScndChildUMSB() - lMSB + relPrec + 4;
  extLong  a  = absPrec + 3;
  
  evalFrstChildAppValue(rf, a);
  evalScndChildAppValue(rs, a);
  
  unsigned frstChildAppStamp = getFrstChildAppStamp();
  unsigned scndChildAppStamp = getScndChildAppStamp();
  
  if (frstSrcAppStamp < frstChildAppStamp
      || scndSrcAppStamp < scndChildAppStamp)
  {
    frstSrcAppStamp = frstChildAppStamp;
    scndSrcAppStamp = scndChildAppStamp;
    
    appValue = getFrstChildAppValue() - getScndChildAppValue();
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real SubRep :: incReEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& SubRep :: putDebug(ostream& o) const
{
  if (frstChild)
    frstChild->putDebug(o);
  else
    o << "NULL";
  
  o << " - ";
  
  if (scndChild)
    scndChild->putDebug(o);
  else
    o << "NULL";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
}

//  class MultRep

//  constructor

MultRep :: MultRep(ExprRep* f, ExprRep* s) : BinOpRep(f, s)
{
//  if (f && s)
//  {
//    unsigned long df = getFrstChildDegree();
//    unsigned long ds = getScndChildDegree();
//    unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//    unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//    unsigned long lf = getFrstChildLength();
//    unsigned long ls = getScndChildLength();
//    unsigned long hf = getFrstChildHeight();
//    unsigned long hs = getScndChildHeight();
//    
//    degree = df * ds;
//    length = df * ls + ds * lf + df + ds - 1;
//    height = df * hs + ds * hf + dflds + dsldf + 2 * (df + ds) - 1;
//  }
}

//  the destructor

MultRep :: ~MultRep ()
{}
  
//  assignment

ExprRep* MultRep :: duplicateRep(ExprRep* f, ExprRep* s)
{
  return new MultRep(f, s);
}

//  exact value

void MultRep :: reEvalExValue()
{
  uMSB = getFrstChildUMSB() + getScndChildUMSB() + 1;
  lMSB = getFrstChildLMSB() + getScndChildLMSB();
  sign = getFrstChildSign() * getScndChildSign();
  
  unsigned long df = getFrstChildDegree();
  unsigned long ds = getScndChildDegree();
//  unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//  unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
  unsigned long lf = getFrstChildLength();
  unsigned long ls = getScndChildLength();
//  unsigned long hf = getFrstChildHeight();
//  unsigned long hs = getScndChildHeight();
  
  degree = df * ds;
  length = df * ls + ds * lf + df + ds - 1;
//  height = df * hs + ds * hf + dflds + dsldf + 2 * (df + ds) - 1;
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void MultRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  extULong r   = relPrec + 4;
  extLong  afr = - getFrstChildLMSB() + 1;
  extLong  afa = getScndChildUMSB() + absPrec + 5;
  extLong  af  = afr > afa ? afr : afa;
  extLong  asr = - getScndChildLMSB() + 1;
  extLong  asa = getFrstChildUMSB() + absPrec + 5;
  extLong  as  = asr > asa ? asr : asa;
  
  evalFrstChildAppValue(r, af);
  evalScndChildAppValue(r, as);
  
  unsigned frstChildAppStamp = getFrstChildAppStamp();
  unsigned scndChildAppStamp = getScndChildAppStamp();
  
  if (frstSrcAppStamp < frstChildAppStamp
      || scndSrcAppStamp < scndChildAppStamp)
  {
    frstSrcAppStamp = frstChildAppStamp;
    scndSrcAppStamp = scndChildAppStamp; 
    
    appValue = getFrstChildAppValue() * getScndChildAppValue();
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real MultRep :: incReEvalAppValue(const extULong& incRelPrec,
				  const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& MultRep :: putDebug(ostream& o) const
{
  if (frstChild)
    frstChild->putDebug(o);
  else
    o << "NULL";
  
  o << " * ";
  
  if (scndChild)
    scndChild->putDebug(o);
  else
    o << "NULL";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

//  class DivRep

//  constructor

DivRep :: DivRep(ExprRep* f, ExprRep* s) : BinOpRep(f, s)
{
//  if (f && s)
//  {
//    unsigned long df = getFrstChildDegree();
//    unsigned long ds = getScndChildDegree();
//    unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//    unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
//    unsigned long lf = getFrstChildLength();
//    unsigned long ls = getScndChildLength();
//    unsigned long hf = getFrstChildHeight();
//    unsigned long hs = getScndChildHeight();
//    
//    degree = df * ds;
//    length = df * ls + ds * lf + df + ds - 1;
//    height = df * hs + ds * hf + dflds + dsldf + 2 * (df + ds) - 1;
//  }
}

//  the destructor

DivRep :: ~DivRep ()
{}

//  assignment

ExprRep* DivRep :: duplicateRep(ExprRep* f, ExprRep* s)
{
  return new DivRep(f, s);
}

//  exact value

void DivRep :: reEvalExValue()
{
  uMSB = getFrstChildUMSB() - getScndChildLMSB();
  lMSB = getFrstChildLMSB() - getScndChildUMSB() - 1;
  sign = getFrstChildSign() * getScndChildSign();
  
  if (!sign)
    error("zero divisor.");
  
  unsigned long df = getFrstChildDegree();
  unsigned long ds = getScndChildDegree();
//  unsigned long dflds = df * (flrLg(ds + 1) >> 1);
//  unsigned long dsldf = ds * (flrLg(df + 1) >> 1);
  unsigned long lf = getFrstChildLength();
  unsigned long ls = getScndChildLength();
//  unsigned long hf = getFrstChildHeight();
//  unsigned long hs = getScndChildHeight();
  
  degree = df * ds;
  length = df * ls + ds * lf + df + ds - 1;
//  height = df * hs + ds * hf + dflds + dsldf + 2 * (df + ds) - 1;
  
  exStamp = exStamp > appStamp ? exStamp + 1 : appStamp + 1;
}

//  approximation

void DivRep :: reEvalAppValue(const extULong& relPrec, const extLong& absPrec)
{
  extULong rr  = relPrec + 7;
  extULong ra  = uMSB + absPrec + 8;
  extULong ra2 = ra.asLong() > 2 ? ra : extULong(2);
  extULong r   = rr < ra2 ? rr : ra2;
  extLong  af  = - getFrstChildLMSB() + r;
  extLong  as  = - getScndChildLMSB() + r;
  
  evalFrstChildAppValue(r, af);
  evalScndChildAppValue(r, as);
  
  unsigned frstChildAppStamp = getFrstChildAppStamp();
  unsigned scndChildAppStamp = getScndChildAppStamp();
  
  extULong pr = relPrec + 6;
  extULong pa = uMSB + absPrec + 7;
  extULong p  = pr < pa ? pr : pa;
  
  if (frstSrcAppStamp < frstChildAppStamp
      || scndSrcAppStamp < scndChildAppStamp)
  {
    frstSrcAppStamp = frstChildAppStamp;
    scndSrcAppStamp = scndChildAppStamp; 
    
    appValue = getFrstChildAppValue().div(getScndChildAppValue(), p);
    
    appStamp = appStamp > exStamp ? appStamp + 1 : exStamp + 1;
  }
}

Real DivRep :: incReEvalAppValue(const extULong& incRelPrec,
				 const extLong& incAbsPrec)
{
  return Real(0);
}

//  stream

ostream& DivRep :: putDebug(ostream& o) const
{
  if (frstChild)
    frstChild->putDebug(o);
  else
    o << "NULL";
  
  o << " / ";
  
  if (scndChild)
    scndChild->putDebug(o);
  else
    o << "NULL";
  
//  o << " : " << degree << " " << length;
  
  if (!isRepThis())
  {
    o << endl;
    rep->putDebug(o);
  }
  
  return o;
}

