//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: BigFloat.h
//    class BigFloat
//

#ifndef _BIGFLOAT_H
#define _BIGFLOAT_H

#include <iostream.h>
#include <Rational.h>
#include "Defs.h"
#include "extLong.h"
#include "BigInt.h"

//  forward reference

class BigFloat;

//  class BigFloatRep

class BigFloatRep
{
friend class BigFloat;
  
  BigInt        m;
  unsigned long err;
  long          exp;
  
  unsigned      refCount;
  
  //  constructors
  
  BigFloatRep(int);
  BigFloatRep(long);
  BigFloatRep(double);
  BigFloatRep(const BigInt&, unsigned long, long);
  
  Rational Rationalize() const;
  
  //  the destructor
  
  ~BigFloatRep();
  
  //  approximation
  
  void trunc(const BigInt&, const extULong&, const extLong&);
  
  void truncM(const BigFloatRep&, const extULong&, const extLong&);
  void approx(const BigFloatRep&, const extULong&, const extLong&);
  
  void div(const BigInt&, const BigInt&, const extULong&, const extLong&);
  void approx(const Rational&, const extULong&, const extLong&);
  
  //  error-normalization
  
  void normal();
  void bigNormal(BigInt&);
  
  //  arithmetics
  
  void add(const BigFloatRep&, const BigFloatRep&);
  void sub(const BigFloatRep&, const BigFloatRep&);
  void mul(const BigFloatRep&, const BigFloatRep&);
  void div(const BigFloatRep&, const BigFloatRep&, const extULong&);
  
  //  squareroot
  
  void sqrt(const BigInt&, const extLong& a);
  void sqrt(const BigFloatRep&, const extLong& a);
  
  //  comparison
  
  int compareMExp(const BigFloatRep&) const;
  
  //  bultin functions
  
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  
  int     isZeroIn() const;
  int     signM() const;
  
  //  stream operators
  
  ostream& operator <<(ostream&) const;
  
  //  error detection
  
  void error(const char*) const;
};

//  class BigFloat

class BigFloat
{
  BigFloatRep* rep;
  
  //  comparisons
  
  int compare(const BigFloat&) const;
  
public:
  
  //  constructors
  
  BigFloat();
  BigFloat(int);
  BigFloat(long);
  BigFloat(double);
  BigFloat(const BigInt&, unsigned long = 0, long = 0);
  
  operator Rational() const;
  
  BigFloat(const BigFloat&);
  
  //  the destructor
  
  ~BigFloat();
  
  //  assignment operator
  
  BigFloat& operator =(const BigFloat&);
  
  //  approximation
  
  void approx(const BigInt&, const extULong&, const extLong&);
  void approx(const BigFloat&, const extULong&, const extLong&);
  void approx(const Rational&, const extULong&, const extLong&);
  
  //  unary minus
  
  BigFloat operator -() const;
  
  //  arithmetics
  
  BigFloat operator +(const BigFloat&) const;
  friend BigFloat operator +(const BigFloat&, const BigFloat&);
  
  BigFloat operator -(const BigFloat&) const;
  friend BigFloat operator -(const BigFloat&, const BigFloat&);
  
  BigFloat operator *(const BigFloat&) const;
  friend BigFloat operator *(const BigFloat&, const BigFloat&);
  
  BigFloat div(const BigFloat&, const extULong&) const;
  friend BigFloat operator /(const BigFloat&, const BigFloat&);
  
  //  squareroot
  
  BigFloat sqrt(const extLong&) const;
  friend BigFloat sqrt(const BigFloat&);
  
  //  comparisons
  
  friend int operator ==(const BigFloat&, const BigFloat&);
  friend int operator !=(const BigFloat&, const BigFloat&);
  friend int operator  <(const BigFloat&, const BigFloat&);
  friend int operator <=(const BigFloat&, const BigFloat&);
  friend int operator  >(const BigFloat&, const BigFloat&);
  friend int operator >=(const BigFloat&, const BigFloat&);
  
  //  arithmetic and assignment opeartors
  
  BigFloat& operator +=(const BigFloat&);
  BigFloat& operator -=(const BigFloat&);
  BigFloat& operator *=(const BigFloat&);
  BigFloat& operator /=(const BigFloat&);
  
  //  builtin function
  
  int     isExact() const;
  extLong lMSB() const;
  extLong MSB() const;
  extLong flrLgErr() const;
  extLong clLgErr() const;
  int     isZeroIn() const;
  
  int sign() const;
  friend int sign(const BigFloat&);
  
  BigFloat abs() const;
  friend BigFloat abs(const BigFloat&);
  
  //  stream operators
  
  ostream& operator <<(ostream&) const;
  friend ostream& operator <<(ostream&, const BigFloat&);
};

#endif

