function xstep = qcbound(x,dx,blk)
% QCBOUND  Computes the min of the steps from x_i to the
%          boundary of the Lorentz cones
%
% xstep = qcbound(x,dx,blk)
%
%  So we need
%  (*)      xi + alpha dxi >= 0,
%  for each block 1 <= i <= length(blk)
%  Let ni = blk(i) and
%     a = dxi(1)^2 - dxi(2:ni)'*dxi(2:ni)
%     b = xi(1)*dxi(1) - xi(2:ni)'*dxi(2:ni)
%     c = xi(1)^2 - xi(2:ni)'*xi(2:ni) >= 0, since xi >= 0
%  Then (*) is equivalent to
%
%     a alpha^2 + 2 b alpha + c >= 0,  where c >= 0
%
%  Thus we have
%  a = 0, b >= 0                   ==> alpha = Inf
%  a = 0, b < 0                    ==> alpha = -c/(2 b)
%  a > 0, b^2 - a*c < 0 or b >= 0  ==> alpha = Inf
%  a > 0, b^2 - a*c >= 0 and b < 0 ==> alpha = -(b+sqrt(b^2-a*c))/a
%  a < 0,                          ==> alpha = -(b+sqrt(b^2-a*c))/a
%
% input variables:
%     - x         vector in the product of Lorentz cones
%     - dx        direction
%     - blk       block info vector
%
% output variables:
%     - xstep     stepsize to the boundary of the positive orthant

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified : 7/2/97
%
 nblk = length(blk);
 valpha = zeros(nblk,1);
 fin = 0;
 for i = 1:nblk,
    start = fin + 1;
    fin = fin + blk(i);
    x1 = x(start);
    dx1 = dx(start);
    start = start + 1;
    xbar = x(start:fin);
    dxbar = dx(start:fin);
    a = dx1*dx1- dxbar'*dxbar;
    b = x1*dx1 - xbar'*dxbar;
    c = x1*x1 - xbar'*xbar;
    if a == 0
       if b >= 0
          valpha(i) = Inf;
       else
          valpha(i) = -c/(2*b);
       end
    else
       d = b*b - a*c;
       if a > 0
          if d < 0
             valpha(i) = Inf;
          elseif  b >= 0
             valpha(i) = Inf;
          else
             t = -(b - sqrt(d));  % careful about cancellation: b < 0; root = t/a
             valpha(i) = c/t;     % we want the other root; use: product of roots = c/a
          end
       else
          if b >= 0
             valpha(i) = -(b + sqrt(d))/a;
          else
             t = -(b - sqrt(d));  % careful about cancellation: b < 0; root = t/a
             valpha(i) = c/t;     % we want the other root; use: product of roots = c/a
          end
       end
    end
 end
 I = valpha < 0;
 valpha(I) = Inf;
 xstep = min(valpha);
%
% END function
