/*
 * arwimul.c - mex file: implements multiplication of a vector v
 *             by the inverse of an arrow matrix of a vector x;
 *             this is done blockwise where blk is the block
 %             structure vector.
 *
 * synopsis:   w = arwimul(x,v,blk)
 *
 * Single block case:
 * if x = (x0,x1,...,xn) then
 *             [ x0 x1 x2 ... xn ]
 *             [ x1 x0  0 ...  0 ]
 *    arw(x) = [ x2  0 x0 ...  0 ]
 *             [      ...        ]
 *             [ xn  0  0 ... x0 ]
 * Let xbar = (0,x1,...,xn) (or (x1,...,xn) depending upon context),
 * let e0 = (1,0,...,0) of length n+1,  and let r = x0^2 - ||xbar||^2.
 * We have
 *    arw(x)^{-1} = 1/r [   x0         -xbar^T         ]
 *                      [ -xbar  1/x0(rI + xbar*xbar^T ]
 *
 *   1      1  ||xbar||^2                                   1
 * = -- I + -{ ---------- e0*e0^T - e0*xbar^T - xbar*e0^T + -- xbar*xbar^T }
 *   x0     r       x0                                      x0
 *
 * inputs:
 *     - x         vector of quadratic blocks
 *     - v         vector of length sum(blk)
 *     - blk       block structure vector
 *
 * output:
 *     - w         vector of length sum(blk)
 *
 * SDPPACK Version 0.9 BETA
 * Copyright (c) 1997 by
 * F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
 * Last modified : 5/28/97
 */
#include <math.h>
#include "mex.h"

/* Input Arguments */
#define  x_IN     prhs[0]
#define  v_IN     prhs[1]
#define  blk_IN   prhs[2]

/* Output Arguments */
#define  w_OUT    plhs[0]

#if !defined(max)
#define  max(A, B)   ((A) > (B) ? (A) : (B))
#endif

#if !defined(min)
#define  min(A, B)   ((A) < (B) ? (A) : (B))
#endif

static void arwimul(
   double  *wpr,
   double  *xpr,
   double  *vpr,
   int  n,
   double  *blk,
   int  nblk,
   int  maxbsize,
   int eltsize
)
{
   int i,j,bsize;
   double x0,v0,s,t,alpha;
   double *widx,*xidx,*vidx;     /* will point to the start of the blocks */
   double *xblkpr,*vblkpr,*lhspr;
   mxArray *xblk,*vblk,*plhs[1],*prhs[2];

/* Create temporary matrices for the blocks */
   xblk = mxCreateDoubleMatrix(maxbsize,1,mxREAL);
   vblk = mxCreateDoubleMatrix(maxbsize,1,mxREAL);
   xblkpr = mxGetPr(xblk);
   vblkpr = mxGetPr(vblk);

   widx = wpr;    /* initially widx, xidx, and vidx point the beginning */
   xidx = xpr;    /* of the arrays wpr, xpr, and vpr respectively */
   vidx = vpr;
   for(i = 0; i < nblk; i++) {
      bsize = blk[i];
/*
 * start of computation
 */
      memcpy(xblkpr,xidx,bsize*eltsize);   /* copy x block to xblkpr */
/*
 * save x[0] to x0 and then set x[0] to 0
 */
      x0 = xblkpr[0];
      xblkpr[0] = 0.0;
/*
 * first compute s = xbar'*xbar
 */
      memcpy(vblkpr,xblkpr,bsize*eltsize);   /* copy x block to vblkpr */
      mxSetM(vblk,bsize);     /* a column vector */
      mxSetN(vblk,1);
      mxSetM(xblk,1);         /* set the sizes of xblk to (1,bsize), so it */
      mxSetN(xblk,bsize);     /* is the transpose of the block of x */
      prhs[0] = xblk;         /* xblk is first input parameter */
      prhs[1] = vblk;         /* vblk is second input parameter */
      mexCallMATLAB(1,plhs,2,prhs,"*");   /* compute xbar'*xbar */
      lhspr = mxGetPr(plhs[0]);
      s = lhspr[0];                       /* and save result in s */
/*
 * next compute t = xbar'*v
 */
      memcpy(vblkpr,vidx,bsize*eltsize);  /* copy block of v to vblkpr */
      v0 = vblkpr[0];                     /* set v0   */
      mexCallMATLAB(1,plhs,2,prhs,"*");   /* compute xbar'*v */
      lhspr = mxGetPr(plhs[0]);
      t = lhspr[0];                       /* and save result in t */
/*
 * now set alpha = square of quadratic norm of x, i.e. xo^2 - ||xbar||^2
 */
      alpha = x0*x0 - s;
/*
 *  set w = t*xbar/x0 - v0*xbar, w[0] = v0*s/x0 - t, and then divide by alpha
 */
      memcpy(widx,xblkpr,bsize*eltsize); /* copy xbar into w */
      widx[0] = (v0*s/x0 - t)/alpha;     /* s is no longer needed so can be reused */
      s = t/x0 - v0;
      for(j = 1; j < bsize; j++)
         widx[j] *= s/alpha;
/*
 * finally set w = w + v/x0
 */
      for(j = 0; j < bsize; j++)
         widx[j] += vblkpr[j]/x0;
/*
 * and update the pointers to block positions
 */
      widx += bsize;
      xidx += bsize;
      vidx += bsize;
   }
}

void mexFunction(
   int nlhs,       mxArray *plhs[],
   int nrhs, const mxArray *prhs[]
)
{
   double *xpr,*vpr,*wpr,*blk;
   int i,j,n,nblk,sumblk,maxbsize,eltsize;

/* Check for proper number of arguments */
   if (nrhs != 3) {
      mexErrMsgTxt("arwmul requires three input arguments.");
   } else if (nlhs != 1) {
      mexErrMsgTxt("arwmul requires one output argument.");
   }

/* consistency check */
   n = mxGetM(x_IN);
   i = mxGetN(x_IN);
   if(i != 1)
      mexErrMsgTxt("arwimul: x must be a column vector.");
   if(n <= 1)
      mexErrMsgTxt("arwimul: x must have length at least two.");
   if(mxIsSparse(x_IN))
      mexErrMsgTxt("arwimul: x must be full.");

   i = mxGetM(v_IN);
   j = mxGetN(v_IN);
   if(j != 1)
      mexErrMsgTxt("arwimul: v must be a column vector.");
   if(n != i)
      mexErrMsgTxt("arwimul: x and v have incompatible lengths.");
   if(mxIsSparse(v_IN))
      mexErrMsgTxt("arwimul: v must be full.");

   i = mxGetM(blk_IN);
   nblk = mxGetN(blk_IN);
   nblk = max(i,nblk);
   if(nblk < 1)
      mexErrMsgTxt("arwmul: block structure vector is empty.");

   eltsize = mxGetElementSize(x_IN);

/* Assign pointers to the various input parameters */
   xpr = mxGetPr(x_IN);
   vpr = mxGetPr(v_IN);
   blk = mxGetPr(blk_IN);
   sumblk = 0;
   maxbsize = 0;
   for(i = 0; i < nblk; i++) {
      j = blk[i];
      if(j > maxbsize)
         maxbsize = j;
      sumblk += j;
   }
   if (n != sumblk)
      mexErrMsgTxt("arwmul: block structure is incompatible with length of x.");

/* Create a matrix for the return argument */
   w_OUT = mxCreateDoubleMatrix(n,1,mxREAL);

/* Assign pointers to the output parameter */
   wpr = mxGetPr(w_OUT);

/* Do the actual computations in a subroutine */
   arwimul(wpr,xpr,vpr,n,blk,nblk,maxbsize,eltsize);
   return;
}
