function [cndprimal,D] = primalcond(A,blk,X,eigtol)
% PRIMALCOND   estimates whether an SDP is primal degenerate (with respect to
%              its block structure) at its computed solution.
%              It constructs the matrix D whose rows are vector
%              representations of the matrices
%                    [ Q1^T Ak Q1  Q1^T Ak Q2]    k=1,...,m,
%              where the columns of Q1 form an orthonormal basis for the
%              range space of X and Q2 for the null space of X
%              (corresponding to eigenvalues smaller than eigtol).
%              So D has m rows and qq columns, where qq depends on the block
%              structure and the ranks of each block.  The SDP is said to be
%              primal nondegenerate if its rows are linearly independent.
%              If m > qq, the SDP is primal degenerate by definition, and
%              cndprimal is set to infinity.  If m <= qq, cndprimal is set to
%              the 2-norm condition number of D.  A very large condition number
%              is a strong indication that the SDP is primal degenerate.
%
% [cndprimal,D] = primalcond(A,blk,X,eigtol)
%
% input variables:
%     - A         matrix of primal constraints
%     - blk       block structure vector
%     - X         the primal variable
%     - eigtol    a tolerance to determine rank of X
%
% output variables:
%     - cndprimal the condition number of the primal problem
%     - D         matrix whose condition number is cndprimal (see AHO)

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 nblk = length(blk);
 [m,n2] = size(A);
%
 [lam,Q] = blkeig(X,blk);
 I = lam >= eigtol;                    % for Q1
 if sum(I) == 0,                       % primal nondegeneracy not defined
    fprintf('primalcond: X has no eigenvalues greater than tolerance: D is empty\n');
    cndprimal = Inf;
    fprintf('primalcond = %11.3e\n',cndprimal);
    return;
 end;
 J = ~I;                                % for Q2
 start = 1;                             % find block structure of Q1 and Q2,
 fin = 0;
 nextQ1 = 0;
 for i = 1:nblk
    bsize = blk(i);
    fin = fin + bsize;
    tmp = sum(I(start:fin));
    Q1blk(i) = tmp;
    if tmp > 0,
       nextQ1 = nextQ1 + 1;
       Q11blk(nextQ1) = tmp;
    end;
    tmp = sum(J(start:fin));
    Q2blk(i) = tmp;
    start = start + bsize;
 end;
 Q1 = Q(:,find(I));                    % e-vectors for nonzero eigenvalues of X
 Q2 = Q(:,find(J));                    % e-vectors for zero eigenvalues of X
 base = sum(Q1blk .* (Q1blk+1))/2;     % # cols in D due to the Q1^T Ak Q1 blocks
 ncols = base;
 for i = 1:nblk,
    Q1bsize = Q1blk(i);
    Q2bsize = Q2blk(i);
    ncols = ncols + Q1bsize*Q2bsize;
 end;
 D = zeros(m,ncols);
 for k=1:m
    Ak = smat(A(k,:),blk);
    AkQ1 = Ak*Q1;
    block11 = AkQ1'*Q1;                % for this part simply take svec with block
    v = svec(block11,Q11blk)';          % structure Q1blk
    block12 = AkQ1'*Q2;                % the nonzero blocks of this part are of size
                                       % Q1blk(j)*Q2blk(j) for j=1,...,min(Q1nblk,Q2nblk)
    rstart = 1;                        % row start index
    rfin = 0;                          % row end index
    cstart = 1;                        % column start index
    cfin = 0;                          % column end index
    colidx = base + 1;
    for i = 1:nblk,
       Q1bsize = Q1blk(i);
       Q2bsize = Q2blk(i);
       rfin = rfin + Q1bsize;
       cfin = cfin + Q2bsize;
       len = Q1bsize*Q2bsize;
       if len > 0,
          v(colidx:colidx+len-1) = block12(rstart:rfin,cstart:cfin);
          colidx = colidx + len;
       end;
       rstart = rstart + Q1bsize;
       cstart = cstart + Q2bsize;
    end;
    D(k,:) = v;
 end;
 qq = size(D,2);                      % number of columns of D
 if m > qq
    cndprimal = Inf;                  % rows cannot be linearly independent
 else
    cndprimal = cond(D);              % condition number of D
 end;
 fprintf('primalcond = %11.3e\n',cndprimal);
%
% END function
