function M = lyapsol(Q,Evsum,R,blk)
% LYAPSOL   Lyapunov equation solver for SDP (XZ+ZX method)
%
% M = lyapsol(Q,Evsum,R,blk)
%
%   Solves the Lyapunov equation
%                    M Z + Z M = R
%   for the symmetric matrix M given the symmetric positive
%   definite matrix Z and the symmetric matrix R, where
%   Z = Q diag(lam) Q', with Q an orthogonal matrix and lam
%   a vector.
%   Thus we solve
%
%        diag(lam) (Q'MQ) + (Q'MQ) diag(lam) = Q'RQ
%
%   to get: (Q'MQ)(i,j) = (Q'RQ)(i,j)/(lam(i)+lam(j)),
%   and then extract M.
%
%   Note: Evsum(i,j) = lam(i) + lam(j) > 0
%
% input variables:
%     - Q         an orthogonal nxn matrix, the eigenvectors of the dual slack Z
%     - R         an nxn block diagonal matrix, the right-hand-side of Lyapunov system
%     - Evsum     an nxn block diagonal matrix, the sums of the eigenvalues of Z
%     - blk       block structure vector
%
% output variables:
%     - M         an nxn block diagonal matrix, the solution to Lyapunov system

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 nblk = length(blk);
 n = sum(blk);
 if nblk > 1,
    M = sparse(n,n);
 else,
    M = zeros(n,n);
 end;
%
 if (size(Q) ~= [n n]) | (size(R) ~= [n n]),
    error('lyapsol: input improperly dimensioned!');
 end;
%
 R = Q'*R*Q;
%
 start = 1;  fin = 0;
 for i=1:nblk,
    bsize = blk(i);
    fin = fin + bsize;
    M(start:fin,start:fin) = R(start:fin,start:fin) ./ Evsum(start:fin,start:fin);
    start = start + bsize;
 end;
%
 M = Q*M*Q';
%
% END function
