% LSDP   driver for the function flsdp
%
%   Semi Definite Programming Solver using either the XZ or the
%   XZ + ZX method for SDP problems that compute the Lovasz theta
%   function of a graph with n vertices.
%   It is assumed that the matrix G holds a list of the (m-1)
%   edges of the graph stored as follows:
%                 G(k,1) = p, G(k,2) = q,
%   if the k-th edge of the graph joins vertices p and q.
%   There is one constraint matrix for each edge which is of the
%   form
%         A_k = e_p e_q^T + e_q e_p^T,   1 <= k <= m-1
%   and the last constraint matrix is: A_m = eye(n)
%   The matrix C is defined by C(i,j) = - sqrt(w(i)w(j)) where w
%   is a vector of weights, and the vector b, of length
%   m, is [0 ... 0 1].
%
%   Note: lsdp can call either flsdp.m or fsdp.m depending upon
%   the value of the flag useXZ; when calling fsdp.m, lsdp must
%   build the matrices A and C, and the vector b.
%
%   The following variables must be available in the Matlab workspace:
%      - G            list of edges in the graph
%      - w            positive vector of weights
%      - X            initial guess for primal variable
%      - y            initial guess for dual variable
%      - Z            initial guess for dual slack variable
%      - maxit        maximum iterations allowed
%      - tau          fraction of step to boundary of psd cone
%      - steptol      tolerance for short steps
%      - abstol       absolute tolerance on gap
%      - reltol       relative tolerance on gap
%      - gapprogtol   tolerance on sufficient progress
%      - feasprogtol  tolerance on sufficient progress
%      - bndtol       upper bound on size of solution
%      - prtlevel     verbosity level (accessed only by flsdp.m or fsdp.m)

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97

% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> new point rejected since X indefinite: chol(X) failed
%    termflag =  2  ==> new point rejected since Z indefinite:
%                       XZ method: chol(Z) failed or Z is singular
%                       XZ+ZX method: chol(Z) failed or Z has eigenvalues <= 0
%    termflag =  3  ==> XZ method: SchurComp is numerically indefinite or singular
%                       XZ+ZX method: SchurComp is numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X is not positive definite
%    termflag =  2 and iter = 0 ==> initial Z is not positive definite
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = tau;
 scalefac_orig = scalefac;
 if useXZ,
    if tau > 0.99
       fprintf('\nlsdp: tau is set to %7.5f\n',tau);
       fprintf('   the XZ method does not perform well with such a high ');
       fprintf('value of tau!\n');
       choice = input('   Do you wish to proceed? <y/n> ','s');
       if choice ~= 'y'
          break;
       end;
    end;
    fprintf('\nlsdp: using XZ method...\n\n');

    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',tau,scalefac);
       [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
           flsdp(G,w,X,y,Z,maxit,tau,steptol,abstol,reltol,...
                gapprogtol,feasprogtol,bndtol,prtlevel,validate);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\nlsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & maxit == 0,
             fprintf('\nlsdp: Data passed validation test!\n');
          elseif termflag == 7 & prtlevel == 0,
             fprintf('\nlsdp: Data failed validation tests\n');
             fprintf('For more information, rerun lsdp with prtlevel = 1\n');
          end;
          break;
       end;

       if autorestart & (termflag == 4 | termflag == 5)...
         & (gapval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          tau = min(tau,.9);
          scalefac = 100*scalefac;
          linitvars;
       else,
          break;
       end;
    end;  % runs
 else,                % use XZ+ZX method: must construct the matrices
    n2 = n*(n+1)/2;   % A and C, and the vector b
    m = length(y);
    A = sparse(m,n2);
    for i = 1:(m-1),
       Ai = sparse(n,n);
       if G(i,1) < G(i,2),       % this should be true; it is when G
          Ai(G(i,2),G(i,1)) = 1; % is constructed by thetarnd.m
       else
          Ai(G(i,1),G(i,2)) = 1;
       end;
       A(i,:) = svec(Ai,n,1)';  % svec only uses lower triangle
    end;
    Ai = speye(n);
    A(m,:) = svec(Ai,n,1)';
    C = -sqrt(w*w');
    b = zeros(m,1);
    b(m) = 1;
    fprintf('\nlsdp: using XZ+ZX method...\n\n');

    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',tau,scalefac);
       [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
           fsdp(A,b,C,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
                gapprogtol,feasprogtol,bndtol,prtlevel,validate);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\nlsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & maxit == 0,
             fprintf('\nlsdp: Data passed validation test!\n');
          elseif termflag == 7 & prtlevel == 0,
             fprintf('\nlsdp: Data failed validation tests\n');
             fprintf('For more information, rerun lsdp with prtlevel = 1\n');
          end;
          break;
       end;

       if autorestart & (termflag == 4 | termflag == 5)...
         & (gapval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          if runs == 1,
             tau = min(tau,.99);
          elseif runs == 2,
             tau = min(tau,.9);
          end;
          scalefac = 100*scalefac;
          linitvars;
       else,
          break;
       end;
    end;  % runs
 end;
 tau = tau_orig;
 scalefac = scalefac_orig;
%
%
 if iter > 0,
    if termflag == 6,
       fprintf('\nlsdp: reached the maximum number of iteration\n');
    end;
    gapval = gapval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    elapsedtime = etime(clock,tstart);
    cputend = cputime - cputstart;
    if runs > 1,
       fprintf('\nlsdp: number of runs = %3.0f\n',min(runs,3));
    end;

    fprintf('\nlsdp: elapsed time                = %- 9.3f seconds\n',elapsedtime);
    fprintf('lsdp: elapsed cpu time            = %- 9.3f seconds\n',cputend);
    fprintf('lsdp: flops                       = %- 13.0f\n',flops);
    fprintf('lsdp: number of iterations        = %- 4.0f\n',iter);
    fprintf('lsdp: final value of X.Z          = %- 11.3e\n',gapval(iter+1));
    fprintf('lsdp: final primal infeasibility  = %- 11.3e\n',feasval(iter+1,1));
    fprintf('lsdp: final dual infeasibility    = %- 11.3e\n',feasval(iter+1,2));
    fprintf('lsdp: primal objective value      = %- 24.16e\n',objval(iter+1,1));
    fprintf('lsdp: dual objective value        = %- 24.16e\n',objval(iter+1,2));

% Lovasz theta function is the negative of what the code computes due
% to a max-min conflict in the formulation
    fprintf('lsdp: Lovasz theta function value = %- 24.16e\n',...
           -0.5*(objval(iter+1,1)+objval(iter+1,2)));
 end;
