//Title:        JET
//Version:      1.10
//Copyright:    Copyright (c) 2003
//Author:       Ralph Grishman
//Description:  A Java-based Information Extraction Tool

package Jet;
import java.util.*;
import java.io.*;
import Jet.Tipster.*;
import Jet.Lisp.*;
import Jet.Concepts.*;
import Jet.Lex.*;
import Jet.Pat.*;
import Jet.Refres.*;
import Jet.Parser.*;
import Jet.Zoner.*;
import Jet.HMM.*;
import Jet.Chunk.Chunker;

/**
 *  the Control class provides the methods for interpreting Jet scripts.
 */

public class Control {
	
	/**
	 *  apply the <B>processDocument</B> script to (all of) document 
	 *  <CODE>doc</CODE>.  In addition, if Jet parameter WriteSGML.type
	 *  is set, write the document in SGML format to file response-
	 *  <CODE>inputFileName</CODE>.  If <CODE>viewable</CODE> is true,
	 *  open a window displaying the document, and label the window
	 *  "Jet Document <CODE>docNo</CODE>".
	 */
	 
	public static void processDocument (Document doc, OutputStreamWriter writer, 
	                                    boolean viewable, int docNo) throws IOException {

	    String script = JetTest.config.getProperty("processDocument");
	    if (script == null || script.length() == 0) {
			Console.println ("*** System error: no processDocument script.");
			return;
		}
		// if there is a name tagger, clear its cache
		if (JetTest.nameTagger != null) JetTest.nameTagger.nameHMM.newDocument();
	    applyScript (doc, new Span(0, doc.length()), script);
	    String type = JetTest.config.getProperty("WriteSGML.type");
	    if (type != null) {
	    	// String outputFile = JetTest.dataPath + File.separatorChar + "response-" + inputFileName;
	        // PrintWriter writer = new PrintWriter (new FileWriter (outputFile));
	        // writer.println (doc.writeSGML (type));
	        System.out.println ("Writing document " + docNo);
			writer.write (doc.writeSGML (type).toString());
			writer.write (System.getProperty("line.separator"));
	        // writer.close ();
	    }
	    if (viewable) {
	        View view = new View (doc, docNo);
	        JetTest.views.add(view);
	        Vector v = doc.annotationsOfType("entity");
		    if (v != null && v.size() > 0) {
		    	EntityView eview = new EntityView (doc, docNo);
		    	JetTest.views.add(eview);
		    }
	    }
	}
	
	/**
	 *  apply the <B>processSentence</B> script to span <CODE>sentenceSpan</CODE>
	 *  of document <CODE>doc</CODE>.
	 */
	 
	public static void processSentence (Document doc, Span sentenceSpan) {
  		String script = JetTest.config.getProperty("processSentence");
  		if (script == null || script.length() == 0)
  			Console.println ("*** No processSentence script.");
		else
	    	Control.applyScript(doc, sentenceSpan, script);
	}
	
	/**
	 *  apply script <CODE>script</CODE> to span <CODE>span</CODE> of document
	 *  <CODE>doc</CODE>.
	 */
	
	public static void applyScript (Document doc, Span span, String script) {
		String[] actions = splitAtComma(script);
		
		for (int j=0; j<actions.length; j++) {
			String action = actions[j].intern();
			
			int colon = action.indexOf(':');
			if (colon > 0 && colon < action.length()-1) {
				String zoneType = action.substring(0,colon).trim();
				String scriptName = action.substring(colon+1).trim();
				String zoneScript = JetTest.config.getProperty(scriptName);
				if (zoneScript == null || zoneScript.length() == 0) {
					Console.println ("*** No script for " + scriptName);
					continue;
				}
				Vector zones = doc.annotationsOfType (zoneType, span);
				if (zones == null) {
					Console.println ("*** No annotations of type " + zoneType);
					continue;
				}
				for (int i=0; i<zones.size(); i++) {
					Annotation zone = (Annotation) zones.get(i);
					applyScript (doc, zone.span(), zoneScript);
				}
			} else if (action.startsWith("tag(") && action.endsWith(")")) {
				String tagName = action.substring(4,action.length()-1).trim();
				doc.annotateWithTag (tagName);
			} else if (action == "tokenize") {
				Tokenizer.tokenize (doc, span);
			} else if (action == "sentenceSplit") {
			 	SentenceSplitter.split (doc, span);
			} else if (action == "speechSplit") {
				SpeechSplitter.split (doc, span);
			} else if (action == "lexLookup") {
				Lexicon.annotateWithDefinitions(doc, span.start(), span.end());
			} else if (action == "tagPOS") {
	        	if (JetTest.tagger == null)
	        		Console.println ("Error:  no POS model loaded");
	        	else
	        		JetTest.tagger.tagPenn (doc, span); 
	        } else if (action == "tagJet") {
	        	if (JetTest.tagger == null)
	        		Console.println ("Error:  no POS model loaded");
	        	else
	        		JetTest.tagger.tagJet (doc, span); 
	        } else if (action == "pruneTags") {
	        	if (JetTest.tagger == null)
	        		Console.println ("Error:  no POS model loaded");
	        	else
	        		JetTest.tagger.prune (doc, span);
	        } else if (action == "tagNames") {
	        	if (JetTest.nameTagger == null)
	        		Console.println ("Error:  no name model loaded");
	        	else
	        		JetTest.nameTagger.tag (doc, span);
	        // added June 23rd, 2003 ***
	        } else if (action == "chunk") {
	        	if (Chunker.model == null)
	        		Console.println ("Error:  no chunker model loaded");
	        	else
	        		Chunker.chunk (doc, span);
	        } else if (action == "parse") {
	        	Vector parses = Parsers.parse (doc,0,doc.length(),JetTest.gram);
	        	for (int i=0; i<parses.size(); i++) {
	            	ParseTreeNode parse = (ParseTreeNode) parses.elementAt(i);
	            	if (parse != null) {
	            		ParseTreeNode.makeParseAnnotations (doc, parse);
	              		if (parses.size() == 1) {
	                		Console.println ("Parse:");
	              		} else {
	                		Console.println ("Parse " + (i+1) + ":");
	              		}
	            	parse.printTree();
	          		}
	        	}
	        // added Dec. 11th, 2003 ***
	        } else if (action == "statParse") {
	        	if (!StatParser.isInitialized())
	        		Console.println ("Error:  no grammar for parser");
	        	else
	        		StatParser.parse (doc, span);
	        // added Jan. 7th, 2004 ***
	       	} else if (action == "syntacticRelations") {
	       			AddSyntacticRelations.annotate (doc, span);
	        } else if (action.startsWith("pat(") && action.endsWith(")")) {
	        	String patternSetName = action.substring(4,action.length()-1).trim();
	        	JetTest.pc.apply (patternSetName, doc, span);
	        } else if (action == "resolve") {
	        	Resolve.references (doc, span);
	        } else {
	            System.out.println ("Unknown Jet.processSentence action: " + action);
	        }
			
		}
	}
	
	private static String[] splitAtComma (String str) {
		StringTokenizer tok = new StringTokenizer(str, ",");
		int tokenCount = tok.countTokens();
		String[] result = new String[tokenCount];
		for (int i=0; i<tokenCount; i++)
			result[i] = tok.nextToken().trim();
		return result;
	}
}