function [cndp,Dsize,D] = pcond(A,blk,X,tol)
% PCOND  estimates whether an SQLP is primal degenerate at its computed
%        solution.
%        It constructs a certain matrix D with m rows that has the
%        property that the primal solution X to the SQLP is primal
%        nondegenerate if and only if D has maximal rank m.
%        The columns of D are partitioned into blocks corresponding
%        to some of the blocks of the SQLP. The SQLP is said to be
%        primal nondegenerate if the rows of D are linearly independent.
%        Let ncols denote the number of columns of D.
%        If m > ncols, the SQLP is primal degenerate by definition,
%        and cndp is set to infinity.
%        If ncols = 0, i.e. then the matrix D is empty, then the SQLP
%        is primal degenerate and we set cndp to infinity unless m = 0
%        in which case it is nondegenerate and we set cndp = 0.
%        If ncols > 0 and m <= ncols, cndp is set to the 2-norm
%        condition number of D.  A very large condition number is a
%        strong indication that the SQLP is primal degenerate.
%        The tolerance tol is used to decide which columns to include
%        in D.  If strict complementarity does not hold, it is advisable
%        not to make tol too small, e.g. sqrt(abstol).
%
% [cndp,Dsize,D] = pcond(A,blk,X,tol)
%
% input variables:
%     - A            structure of constraints data
%     -     A.s      constraint matrix for SD
%     -     A.q      constraint matrix for QC
%     -     A.l      constraint matrix for LP
%     - blk          block info structure
%     -     blk.s    block info for SD
%     -     blk.q    block info for QC
%     -     blk.l    block info for LP
%     - X            initial guess for primal structure
%     -     X.s      SD primal variable
%     -     X.q      QC primal variable
%     -     X.l      LP primal variable
%     - tol          tolerance
%
% output variables:
%     - cndp         the condition number of the primal problem
%     - Dsize        partition info for columns of D
%     -     Dsize.s  # columns of D coming from SD component
%     -     Dsize.q  # columns of D coming from QC component
%     -     Dsize.l  # columns of D coming from LP component
%     - D            matrix whose condition number is cndp

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/17/97
%
 if isfield(blk,'s')
    n.sm = sum(blk.s);
    n.sv = sum(blk.s .* (1+blk.s))/2;
    nblk_s = length(blk.s);
 else
    n.sm = 0;
    n.sv = 0;
    nblk_s = 0;
 end
 if isfield(blk,'q')
    n.q = sum(blk.q);
    nblk_q = length(blk.q);
 else
    n.q = 0;
    nblk_q = 0;
 end
 if isfield(blk,'l')
    n.l = sum(blk.l);
 else
    n.l = 0;
 end
 if n.sm + n.q + n.l <= 0
    fprintf('pcond: the block structure is empty.  Aborting...\n');
    return
 end
%
% Note:  n.sm = 0 means that the mixed problem does not contain any SD part
%        n.q  = 0 means that the mixed problem does not contain any QC part
%        n.l  = 0 means that the mixed problem does not contain any LP part
%
 m = 0;
 if n.sm > 0
    m = size(A.s,1);
 end
 if n.q > 0
    m = max(m,size(A.q,1));
 end
 if n.l > 0
    m = max(m,size(A.l,1));
 end
 ncols = 0;       % the number of columns of D
 D = zeros(m,0);  % empty matrix with m rows
%
% First the semidefinite part
% The part of D coming from the SD component of the SQLP consists
% of representations of the matrices
%       [ Q1^T Ak Q1  Q1^T Ak Q2]    k=1,...,m,
% where the columns of Q1 form an orthonormal basis for the
% range space of X and Q2 for the null space of X (corresponding
% to eigenvalues smaller than tol), and Ak are the matrices corresponding
% to the rows of A.s.
%
 if n.sm > 0   % SQL program has an SD component
    [lam,Q] = blkeig(X.s,blk.s); % eigenvalues and eigenvectors of SD component of solution
    I = lam >= tol;         % to determine Q1, the eigenvectors correspondng to
                            % nonzero (within tolerance tol) eigenvalues of X.s
    if sum(I) > 0           % otherwise skip the SD component
       J = ~I;              % for Q2
       fin = 0;             % find block structure of Q1 and Q2,
       nextQ1 = 0;
       for k = 1:nblk_s
          bsize = blk.s(k);
          start = fin + 1;
          fin = fin + bsize;
          tmp = sum(I(start:fin));
          Q1blk(k) = tmp;           % number of columns in kth block of Q1
          if tmp > 0                % if Q1blk(k) = 0, this block does not contribute to D
             nextQ1 = nextQ1 + 1;
             Q11blk(nextQ1) = tmp;  % block structure for the matrices Q1'*Ak*Q1
          end
          tmp = sum(J(start:fin));
          Q2blk(k) = tmp;           % number of columns in ith block of Q2
       end
       Q1 = Q(:,find(I));                 % e-vectors for nonzero eigenvalues of X
       Q2 = Q(:,find(J));                 % e-vectors for zero eigenvalues of X
       base = sum(Q1blk .* (Q1blk+1))/2;  % # cols in D due to the Q1^T Ak Q1 blocks
       ncols = base;
       for k = 1:nblk_s
          Q1bsize = Q1blk(k);
          Q2bsize = Q2blk(k);
          ncols = ncols + Q1bsize*Q2bsize; % if ncols = 0, X.s = 0 and the SD component
       end                                 % does not contribute to D
       if ncols > 0
          D = zeros(m,ncols);
          for k = 1:m   % loop over the rows of A.s
             Ak = smat(A.s(k,:),blk.s);
             AkQ1 = Ak*Q1;
             block11 = AkQ1'*Q1;        % for this part simply take svec with block
             v = svec(block11,Q11blk)'; % structure Q1blk
             block12 = AkQ1'*Q2;        % the nonzero blocks of this part are of size
                                        % Q1blk(j)*Q2blk(j) for j=1,...,min(Q1nblk,Q2nblk)
             rfin = 0;                  % row end index
             cfin = 0;                  % column end index
             colidx = base + 1;
             for i = 1:nblk_s
                Q1bsize = Q1blk(i);
                Q2bsize = Q2blk(i);
                rstart = rfin + 1;         % row start index
                rfin = rfin + Q1bsize;
                cstart = cfin + 1;         % column start index
                cfin = cfin + Q2bsize;
                len = Q1bsize*Q2bsize;
                if len > 0
                   v(colidx:colidx+len-1) = block12(rstart:rfin,cstart:cfin);
                   colidx = colidx + len;
                end
             end  % loop over SD blocks
             D(k,:) = v;
          end  % loop over the rows of A.s
      end   % if ncols > 0
    end  % if sum(I) > 0
 end  % if n.sm > 0
 Dsize.s = ncols;
%
% Next the quadratic part.
% We loop over the blocks; for each block k there are
% three possibilities to consider:
% 1. the block variable X.q(k) is zero we simply move on to the next block
% 2. the block variable X.q(k) lies in the interior of the Lorentz cone;
%    then we append the corresponding block of A.q to D
% 3. the block variable X.q(k) lies on the boundary of the Lorentz cone;
%    let x = X.q(k), a vector of length nk+1 = blk.q(k), say x = (x0,x1,...,xnk),
%    and let Ak denote the kth block of A.q, an m by (nk+1) matrix whose rows
%    are a1,...,am.
%    Now we construct the matrix
%           [ x1/x0  x2/x0 ... xnk/x0 ]
%           [   1      0   ...    0   ]
%       T = [   0      1          0   ]   of size (nk+1) by nk,
%           [         ...             ]
%           [   0      0          1   ]
%    whose columns span the tangent space T_x to the boundary of the cone at x.
%    Then we let Q = orth(T), an (nk+1) by nk matrix with orthonormal columns
%    that also span the space T_x. We then append to D the m by nk matrix whose
%    rows consist of the vectors Q^T times the vectors aj, j = 1,...,m, namely
%    the matrix
%        Ak * Q
%    Alternatively, we may take Q = null(g') where g is the vector
%        g = [ x0 -x1 ... -xnk ]^T
%    namely he gradient of the function f(x) = x0^2 - sum_1^nk xi^2; clearly
%    the vector g spans the normal space to the Lorentz cone at x.
%
 if n.q > 0   % SQL program has an QC component
    fin = 0;
    for k = 1:nblk_q
       bsize = blk.q(k);       % what we called nk+1 in comments above
       start = fin + 1;
       fin = fin + bsize;
       if X.q(start) > tol     % otherwise x = 0, so move on to the next block
          x = X.q(start:fin);
          xbar = x(2:bsize);
          Ak = A.q(:,start:fin);
          if x(1)*x(1) - xbar'*xbar > tol    % x in the interior of the cone
             if ncols > 0
                D = cat(2,D,full(Ak));
                ncols = ncols + bsize;
             else
                D = full(Ak);
                ncols = bsize;
             end
          else                            % x on the boundary of the cone
             coldim = length(xbar);       % what we called nk in comments above
             g = -x;
             g(1) = x(1);
             Q = null(g');
             if size(Q,2) ~= coldim
                error('pcond: null computed a matrix with the wrong number of columns');
             end
             if ncols > 0
                D = cat(2,D,full(Ak*Q));
                ncols = ncols + coldim;
             else
                D = full(Ak*Q);
                ncols = coldim;
             end
          end
       end  % if vector block is nonzero
    end  % loop over the blocks
 end  % if n.q > 0
 Dsize.q = ncols - Dsize.s;
%
% Finally the linear part
% This is very simple: we simply append to D the columns of the constraint
% matrix A.l corresponding to the nonzero entries of the vector X.l
%
 if n.l > 0   % SQL program has an LP component
    I = X.l >= tol;     % indices of nonzero entries of X.l
    if ncols > 0
       D = cat(2,D,full(A.l(:,find(I))));
       ncols = ncols + sum(I);
    else
       D = full(A.l(:,find(I)));
       ncols = sum(I);
    end
 end
 Dsize.l = ncols - Dsize.s - Dsize.q;
%
 if m > ncols        % too many rows: degenerate
    cndp = Inf;
 elseif ncols == 0   % ncols = 0 and m = 0: nondegenerate
    cndp = 0;
 else
    cndp = cond(D);  % condition number of D
 end;
 fprintf('primal condition number = %12.4e\n',cndp);
%
% END function
