/*
 * arwmul.c - mex file: implements the multiplication of a
 *            vector v by the arrow matrix of a vector x; this is
 *            done blockwise where blk is the block structure vector.
 *
 * synopsis:   w = arwmul(x,v,blk)
 *
 * Single block case:
 * if x = (x0,x1,...,xn) then
 *             [ x0 x1 x2 ... xn ]
 *             [ x1 x0  0 ...  0 ]
 *    arw(x) = [ x2  0 x0 ...  0 ]
 *             [      ...        ]
 *             [ xn  0  0 ... x0 ]
 *
 *     = x0*I + e0*xbar^T + xbar*e0^T
 * where
 *     e0   = (1,0,...,0) of length n+1
 *     xbar = (0,x1,...,xn)
 *
 * inputs:
 *     - x         vector of quadratic blocks
 *     - v         vector of length n
 *     - blk       block structure vector
 *
 * output:
 *     - w         vector of length n
 *
 * SDPPACK Version 0.9 BETA
 * Copyright (c) 1997 by
 * F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
 * Last modified : 5/26/97
 */
#include <math.h>
#include "mex.h"

/* Input Arguments */
#define  x_IN     prhs[0]
#define  v_IN     prhs[1]
#define  blk_IN   prhs[2]

/* Output Arguments */
#define  w_OUT    plhs[0]

#if !defined(max)
#define  max(A, B)   ((A) > (B) ? (A) : (B))
#endif

#if !defined(min)
#define  min(A, B)   ((A) < (B) ? (A) : (B))
#endif

static void arwmul(
   double  *wpr,
   double  *xpr,
   double  *vpr,
   int  n,
   double  *blk,
   int  nblk,
   int  maxbsize,
   int eltsize
)
{
   int i,j,bsize;
   double *widx,*xidx,*vidx;     /* will point to the start of the blocks */
   double *xblkpr,*vblkpr,*lhspr;
   mxArray *xblk,*vblk,*plhs[1],*prhs[2];

/* Create temporary matrices for the blocks */
   xblk = mxCreateDoubleMatrix(maxbsize,1,mxREAL);
   vblk = mxCreateDoubleMatrix(maxbsize,1,mxREAL);
   xblkpr = mxGetPr(xblk);
   vblkpr = mxGetPr(vblk);

   widx = wpr;    /* initially widx, xidx, and vidx point the beginning */
   xidx = xpr;    /* of the arrays wpr, xpr, and vpr respectively */
   vidx = vpr;
   for(i = 0; i < nblk; i++) {
      bsize = blk[i];
      mxSetM(vblk,bsize);
      mxSetN(vblk,1);
      memcpy(xblkpr,xidx,bsize*eltsize);   /* copy data to xblkpr */
      memcpy(vblkpr,vidx,bsize*eltsize);   /* and to vblkpr */
/* start of computation */
      prhs[0] = xblk;      /* xblk is first input parameter */
      prhs[1] = vblk;      /* vblk is second input parameter */
/* first compute x(1)*v */
      mxSetM(xblk,1);      /* set the sizes of xblk to (1,1), so it */
      mxSetN(xblk,1);      /* is the first entry of the block of x */
      mexCallMATLAB(1,plhs,2,prhs,"*");   /* compute x(1)*v */
      lhspr = mxGetPr(plhs[0]);
      memcpy(widx,lhspr,bsize*eltsize); /* copy result into wpr */
/* next set x(1) = 0 */
      xblkpr[0] = 0.0;
/* next compute w(1) = w(1) + x'*v */
      mxSetM(xblk,1);         /* set the sizes of xblk to (1,n), so it */
      mxSetN(xblk,bsize);     /* is the transpose of the block of x */
      mexCallMATLAB(1,plhs,2,prhs,"*");   /* compute x'*v */
      lhspr = mxGetPr(plhs[0]);
      widx[0] += lhspr[0];
/* finally compute w = w + v(1)*x */
      mxSetM(xblk,bsize);     /* reset the sizes of xblk to (n,1) */
      mxSetN(xblk,1);
      mxSetM(vblk,1);         /* and those of v to (1,1) */
      mexCallMATLAB(1,plhs,2,prhs,"*");   /* compute v(1)*x */
      lhspr = mxGetPr(plhs[0]);
      for(j = 0; j < bsize; j++)
         widx[j] += lhspr[j];
/* and update the pointers to block positions */
      widx += bsize;
      xidx += bsize;
      vidx += bsize;
   }
}

void mexFunction(
   int nlhs,       mxArray *plhs[],
   int nrhs, const mxArray *prhs[]
)
{
   double *xpr,*vpr,*wpr,*blk;
   int i,j,n,nblk,sumblk,maxbsize,eltsize;

/* Check for proper number of arguments */
   if (nrhs != 3) {
      mexErrMsgTxt("arwmul requires three input arguments.");
   } else if (nlhs != 1) {
      mexErrMsgTxt("arwmul requires one output argument.");
   }

/* consistency check */
   n = mxGetM(x_IN);
   i = mxGetN(x_IN);
   if(i != 1)
      mexErrMsgTxt("arwmul: x must be a column vector.");
   if(n <= 1)
      mexErrMsgTxt("arwmul: x must have length at least two.");
   if(mxIsSparse(x_IN))
      mexErrMsgTxt("arwmul: x must be full.");

   i = mxGetM(v_IN);
   j = mxGetN(v_IN);
   if(j != 1)
      mexErrMsgTxt("arwmul: v must be a column vector.");
   if(n != i)
      mexErrMsgTxt("arwmul: x and v have incompatible lengths.");
   if(mxIsSparse(v_IN))
      mexErrMsgTxt("arwmul: v must be full.");

   i = mxGetM(blk_IN);
   nblk = mxGetN(blk_IN);
   nblk = max(i,nblk);
   if(nblk < 1)
      mexErrMsgTxt("arwmul: block structure vector is empty.");

   eltsize = mxGetElementSize(x_IN);

/* Assign pointers to the various input parameters */
   xpr = mxGetPr(x_IN);
   vpr = mxGetPr(v_IN);
   blk = mxGetPr(blk_IN);
   sumblk = 0;
   maxbsize = 0;
   for(i = 0; i < nblk; i++) {
      j = blk[i];
      if(j > maxbsize)
         maxbsize = j;
      sumblk += j;
   }
   if (n != sumblk)
      mexErrMsgTxt("arwmul: block structure is incompatible with length of x.");

/* Create a matrix for the return argument */
   w_OUT = mxCreateDoubleMatrix(n,1,mxREAL);

/* Assign pointers to the output parameter */
   wpr = mxGetPr(w_OUT);

/* Do the actual computations in a subroutine */
   arwmul(wpr,xpr,vpr,n,blk,nblk,maxbsize,eltsize);
   return;
}
/*
%     x0 = x(bs);
%     v0 = v(bs);
%     xbar = x;
%     xbar(bs) = 0;
%     w = zeros(n);  % n = sum(blk)
%     w(bs) = qb*(xbar .* v);
%     w = w + x0'*qb*v;
%     w = w + v0'*qb*xbar;
 n =length(x);
 nblk = length(blk);
 w = zeros(n,1);
 if nargin == 3 | nblk == 1   % slow: loop over the blocks
    fin = 0;
    for i = 1:nblk
       start = fin + 1;
       fin = fin + blk(i);
       xtmp = x(start:fin);
       vtmp = v(start:fin);
       wtmp = xtmp(1)*vtmp;
       xtmp(1) = 0;
       wtmp(1) = wtmp(1) + xtmp'*vtmp;
       wtmp = wtmp + vtmp(1)*xtmp;
       w(start:fin) = wtmp;
    end
 elseif nargin == 5  % fast: vectorized version
    x0 = x(bs);   % of length nblk
    v0 = v(bs);
    xbar = x;
    xbar(bs) = 0;
    w(bs) = ((xbar .* v)'*qb)';
    w = w + (qb*x0) .* v;
    w = w + (qb*v0) .* xbar;
 else
    error('arwmul: incorrect number of arguments')
 end
%
% END function
*/

